--==========================================================================--
-- Design units : 10/100 Ethernet MAC - RxMAC - RxErrorDetection
--
-- File name    : rxerrordetection.vhd
--
-- Purpose      : The error management of recepted frames is quite simple.
--                One register for each possible error is enough to indicate
--                that this special error occured during frame reception.
--                All registers are cleared on a rising edge of RxFifoDV.
--
-- Note         : None
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-09-04  Initial revision
-- 0.1      MB      2000-09-19  RxLate and RxLateCol added,
--                              FullDuplex control now done by GatedCol
--          MB      2000-11-26  RxValid added as input of RxErrorDetection
--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

entity RxErrorDetection is
  port(
    -- Receive clock and asynchronous reset
    RxReset_N:   in  Std_ULogic;
    RxClk:       in  Std_ULogic;
    -- Receive signals from RxSyncFIFO
    RxFifoDV:    in  Std_ULogic;
    RxFifoEr:    in  Std_ULogic;
    RxFifoCol:   in  Std_ULogic;
    -- Host interface control signals and error indicators
    RxValid:     in  Std_ULogic;
    RxPhyError:  out Std_ULogic;
    RxCollision: out Std_ULogic;
    -- Collision management
    RxLate:      in Std_ULogic;
    RxLateCol:   out Std_ULogic
  );
end RxErrorDetection;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

architecture RTL of RxErrorDetection is

  ----------------------------------------------------------------------------
  -- Internal register set to save errors during frame reception.
  ----------------------------------------------------------------------------
  signal RxPhyErReg:   Std_ULogic;
  signal RxColReg:     Std_ULogic;
  signal RxLateColReg: Std_ULogic;

begin  -- Architecture RTL of RxErrorDetection

  ----------------------------------------------------------------------------
  -- To detect physical errors during reception this process watches the
  -- MII signals RxEr and Col.
  ----------------------------------------------------------------------------
  WatchMII: process (RxReset_N, RxClk)
  begin
    if RxReset_N = '0' then
      RxPhyErReg   <= '0';
      RxColReg     <= '0';
      RxLateColReg <= '0';
    elsif Rising_Edge(RxClk) then
      if RxFifoDV = '1' then
        RxPhyErReg   <= RxPhyErReg or RxFifoEr;
        RxColReg     <= RxColReg or RxFifoCol;
        RxLateColReg <= RxLateColReg or (RxFifoCol and RxLate);
      elsif RxValid = '0' then
        RxPhyErReg   <= '0';
        RxColReg     <= '0';
        RxLateColReg <= '0';
      end if;
    end if;
    
  end process WatchMII;

  ----------------------------------------------------------------------------
  -- Propagate registers to outputs.
  ----------------------------------------------------------------------------
  RxPhyError  <= RxPhyErReg;
  RxCollision <= RxColReg;
  RxLateCol   <= RxLateColReg;

end RTL;  -- End of RxErrorDetection (RTL)
