/*****************************************************************
 *  Stuff to create connections --- OS dependent
 *
 *      EstablishNewConnections, CreateWellKnownSockets, ResetWellKnownSockets,
 *      CloseDownConnection, CheckConnections, AddEnabledDevice,
 *	RemoveEnabledDevice, OnlyListToOneClient,
 *      ListenToAllClients,
 *
 *      (WaitForSomething is in its own file)
 *
 *****************************************************************/

#include <signal.h>
#include <time.h>
#undef NULL
#include "X.h"
#include "Xproto.h"
#include "Xos.h"			/* for strings, file, time */

#include <stdio.h>
#include "opaque.h"
#include "dixstruct.h"
#include "osstruct.h"
#include "osdep.h"
#include "config.h"
#include "starnet.h"
#include "funcs.h"

extern char *display;			/* The display number */

#ifdef X_PRINT_PORT
struct tcpport PrinterPort;
static int PrinterSocket;
static int printfd;
#endif
int MaxClients = NPORTS;
unsigned long myaddr;
u_long CloseTimeOut = 30000;

OsCommPtr MyPrivate[NPORTS];

static int listen_socket(int port, struct tcpport *pport);
static void ErrorConnMax(OsCommPtr oc);

OsCommPtr WellKnownOS, WellKnownO2;
static Bool GrabDone = FALSE;
OsCommPtr oneClient;
int tcpportReg;	    /* port with same byte order as server */

/* These are hacks because we know we only do MIT-MAGIC-COOKIE-1. */
int MitMagicLen;
char * MitMagicData;
char MitMagicName[18] = "MIT-MAGIC-COOKIE-1";

/*****************
 * CreateWellKnownSockets
 *    At initialization, create the sockets to listen on for new clients.
 *****************/

extern struct conf Scon;

void
CreateWellKnownSockets()
{

    myaddr = Scon.myip;

    tcpportReg = atoi(display);
    tcpportReg += X_TCP_PORT;

    WellKnownOS = (OsCommPtr)Xalloc(sizeof (*WellKnownOS));
    if (!WellKnownOS)
	FatalError("No memory for socket\n");
    WellKnownOS->flags = 0;
    WellKnownOS->fd = listen_socket(tcpportReg, &WellKnownOS->tcpport);

    WellKnownO2 = (OsCommPtr)Xalloc(sizeof (*WellKnownO2));
    if (!WellKnownO2)
	FatalError("No memory for socket\n");
    WellKnownO2->flags = 0;
    WellKnownO2->fd = listen_socket(tcpportReg, &WellKnownO2->tcpport);

#ifdef X_PRINT_PORT
    PrinterSocket = listen_socket(X_PRINT_PORT, &PrinterPort);
#endif
#ifdef notdef
    signal (SIGHUP, AutoResetServer);
#endif
    signal (SIGINT, GiveUp);
    signal (SIGTERM, GiveUp);
    ResetHosts(display);
#ifdef XDMCP
    XdmcpInit ();
#endif
#ifdef notdef
    EnableLocalHost();
#endif
}

void
ResetWellKnownSockets ()
{
    ResetHosts(display);
#ifdef XDMCP
    XdmcpReset ();
#endif
    reset_socket();
#ifdef notdef
    EnableLocalHost();
#endif
}

/* If an alloc failed, WellKnownOS or WellKnownOS2 will be NULL.  This
 * function is called after some memory is freed.
 */
void
reset_socket()
{

    if (WellKnownOS == 0 &&
	((WellKnownOS = (OsCommPtr)Xalloc(sizeof (*WellKnownOS))) != 0))
	    WellKnownOS->fd = listen_socket(tcpportReg, &WellKnownOS->tcpport);
    if (WellKnownO2 == 0 &&
	((WellKnownO2 = (OsCommPtr)Xalloc(sizeof (*WellKnownO2))) != 0))
	    WellKnownO2->fd = listen_socket(tcpportReg, &WellKnownO2->tcpport);
}

/*****************************************************************
 * ClientAuthorized
 *
 *     	It is hoped that eventually one protocol will be agreed upon.  In the
 *        mean time, a server that implements a different protocol than the
 *        client expects, or a server that only implements the host-based
 *        mechanism, will simply ignore this information.
 *
 *****************************************************************/

char * 
ClientAuthorized(client, proto_n, auth_proto, string_n, auth_string)
    ClientPtr client;
    char *auth_proto, *auth_string;
    unsigned short proto_n, string_n;
{
    OsCommPtr priv;
    Bool authorized = FALSE;

    if (MitMagicLen && proto_n == sizeof MitMagicName &&
	memcmp(auth_proto, MitMagicName, sizeof MitMagicName) == 0 &&
	string_n == MitMagicLen &&
	memcmp(auth_string, MitMagicData, MitMagicLen) == 0)
	    authorized = TRUE;

    priv = (OsCommPtr)client->osPrivate;

    if (!authorized &&
	InvalidHost((pointer)&priv->tcpport.outpkt.i.ipdest,
			sizeof (priv->tcpport.outpkt.i.ipdest))) {
	    return "Client is not authorized to connect to Server";
    }

    priv->conn_time = 0;

    /* At this point, if the client is authorized to change the access control
     * list, we should getpeername() information, and add the client to
     * the selfhosts list.  It's not really the host machine, but the
     * true purpose of the selfhosts list is to see who may change the
     * access control list.
     */
    return((char *)0);
}    

/*****************
 * EstablishNewConnections
 *    If anyone is waiting on listened sockets, accept them.
 *****************/

void
EstablishNewConnections(poc)
OsCommPtr *poc;		/* which WellKnown socket */
{
    int newconn;                  /* fd of new client */
    ClientPtr client;
    u_long connect_time;
    OsCommPtr	port, oc = *poc;
    int i;

    newconn = oc->fd;

    connect_time = GetTimeInMillis();

    /* clean up */
    for (i = 0; i < NPORTS; i++)
	if ((port = MyPrivate[i]) != 0) {
	    if (port->flags & OS_CLOSING && port->conn_time != 0 &&
		connect_time - port->conn_time >= CloseTimeOut) {
		    SoFree(i);
		    MyPrivate[i] = 0;
		    xfree(port);
	    }
	}

    /* kill off stragglers */
    for (i=1; i<currentMaxClients; i++)
    {
	if ((client = clients[i]) != 0)
	{
	    port = (OsCommPtr)client->osPrivate;
	    if (port && (port->conn_time != 0) &&
		(connect_time - port->conn_time) >= TimeOutValue)
		CloseDownClient(client);     
	}
    }

    /* If we can't allocate any more space for the listening
     * connection.  Let this client have it anyway and just shut
     * this port down until we get more memory. */
    port = (OsCommPtr)xalloc(sizeof (OsCommRec));
    if (port)
	port->fd = listen_socket(tcpportReg, &port->tcpport);
    *poc = port;

    oc->conn_time = connect_time;
    oc->input = 0;
    oc->output = 0;
    oc->flags = 0;
    MyPrivate[newconn] = oc;

    if ((client = NextAvailableClient((pointer)oc)) != 0)
    {
	oc->client = client->index;
    }
    else
    {
	ErrorConnMax(oc);
	CloseDownFileDescriptor(oc);	/* this may free some memory */
	if (!WellKnownOS || !WellKnownO2)
	    reset_socket();
	return;
    }
/*
printf("new connection %d\n", client->index);
*/
#ifdef XDMCP
    /* indicate to Xdmcp protocol that we've opened new client */
    XdmcpOpenDisplay(newconn, oc->tcpport.outpkt.i.ipdest);
#endif /* XDMCP */
}

static char noroom[] = "Maximum number of clients reached";

/************
 *   ErrorConnMax
 *     Fail a connection due to lack of client or file descriptor space
 ************/

static void
ErrorConnMax(oc)
OsCommPtr	oc;
{
    xConnSetupPrefix csp;
    char byteOrder = 0;
    unsigned long timeout, now;
    int fd = oc->fd;

    /* Let's wait a half second for the client to send us at least its
     * byte order. */
    timeout = n_clicks();
    now = timeout;
    while (oc->tcpport.inbase == oc->tcpport.infree && now - timeout < 6L) {
	now = n_clicks();
	netsleep();
    }
    (void)SoRead(fd, (u_char *)&byteOrder, 1);
    if (byteOrder != 'l' && byteOrder != 'B')
	return;
    csp.success = xFalse;
    csp.lengthReason = sizeof(noroom) - 1;
    csp.length = (sizeof(noroom) + 2) >> 2;
    csp.majorVersion = X_PROTOCOL;
    csp.minorVersion = X_PROTOCOL_REVISION;
    if (byteOrder == 'B') {
	csp.majorVersion = lswaps(csp.majorVersion);
	csp.minorVersion = lswaps(csp.minorVersion);
	csp.length = lswaps(csp.length);
    }
    (void)SoWrite(fd, (u_char *)&csp, sz_xConnSetupPrefix);
    (void)SoWrite(fd, (u_char *)noroom, csp.lengthReason + (-csp.lengthReason & 3));
}

/*****************
 * CloseDownConnection
 *    Delete client from AllClients and free resources 
 *****************/

Bool
CloseDownConnection(client)
    ClientPtr client;
{
    OsCommPtr oc = (OsCommPtr)client->osPrivate;

    if (oc == NULL) {
	printf("HELP: osPrivate is NULL!\n");
	return FALSE;
    }
#ifdef XDMCP
    XdmcpCloseDisplay(oc->fd);
#endif
    oc->client = -1;
    client->osPrivate = 0;
    FlushClient(oc, (char *)0, 0);
    CloseDownFileDescriptor(oc);
    return TRUE;
}

/*****************
 * OnlyListenToOneClient:
 *    Only accept requests from  one client.  Continue to handle new
 *    connections, but don't take any protocol requests from the new
 *    ones.
 *    Note also that there is no timeout for this in the protocol.
 *    This routine is "undone" by ListenToAllClients()
 *****************/

void
OnlyListenToOneClient(client)
    ClientPtr client;
{
    OsCommPtr oc = (OsCommPtr)client->osPrivate;

    if (!GrabDone) {
	oneClient = oc;
	GrabDone = TRUE;
    }
}

/****************
 * ListenToAllClients:
 *    Undoes OnlyListentToOneClient()
 ****************/

void
ListenToAllClients()
{
    if (GrabDone) {
	oneClient = 0;
	GrabDone = FALSE;
    }
}

static int
listen_socket(port, pport)
int port;
struct tcpport *pport;
{
    struct sockaddr addr;
    int request;

    addr.port = htons(port);
    addr.addr = 0;

    request = Socket(SOCK_STREAM, &addr, 0, (struct port *)pport);
    if (request < 0) {
	    printf("socket: error %d\n", request);
	    return -1;
    }
    if (SoBind(request, &addr) < 0) {
	    printf("Bind: error\n");
	    SoClose(request);
	    return -1;
    }
    if (SoListen(request) < 0) {
	    printf("SoListen: error\n");
	    SoClose(request);
	    return -1;
    }
    return request;
}

#ifdef X_PRINT_PORT
int
PrintOpen()
{
	printfd = open("prn", O_WRONLY);
	if(printfd < 0) {
	    SoWrite(PrinterSocket, "Can't open printer.\r\n", 21);
	    close(printfd);
	    printfd = -1;
	    SoClose(PrinterSocket);
	    PrinterSocket = listen_socket(X_PRINT_PORT, &PrinterPort);
	    return 0;
	}
	return 1;
}

int
PrintIt()
{
	char printer_buf[256];
	int print_bytes;

	print_bytes = SoRead(PrinterSocket, printer_buf, sizeof printer_buf);
	if (print_bytes <= 0) {
	    SoClose(PrinterSocket);
	    close(printfd);
	    printfd = -1;
	    PrinterSocket = listen_socket(X_PRINT_PORT, &PrinterPort);
	    return 0;
	}
	(void)write(printfd, printer_buf, print_bytes);
	return 1;
}
#endif

unsigned long
ClientNetAddr(client)
ClientPtr client;
{
    return ((OsCommPtr)client->osPrivate)->tcpport.outpkt.i.ipdest;
}

void
debug_print()
{
#ifdef DEBUG
    OsCommPtr oc;
    int i;
    ConnectionPtr occ;
    xReq *request;

/*    PrintWindowTree(); */
    if (oneClient)
	printf("Only %d\n", oneClient->client);

    for (i = 0; i < NPORTS; i++) {
	if (!MyPrivate[i])
	    continue;
	oc = MyPrivate[i];
	printf("client %d fd %d ptr %lx flags=%x", oc->client, i, oc, oc->flags);
	if ((occ = oc->output) != 0 && occ->count) {
		printf(" output count %d ", occ->count);
	}
	if ((occ = oc->input) != 0 && occ->count) {
		printf(" input count %d ", occ->count);
		request = (xReq *)occ->buffer;
		printf(" request %d len %d",  request->reqType, request->length);
	}
	printf("\n");
    }
#endif
}
