; Cute Mouse Driver - a tiny mouse driver
; Copyright (c) 1997-2000 Nagy Daniel <nagyd@almos.vein.hu>
;
; This program is free software; you can redistribute it and/or
; modify it under the terms of the GNU General Public License
; as published by the Free Software Foundation; either version 2
; of the License, or (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU Library General Public
; License along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
;

;PS2=1

WARN

j		equ	jmp short

movSeg		macro	dest,src
		push	src
		pop	dest
	endm

saveFAR		macro	addr,segm,offs
		mov	word ptr addr[0],offs
		mov	word ptr addr[2],segm
	endm

IFz		macro	var,addr
		cmp	var,0
		jz	addr
	endm

IFnz		macro	var,addr
		cmp	var,0
		jnz	addr
	endm

PS2serv		macro	serv,errlabel
		mov	ax,serv
		int	15h
	IFNB <errlabel>
		jc	errlabel
		or	ah,ah
		jne	errlabel
	ENDIF
	endm

PrintS		macro	addr
	IFNB <addr>
	 IFDIFI <addr>,<dx>
		mov	dx,offset addr
	 ENDIF
	ENDIF
		mov	ah,9
		int	21h
	endm

POINT		struc
  X		dw	0
  Y		dw	0
	ends

driverversion	equ	626h		; Microsoft driver version

.model tiny				; it is a COM file
.code
		org	100h
start:		jmp	real_start

; UNINITIALIZED DATA 
;!!! WARNING: don't modify uninitialized data in initialization part,
;	      because them not copied to new TSR

		org	4Eh		; reuse part of PSP

;----- application state begins here -----

		evendata
StartSaveArea = $

rangemax	POINT	?		; horizontal/vertical range max
pos		POINT	?
granpos		POINT	?
mickey8		POINT	?		; mickeys per 8 pixel
;;+*doublespeed	dw	?		; double speed threshold (mickeys/sec)
startscan	dw	?		; screen mask/cursor start scanline
endscan		dw	?		; cursor mask/cursor end scanline
screenmask	db	2*16 dup (?)	; user defined screen mask
cursormask	db	2*16 dup (?)	; user defined cursor mask
cursorhidden	db	?		; 0 - cursor visible, else hidden

		evendata
StartInitArea = $

mickey		POINT	?
remain		POINT	?
		evendata
LenInitArea1 = $ - StartInitArea	; cleared by setpos_04

rangemin	POINT	?		; horizontal/vertical range min
		evendata
LenInitArea2 = $ - StartInitArea	; cleared by setupvideo

cursortype	db	?		; software/hardware
regionchk?	db	?		; indicate if region must be checked
upleft		POINT	?		; upper left of update region
lowright	POINT	?		; lower right of update region
hotspot		POINT	?		; hot spot in cursor bitmap
callmask	dw	?		; user call mask
buttstatus	dw	?		; buttons status
BUTTLASTSTATE	struc
  counter	dw	?
  lastrow	dw	?
  lastcol	dw	?
	ends
buttpress	BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
buttrelease	BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
		BUTTLASTSTATE	?
;;+*nolightpen?	db	?		; 0 = emulate light pen
		evendata
LenInitArea3 = $ - StartInitArea	; cleared by softreset_21
LenSaveArea = $ - StartSaveArea

;----- registers values for RIL -----
;!!! WARNING: registers order and RGROUPDEF contents must be fixed !!!
		evendata
StartVRegsArea = $

regs_SEQC	db	5 dup (?)
reg_MISC	db	?
regs_CRTC	db	25 dup (?)
regs_ATC	db	21 dup (?)
regs_GRC	db	9 dup (?)
reg_FC		db	?
reg_GPOS1	db	?
reg_GPOS2	db	?
		evendata
LenVRegsArea = $ - StartVRegsArea

		evendata
StartDefVRegsArea = $

def_SEQC	db	5 dup (?)
def_MISC	db	?
def_CRTC	db	25 dup (?)
def_ATC		db	21 dup (?)
def_GRC		db	9 dup (?)
def_FC		db	?
def_GPOS1	db	?
def_GPOS2	db	?
		evendata
ERRIF ($-StartDefVRegsArea ne LenVRegsArea) "VRegs area contents corrupted!"


; INITIALIZED DATA 

		evendata
StartTSRpart	label

IDstring	db	'CuteMouse 1.73',0
IDstringlen = $ - IDstring
		evendata
RILversion	label
msversion	db	driverversion / 100h,driverversion MOD 100h

shape		dw	03FFFh, 01FFFh, 00FFFh, 007FFh
		dw	003FFh, 001FFh, 000FFh, 0007Fh
		dw	0003Fh, 0001Fh, 001FFh, 000FFh
		dw	030FFh, 0F87Fh, 0F87Fh, 0FCFFh

		dw	00000h, 04000h, 06000h, 07000h
		dw	07800h, 07C00h, 07E00h, 07F00h
		dw	07F80h, 07C00h, 06C00h, 04600h
		dw	00600h, 00300h, 00300h, 00000h

;----- FAR pointers storage -----

oldint33	dd	?		; old INT 33h handler address
IFNDEF PS2
oldIRQaddr	dd	?		; old IRQ handler address
ENDIF

;----- driver state begins here -----

memtype		db	0		; 0=DOS UMB, 1=XMS UMB, 2="as is"
disabled?	db	1		; 1=driver disabled?
videounlock	db	1		; 0=screen manipulation is in progress
userprocunlock	db	1		; 0=user defined proc is in progress

IFNDEF PS2
IO_address	dw	?		; IO port number
IRQintnum	db	?,25h		; INT number of selected IRQ
PICstate	db	?		; indicates which bit to clear in PIC
IOdone		db	?		; indicates processed mouse bytes
IObyte1		db	?		; buttons state in serial protocol
IObyte4		db	?		; middle button state
increment	POINT	?
ENDIF

;----- video state begins here -----

cursorcoord	POINT	?
cursorptr	dd	0		; video memory pointer
nextrow		dw	?		; add this to reach next row
cursordrawn	db	0		; 1=indicate to restore screen data

SHIFTHIVMODE	equ	0Dh-8

videomode	db	?		; video mode number
shift320	db	?		; 1=320x* video mode
planar?		db	?		; 1=multiplanar video modes

		evendata
maxcoord	POINT	?		; max coordinate
granu		POINT	{X=0,Y=0}
spritewidth	dw	0		; sprite row length in bytes
nextxor		dw	?
ptr2buffer	dd	?		; pointer to screen data
buffer		db	3*16 dup (?)	; screen data buffer

vdata1		dw	10h,1, 10h,3, 10h,4, 10h,5, 10h,8, 08h,2
VDATA1cnt	equ	($-vdata1)/4

vdata2		dw	10h,1, 10h,4, 10h,0105h, 10h,0FF08h, 08h,0F02h
VDATA2cnt	equ	($-vdata2)/4

;----- table of pointers to registers values for RIL -----
RGROUPDEF	struc
  @port		dw	?
  @regs		dw	?
  @def		dw	?
  regscnt	db	1
  rmodify?	db	0
RGROUPDEF	ends

@videoregs	label
			; CRTC
	RGROUPDEF {@port=3D4h,@regs=regs_CRTC,@def=def_CRTC,regscnt=25}
			; Sequencer
	RGROUPDEF {@port=3C4h,@regs=regs_SEQC,@def=def_SEQC,regscnt=5}
			; Graphics controller
	RGROUPDEF {@port=3CEh,@regs=regs_GRC, @def=def_GRC, regscnt=9}
			; VGA attrib controller
	RGROUPDEF {@port=3C0h,@regs=regs_ATC, @def=def_ATC, regscnt=20}
			; VGA misc output and input
	RGROUPDEF {@port=3C2h,@regs=reg_MISC, @def=def_MISC}
			; Feature Control
	RGROUPDEF {@port=3DAh,@regs=reg_FC,   @def=def_FC}
			; Graphics 1 Position
	RGROUPDEF {@port=3CCh,@regs=reg_GPOS1,@def=def_GPOS1}
			; Graphics 2 Position
	RGROUPDEF {@port=3CAh,@regs=reg_GPOS2,@def=def_GPOS2}
singlemodify?	db	?


; IRQ HANDLERS 

IFDEF PS2
;
;				Enable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AX, BX, ES
; Call: none
;
enableIRQ	proc
		mov	ah,1			; set mouse on
		movSeg	es,cs
		mov	bx,offset PS2handler
@PS2set:	push	ax
		PS2serv	0C207h			; es:bx=ptr to handler
		pop	bx
		PS2serv	0C200h
		ret
enableIRQ	endp

;
;				Disable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AX, BX, ES
; Call: none
;
disableIRQ	proc
		xor	ax,ax
		mov	es,ax
		mov	bx,ax
		j	@PS2set
disableIRQ	endp

;

PS2handler	proc	far
		cld
		push	ds ax bx cx dx si di bp es
		movSeg	ds,cs
		mov	bp,sp
		mov	cx,[bp+24]		; Y
		mov	bx,[bp+26]		; X
		mov	ax,[bp+28]		; Status
		neg	cx			; reverse Y movement
		test	al,20h			; check Y sign bit
		jz	@@noyneg
		mov	ch,0
@@noyneg:	test	al,10h			; check X sign bit
		jz	@@noxneg
		dec	bh
@@noxneg:	db	024h			; AND AL,byte
buttonsmask	db	?
		call	mouseupdate
		pop	es bp di si dx cx bx ax ds
                retf
PS2handler	endp

ELSE
;
;			Enable serial interrupt in PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address, PICstate
; Modf: AX, DX, IOdone, IObyte4
; Call: none
;
enableIRQ	proc
;---------- set communication parameters (speed, parity, etc.)
		mov	dx,[IO_address]
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		sub	dx,3
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		add	dx,3
		db	0B0h			; MOV AL,byte
COMLCR		db	?
		out	dx,al			; {3FBh} DLAB off, no parity,
						;  stop=1, length=7/8
		inc	dx
		mov	al,0Bh
		out	dx,al			; {3FCh} DTR/RTS/OUT2 on

		sub	dx,3
		mov	al,1
		out	dx,al			; {3F9h} DR int enable

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	[IOdone],al
		mov	[IObyte4],al
;----------
		in	al,21h			; {21h} get PIC mask
		mov	ah,[PICstate]
		not	ah
		and	al,ah			; enable serial interrupts
		out	21h,al
		ret
enableIRQ	endp

;
;			Disable serial interrupt of PIC
;
;
; In:	none
; Out:	none
; Use:	PICstate, IO_address
; Modf: AL, DX
; Call: none
;
disableIRQ	proc
		in	al,21h			; {21h} get PIC mask
		or	al,[PICstate]		; disable serial interrupts
		out	21h,al
;----------
		mov	dx,[IO_address]
		add	dx,3
		mov	al,0
		out	dx,al			; {3FBh} set DLAB off

		inc	dx
		in	al,dx			; {3FCh} modem ctrl
		and	al,0F3h
		out	dx,al			; {3FCh} OUT2 off

		sub	dx,3
		mov	al,0
		out	dx,al			; {3F9h} all interrupts off
		ret
disableIRQ	endp

;

IRQhandler	proc	far
		cld
		push	ds ax bx cx dx si di bp es
		movSeg	ds,cs
		mov	al,20h
		out	20h,al			; {20h} end of interrupt

		mov	dx,[IO_address]
		add	dx,5
		in	al,dx			; {3FDh} check for overrun
		mov	ah,al
		sub	dx,5
		in	al,dx			; {3F8h} flush receive buffer
		mov	cl,[IOdone]
		mov	ch,0

		test	ah,2
		jz	@@nooverrun		; jump if no overrun occured
		mov	[IOdone],ch		; zero counter
		j	@@exitIRQ

@@nooverrun:	test	ah,1
		jz	@@exitIRQ		; jump if data not ready
		db	0E8h			; CALL NEAR to mouseproc
mouseproc	dw	?

@@exitIRQ:	jmp	@rethandler
IRQhandler	endp

;
;		Process mouse bytes the Microsoft/Logitech way
;

MSLGproc	proc
		test	al,40h			; synchro check
		jz	@@MSLGbyte2		; jump if non first byte

		cmp	cx,3			; first byte after 3 bytes?
MSLGJUMP1	db	?			; jmp short for MS3/jne
		db	offset @@MSLGbyte1-offset MSLGJUMP1-2
		mov	[IObyte4],ch		; middle button released
@@MSLGbyte1:	mov	[IOdone],1		; request next 2/3 bytes
		mov	[IObyte1],al
@@MSLGprocret:	ret

@@MSLGbyte2:	jcxz	@@MSLGprocret		; skip nonfirst byte at start
		inc	[IOdone]		; request next byte
		loop	@@MSLGbyte3
		mov	byte ptr [increment.X],al ; store X increment LO
		ret

@@MSLGbyte3:	loop	@@LGbyte4
		mov	cl,al
		mov	ah,[IObyte1]

		mov	al,0
		shr	ax,2			; bits 1/0 - X increment HI
		or	al,byte ptr [increment.X]
		mov	bh,ah
		cbw
		xchg	bx,ax			; X increment

		mov	al,0
		shr	ax,2			; bits 3/2 - Y increment HI
		or	al,cl
		mov	ch,ah			; bits 5/4 - L/R buttons
		cbw
		xchg	cx,ax			; Y increment

		mov	al,[IObyte4]
		mov	dl,al
		and	ax,304h			; clear service info
		or	al,ah
		db	0B6h			; MOV DH,byte
MSLGJUMP2	db	?			; zero if MS3 mouse
		xor	dl,al			; zero if buttons not changed
		or	dx,bx
		or	dx,cx			; zero if mouse not moved
		jz	@@MSLGupdate3
		j	@@MSLGupdate

@@LGbyte4:	mov	[IOdone],ch		; CH=0, request next 3/4 bytes
MSLGJUMP3	dw	?			; nop for LT/j @@MSLGprocret
						; MS mode defines only 3 bytes
		mov	cl,5
		shl	ax,cl
		mov	al,[IObyte4]
		xor	ah,al
		and	ah,4
		jz	@@MSLGprocret		; return if state not changed
		xor	cx,cx
		xor	bx,bx
@@MSLGupdate3:	xor	al,4			; toggle middle button state
@@MSLGupdate:	mov	[IObyte4],al
		j	mouseupdate
MSLGproc	endp

;
;		Process mouse bytes the Mouse Systems way
;

MSMproc		proc
		cbw
		jcxz	@@MSMbyte1
		dec	cx
		jz	@@MSMbyte24
		dec	cx
		jz	@@MSMbyte3
		loop	@@MSMbyte5

@@MSMbyte24:	add	[increment.X],ax
@@MSMnext:	inc	[IOdone]		; request next byte
@@MSMprocret:	ret

@@MSMbyte1:	mov	[IObyte1],al		; save buttons state
		mov	[increment.X],cx	; CX=0
		and	al,not 7
		cmp	al,80h			; sync check
		jne	@@MSMprocret

@@MSMbyte3:	mov	[increment.Y],ax
		j	@@MSMnext

@@MSMbyte5:	mov	[IOdone],ch		; CH=0
		add	ax,[increment.Y]
		mov	cx,ax
		neg	cx
		mov	bx,[increment.X]
		mov	al,[IObyte1]		; bits 2/1/0 - L/M/R buttons
		not	al			;  (0 if pressed)
		and	al,7
		test	al,6			; check the L and M state
		jpe	@@MSMupdate		; skip if the same
		xor	al,6			; swap L and M buttons
@@MSMupdate:	;j	mouseupdate
MSMproc		endp
ENDIF

;
;			Update mouse status
;
;
; In:	AL					(new buttons state)
;	BX					(X increment)
;	CX					(Y increment)
; Out:	none
; Use:	mickey8, granu, callmask
; Modf: AX, BX, CX, DX, mickey, remain, pos, granpos, buttstatus,
;	userprocunlock
; Call: resolute, cutrangex, cutrangey, updatebutton, showcursor
;
mouseupdate	proc
		push	ax

;---------- calculate X movement in pixels
		mov	ax,bx
		or	ax,ax			; is X movement 0?
		jz	@@xmov0			;  jump if yes

		db	0B2h			; MOV DL,byte
mresolution0	db	3
		call	resolute
		add	[mickey.X],ax

		shl	ax,3
		add	ax,[remain.X]
		cwd
		idiv	[mickey8.X]
		mov	[remain.X],dx

		add	ax,[pos.X]
		call	cutrangex
		mov	[pos.X],ax
		xor	dx,dx
		div	[granu.X]
		mul	[granu.X]
		mov	dx,ax
		xchg	ax,[granpos.X]		; the new column is ready
		sub	ax,dx			; X screen shift
@@xmov0:
;---------- calculate Y movement in pixels
		jcxz	@@ymov0			; jump if Y movement is 0
		xchg	ax,cx

		db	0B2h			; MOV DL,byte
mresolution1	db	3
		call	resolute
		add	[mickey.Y],ax

		shl	ax,3
		add	ax,[remain.Y]
		cwd
		idiv	[mickey8.Y]
		mov	[remain.Y],dx

		add	ax,[pos.Y]
		call	cutrangey
		mov	[pos.Y],ax
		xor	dx,dx
		div	[granu.Y]
		mul	[granu.Y]
		mov	dx,ax
		xchg	ax,[granpos.Y]		; the new row is ready
		sub	ax,dx			; Y screen shift
@@ymov0:
;----------
		or	cx,ax			; if both are 0
		neg	cx			; then no carry flag
		sbb	cx,cx
		neg	cx			; and result 0, else 1

		pop	ax
		test	al,3			; check the L and R buttons
LEFTHANDJUMP	db	?			; JMP SHORT for PS2^LHAND/JPE
		db	offset @@updatebut-offset LEFTHANDJUMP-2
		xor	al,3			; swap

@@updatebut:	mov	ah,al			; AH=buttons state
		mov	ch,al
		xchg	al,byte ptr [buttstatus]
		xor	ch,al			; CH=buttons change state
		mov	al,cl			; AL=unrolled change state

		mov	cl,2			; indicate that 1 is pressed
		mov	bx,offset buttpress
		call	updatebutton
		mov	cl,8			; indicate that 2 is pressed
		mov	bx,offset buttpress+SIZE BUTTLASTSTATE
		call	updatebutton
		mov	cl,20h			; indicate that 3 is pressed
		mov	bx,offset buttpress+(SIZE BUTTLASTSTATE)*2
		call	updatebutton
		dec	[userprocunlock]
		jnz	@@updatedone		; exit if user proc running

;---------- call User Defined Handler
		mov	bx,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		mov	si,[mickey.X]
		mov	di,[mickey.Y]
		sti
		and	al,byte ptr [callmask]	; is there a user call mask?
		jz	@@updateshow		; exit if not
		db	9Ah			; CALL FAR to userproc
userproc	dd	0
		movSeg	ds,cs
;----------
@@updateshow:	call	showcursor

@@updatedone:	cli
		inc	[userprocunlock]
		ret
mouseupdate	endp

;
;		Update one button status regs to new values
;
;
; In:	AL					(unrolled button change state)
;	AH					(current button state)
;	CL					(press bit mask)
;	CH					(button change state)
;	BX					(offset to item of buttpress)
; Out:	AX, CH
; Use:	granpos
; Modf: BX, CL, buttpress, buttrelease
; Call: none
;
updatebutton	proc
		shr	ah,1
		jc	@@updset		; jump if pressed
		shl	cl,1			; indicate that is released
		add	bx,offset buttrelease-offset buttpress
@@updset:	shr	ch,1
		jnc	@@updret		; jump if button unchanged
		or	al,cl
		inc	[bx.counter]
		push	ax
		mov	ax,[granpos.Y]
		mov	[bx.lastrow],ax
		mov	ax,[granpos.X]
		mov	[bx.lastcol],ax
		pop	ax
@@updret:	ret
updatebutton	endp

;
;			Use selected resolution
;
;
; In:	AX					(X/Y increment)
;	DL					(multiplier)
; Out:	AX					(scaled increment)
; Use:	none
; Modf: DX
; Call: none
;
resolute	proc
		or	dl,dl
		jnz	@@resolute
		sar	ax,2
		mov	dl,al
		and	dl,0Fh			;*** ??? *** signed value
		cmp	dl,10
		jbe	@@resolute
		mov	dl,10

@@resolute:	cmp	dl,1			; is resolution 1 or 0?
		jbe	@@resret		;  exit if yes
		cmp	ax,3			; was movement smaller than +3?
		jb	@@resret		;  exit if yes
		cmp	ax,-3			; was movement larger than -3?
		ja	@@resret		;  exit if yes
		cmp	ax,8			; was movement between +3 and +8?
		jb	@@ressmall		;  jump if yes
		cmp	ax,-8			; was movement between -3 and -8?
		ja	@@ressmall		;  jump if yes
		mov	dh,0
		imul	dx			; else multiply it with resolution
		ret
@@ressmall:	shl	ax,1			; small movement -> multiply by 2
@@resret:	ret
resolute	endp

;
;			Cut coordinate inside range
;

cutrangex	proc
		cmp	ax,[rangemin.X]
		jg	@@cutxhi
		mov	ax,[rangemin.X]
@@cutxhi:	cmp	ax,[rangemax.X]
		jl	@@cutxret
		mov	ax,[rangemax.X]
@@cutxret:	ret
cutrangex	endp

cutrangey	proc
		cmp	ax,[rangemin.Y]
		jg	@@cutyhi
		mov	ax,[rangemin.Y]
@@cutyhi:	cmp	ax,[rangemax.Y]
		jl	@@cutyret
		mov	ax,[rangemax.Y]
@@cutyret:	ret
cutrangey	endp

; END OF IRQ HANDLERS 


; INT 10 HANDLER 

		evendata
RILfuncs	dw	offset RIL_F0		; RIL functions
		dw	offset RIL_F1
		dw	offset RIL_F2
		dw	offset RIL_F3
		dw	offset RIL_F4
		dw	offset RIL_F5
		dw	offset RIL_F6
		dw	offset RIL_F7

int10handler	proc
		cld
		or	ah,ah			; set video mode?
		jz	@@setmodreq
;;+*		cmp	ah,4			; light pen func?
;;+*		je	@@lightpen
		cmp	ah,0F0h			; RIL func requested?
		jb	@@jmpold10
		cmp	ah,0F7h
		jbe	@@RIL
		cmp	ah,0FAh
		je	@@RIL_FA
@@jmpold10:	db	0EAh			; JMP FAR to old Int10 handler
oldint10	dd	?

;========== RIL
@@RIL:		push	ax cx dx ds es di si
		movSeg	ds,cs
		mov	al,ah
		and	ax,0Fh
		shl	ax,1
		mov	si,ax
		call	RILfuncs[si]
		pop	si di es ds dx cx ax
		iret

@@RIL_FA:	movSeg	es,ds			; INTERROGATE DRIVER
		mov	bx,offset RILversion
		iret

;========== emulate lightpen
;;+*@@lightpen:	IFnz	cs:[nolightpen?],@@jmpold10
;;+*		sti
;;+*		mov	ax,cs:[granpos.Y]
;;+*		mov	ch,al
;;+*		cmp	cs:[videomode],0Fh-SHIFTHIVMODE
;;+*		jb	@@lpen2
;;+*		mov	cx,ax
;;+*@@lpen2:	div	cs:[pixboxheight]
;;+*		mov	dh,al
;;+*		mov	ax,cs:[granpos.X]
;;+*		mov	bx,ax
;;+*		cwd
;;+*		div	cs:[granu.X]
;;+*		xchg	bx,ax
;;+*		div	cs:[pixboxwidth]
;;+*		mov	dl,al
;;+*		mov	ah,1
;;+*		iret

;========== set video mode
@@setmodreq:	push	ax
		mov	cs:[cursorhidden],0
		mov	ax,2
		int	33h			; mouse driver, hide cursor
		pop	ax
		pushf
		call	cs:[oldint10]
		push	ds ax bx cx dx si di bp es
		call	setupvideo
		jmp	@rethandler
int10handler	endp

;
;			RIL functions
;

RIL_F0		proc				; READ ONE REGISTER
		mov	si,dx
		mov	si,@videoregs[si].@regs
		cmp	dx,20h
		jge	@@R0ret
		add	si,bx
@@R0ret:	mov	bl,[si]
		ret
RIL_F0		endp

;

RIL_F1		proc				; WRITE ONE REGISTER
		mov	ax,bx
		mov	si,dx
		cmp	dl,20h
		mov	dx,@videoregs[si].@port
		mov	@videoregs[si].rmodify?,dl
		mov	si,@videoregs[si].@regs
		jb	@@R0group		; jump if group of registers

		mov	[singlemodify?],dl
		mov	[si],al
		out	dx,al
		ret

@@R0group:	mov	bh,0
		mov	[bx+si],ah
		mov	bl,ah
		;j	RIL_write
RIL_F1		endp

;
;
; In:	DX					(IO port)
;	AL					(reg number)
;	AH					(value to write)
; Out:	none
; Use:	none
; Modf: none
; Call: none
;
RIL_write	proc
		cmp	dl,0C0h
		je	@@RwATC			; jump if ATTR controller
		out	dx,ax
		ret
@@RwATC:	push	ax dx
		mov	dl,byte ptr @videoregs[(SIZE RGROUPDEF)*5].@port
		in	al,dx			; {3DAh} force address mode
		pop	dx ax
		out	dx,al			; {3C0h} select ATC register
		xchg	al,ah
		out	dx,al			; {3C0h} modify ATC register
		xchg	al,ah
		ret
RIL_write	endp

;

RIL_F2		proc				; READ REGISTER RANGE
		sti
		mov	di,bx
		mov	si,dx
		mov	si,@videoregs[si].@regs
		xor	ax,ax
		xchg	al,ch
		add	si,ax
RILmemcopy:	shr	cx,1
		rep	movsw
		adc	cx,cx
		rep	movsb
		ret
RIL_F2		endp

;

RIL_F3		proc				; WRITE REGISTER RANGE
		sti
		mov	si,bx
		mov	di,dx
		mov	dx,@videoregs[di].@port
		mov	@videoregs[di].rmodify?,dl
		mov	di,@videoregs[di].@regs
		xor	ax,ax
		xchg	al,ch
		add	di,ax
@@R3loop:	mov	ah,al
		lods	byte ptr es:[si]
		mov	[di],al
		inc	di
		xchg	al,ah
		call	RIL_write
		inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R3loop
		ret
RIL_F3		endp

;

RIL_F4		proc				; READ REGISTER SET
		sti
		mov	di,bx
@@R4loop:	mov	si,es:[di]
		inc	di
		inc	di
		mov	si,@videoregs[si].@regs
		mov	al,es:[di]
		inc	di
		cbw
		add	si,ax
		movsb
		loop	@@R4loop
		ret
RIL_F4		endp

;

RIL_F5		proc				; WRITE REGISTER SET
		sti
		mov	si,bx
@@R5loop:	lods	word ptr es:[si]
		mov	di,ax
		lods	word ptr es:[si]
		mov	dx,@videoregs[di].@port
		mov	@videoregs[di].rmodify?,dl
		mov	di,@videoregs[di].@regs
		push	ax
		cbw
		add	di,ax
		pop	ax
		mov	[di],ah
		call	RIL_write
		loop	@@R5loop
		ret
RIL_F5		endp

;

RIL_F6		proc				; REVERT REGISTERS TO DEFAULT
		sti
		push	bx
		movSeg	es,ds
		mov	bx,offset @videoregs
		xor	cx,cx

@@R6loop:	cmp	[bx].rmodify?,ch
		je	@@R6next

		mov	[bx].rmodify?,ch
		mov	cl,[bx].regscnt
		mov	si,[bx].@def
		mov	di,[bx].@regs
		mov	dx,[bx].@port
		mov	al,0

@@R6rloop:	mov	ah,al
		lodsb
		stosb
		xchg	al,ah
		cmp	dx,03CEh		; {3CEh} graphics controller
		jne	@@R6noGRC
		cmp	al,6
		je	@@R6rnext		; skip register 6 of GRC
@@R6noGRC:	call 	RIL_write
@@R6rnext:	inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R6rloop

@@R6next:	add	bx,SIZE RGROUPDEF
		cmp	bx,offset @videoregs+(SIZE RGROUPDEF)*4
		jb	@@R6loop

		cmp	[singlemodify?],ch
		je	@@R6ret
		mov	[singlemodify?],ch
		mov	dx,@videoregs[(SIZE RGROUPDEF)*5].@port
		mov	al,[def_FC]
		mov	[reg_FC],al
		out	dx,al			; {3DAh}
		mov	dx,3C2h
		mov	al,[def_MISC]
		mov	[reg_MISC],al
		out	dx,al			; {3C2h} EGA misl out reg
		mov	dx,3CCh
		mov	al,[def_GPOS1]
		mov	[reg_GPOS1],al
		out	dx,al			; {3CCh} EGA graphics 1 pos
		mov	dx,3CAh
		mov	al,[def_GPOS2]
		mov	[reg_GPOS2],al
		out	dx,al			; {3CAh} EGA graphics 2 pos

@@R6ret:	pop	bx
		ret
RIL_F6		endp

;

RIL_F7		proc				; DEFINE REGISTERS DEFAULTS
		sti
		mov	si,bx
		mov	di,dx
		mov	cl,@videoregs[di].regscnt
		mov	@videoregs[di].rmodify?,cl
		mov	[singlemodify?],cl
		mov	ch,0
		mov	di,@videoregs[di].@def
		push	es ds
		pop	es ds
		jmp	RILmemcopy
RIL_F7		endp

; END OF INT 10 HANDLER 


;
;			Draw mouse cursor
;

showcursor	proc
		cmp	[cursorhidden],0
		jne	@showret		; exit if cursor hidden
		dec	[videounlock]
		jnz	@showdone		; exit if drawing in progress

		mov	ax,[granu.Y]
		dec	ax
		jnz	drawtextcursor		; jump if text mode
		cmp	[videomode],13h-SHIFTHIVMODE
		ja	@showdone
		call	drawgraphcurs		; else draw graphics cursor
		j	@showdone
showcursor	endp

;
;			Restore old screen content
;

restorescreen	proc
		dec	[videounlock]
		jnz	@showdone		; exit if drawing in progress
		xor	cx,cx
		xchg	cl,[cursordrawn]	; clear indicator
		jcxz	@showdone		; exit if cursor not drawn

		mov	ax,[granu.Y]
		dec	ax
		jz	@@restgraph		; jump if graphics mode
		IFnz	[cursortype],@showdone	; exit if hardware cursor
		call	restoretext
		j	@showdone

@@restgraph:	call	updatevregs
		call	restoresprite
		call	restorevregs

@showdone:	inc	[videounlock]		; drawing stopped
@showret:	ret
restorescreen	endp

;
;			Draw text mode cursor
;

drawtextcursor	proc
		call	gettxtoffset
		IFz	[cursortype],@@softcurs	; jump if software cursor

;---------- draw hardware text mode cursor
		shr	di,1
		mov	dx,@videoregs[0].@port	; CRTC port

		mov	ax,di
		mov	ah,al
		mov	al,0Fh
		out	dx,ax			; cursor position lo

		mov	ax,di
		mov	al,0Eh
		out	dx,ax			; cursor position hi
		j	@showdone

;---------- draw software text mode cursor
@@softcurs:	xor	cx,cx
		xchg	cl,[cursordrawn]	; clear indicator
		jcxz	@@drawtext		; jump if cursor not drawn
		cmp	di,word ptr cursorptr[0]
		je	@@drawtextdone		; exit if position not changed
		push	di
		call	restoretext
		pop	di

@@drawtext:	mov	ax,[granpos.Y]		; get cursor position
		mov	bx,[granpos.X]
		mov	cx,[granu.Y]		; get cursor size
		mov	dx,[granu.X]
		call	checkregion
		jc	@showdone		; exit if in update region

		mov	word ptr cursorptr[0],di
;;*		cli
;;*		call	waitretrace
		mov	ax,es:[di]		; save char under cursor
		mov	word ptr buffer[0],ax
		and	ax,[startscan]
		xor	ax,[endscan]
		stosw				; draw to new position
;;*		sti
		mov	word ptr buffer[2],ax

@@drawtextdone:	inc	[cursordrawn]		; we have to restore later
		j	@showdone
drawtextcursor	endp

;

restoretext	proc
		les	di,[cursorptr]
;;*		cli
;;*		call	waitretrace
		mov	ax,es:[di]
		cmp	ax,word ptr buffer[2]
		jne	@@resttxtret
		mov	ax,word ptr buffer[0]
		stosw				; restore old text char attrib
;;*		sti
@@resttxtret:	ret
restoretext	endp

;
;		Wait for video retrace in text modes
;
;
; In:	ES					(video segment)
; Out:	none
; Use:	@videoregs[(SIZE RGROUPDEF)*5].@port
; Modf: AX, CX, DX
; Call: none
;
;;*waitretrace	proc
;;*		mov	ax,es
;;*		cmp	ax,0B800h
;;*		jne	@@waitret
;;*
;;*		mov	dx,@videoregs[(SIZE RGROUPDEF)*5].@port
;;*						; {3DAh}/{3BAh}
;;*		xor	cx,cx
;;*@@waitstop:	in	al,dx			; wait until retrace stop
;;*		test	al,1
;;*		loopnz	@@waitstop
;;*
;;*		xor	cx,cx
;;*@@waitstart:	in	al,dx			; wait until retrace start
;;*		test	al,9
;;*		loopz	@@waitstart
;;*@@waitret:	ret
;;*waitretrace	endp

;
;			Draw graphics cursor
;

drawgraphcurs	proc
		call	updatevregs
		mov	ax,[granpos.Y]
		sub	ax,[hotspot.Y]		; screen Y
		mov	di,[granpos.X]
		sub	di,[hotspot.X]
		mov	bx,di			; virtual X
		mov	cl,byte ptr granu.X[0]
		dec	cx			; OPTIMIZE: CX instead CL
		sar	di,cl
		xor	cl,[shift320]
		shl	di,cl			; screen X

		xor	cx,cx
		xchg	cl,[cursordrawn]	; clear indicator
		jcxz	@@drawgraph		; jump if cursor not drawn
		cmp	ax,[cursorcoord.Y]
		jne	@@restspr
		cmp	di,[cursorcoord.X]
		je	@@drawgrdone		; exit if position not changed
@@restspr:	call	restoresprite

@@drawgraph:	mov	cx,16			; get cursor size
		mov	dx,cx
		cmp	[videomode],13h-SHIFTHIVMODE
		je	@@checkreg
		and	bl,not 7		; OPTIMIZE: BL instead BX
		mov	dl,3*8			; OPTIMIZE: DL instead DX
@@checkreg:	call	checkregion
		jc	restorevregs		; exit if in update region

		mov	[cursorcoord.Y],ax
		mov	[cursorcoord.X],di
		push	ax			; push [cursorcoord.Y]
		call	getgroffset
		mov	word ptr cursorptr[0],di
		mov	[nextrow],cx
		push	cx ds			; push [nextrow]
		les	di,[ptr2buffer]
		lds	si,[cursorptr]
		mov	bl,0
		call	copysprite		; save new sprite
		pop	ds
;----------
		cmp	[planar?],bl
		jz	@@make
		mov	dx,3CEh
		mov	ax,5
		out	dx,ax			; {3CEh},{3CFh} set write mode

@@make:		les	di,[cursorptr]
		pop	cx			; [nextrow]
		pop	ax			; [cursorcoord.Y]
		xor	bx,bx			; mask offset (2*16 bytes)

@@makeloop:	cmp	ax,[maxcoord.Y]
		jae	@@makenext1		; jump if Y > max || Y < 0

		push	ax bx cx di
		mov	si,[cursorcoord.X]
		cmp	[videomode],13h-SHIFTHIVMODE
		je	make13
		call	makeno13
@makenext:	pop	di cx bx ax

@@makenext1:	inc	ax
		add	di,cx
		xor	cx,[nextxor]
		inc	bx
		inc	bx
		cmp	bx,2*16
		jb	@@makeloop
;----------
@@drawgrdone:	inc	[cursordrawn]
		;j	restorevregs
drawgraphcurs	endp

;
;		Restore graphics card video registers
;

restorevregs	proc
		IFz	[planar?],@nogrparam
		movSeg	es,cs
		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		j	@writevideo
restorevregs	endp

;
;		Save & update graphics card video registers
;

updatevregs	proc
		IFz	[planar?],@nogrparam
		movSeg	es,cs

		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		mov	ah,0F4h			; read register set
		int	10h

		mov	bx,offset vdata2
		mov	cx,VDATA2cnt
@writevideo:	mov	ah,0F5h			; write register set
		int	10h
@nogrparam:	ret
updatevregs	endp

;
;		Transform the cursor mask to screen content
;
;
; In:	SI					(X coordinate)
;	BX					(mask offset)
;	ES:DI					(video memory pointer)
; Out:	none
; Use:	maxcoord.X, screenmask, cursormask
; Modf: AL, BX, CX, DX, SI, DI
; Call: none
;
make13		proc
		mov	dx,word ptr screenmask[bx]
		mov	bx,word ptr cursormask[bx]
		mov	cx,16			; 2*8 bytes per row

@@loopx13:	cmp	si,[maxcoord.X]
		jae	@@nextx13		; jump if X > max || X < 0
		mov	al,0
		or	dx,dx
		jge	@@chkcmask		; jump if most sign bit zero
		mov	al,es:[di]
@@chkcmask:	or	bx,bx
		jge	@@putbyte13		; jump if most sign bit zero
		xor	al,0Fh
@@putbyte13:	mov	es:[di],al
@@nextx13:	inc	si
		shl	dx,1
		shl	bx,1
		inc	di
		loop	@@loopx13
		j	@makenext
make13		endp

;
;		Display cursor in other than mode 13h modes
;
;
; In:	SI					(X coordinate)
;	BX					(mask offset)
;	ES:DI					(video memory pointer)
; Out:	none
; Use:	maxcoord.X, screenmask, cursormask
; Modf: AX, BX, CX, DX, SI, DI
; Call: none
;
makeno13	proc
		mov	cx,si
		and	cl,7

		mov	dx,word ptr screenmask[bx]
		ror	dx,cl
		mov	ax,0ffh
		ror	ax,cl
		push	ax
		or	al,dh
		or	dh,ah			; dh:dl:al shifted screenmask

		mov	bx,word ptr cursormask[bx]
		ror	bx,cl
		pop	cx
		not	cx
		and	ch,bh
		and	bh,cl			; ch:bl:bh shifted cursormask

		mov	cl,dh
		mov	dh,bl
		mov	bl,al			; cl:dl:bl - ch:dh:bh
		push	dx
		call	@@makeno13
		pop	cx
		call	@@makeno13
		mov	cx,bx

@@makeno13:	cmp	si,[maxcoord.X]
		jae	@@nextno13		; jump if X > max || X < 0

		IFnz	[planar?],@@planar
		mov	al,es:[di]
		and	al,cl
		xor	al,ch
		stosb
		j	@@nextxno13

@@planar:	mov	dx,3CEh
		mov	ax,0803h		; data ANDed with latched data
		out	dx,ax
		xchg	es:[di],cl
		mov	ax,1803h		; data XORed with latched data
		out	dx,ax
		xchg	es:[di],ch

@@nextno13:	inc	di
@@nextxno13:	add	si,8
		ret
makeno13	endp

;

restoresprite	proc
		push	bx di ds
		les	di,[cursorptr]
		mov	cx,[nextrow]
		lds	si,[ptr2buffer]
		mov	bl,1
		call	copysprite
		pop	ds di bx
		ret
restoresprite	endp

;
;		Copy screen sprite back and forth
;
;
; In:	BL					(0-screen src./1-screen dst.)
;	ES:DI					(video memory pointer)
;	DS:SI					(= ptr2buffer)
;	CX					(offset to next row)
; Out:	none
; Use:	cursorcoord, maxcoord, nextxor, spritewidth
; Modf: BH, CX, DX, SI, DI
; Call: none
;
copysprite	proc
		push	ax
		mov	bh,16
		mov	dx,cs:[spritewidth]	; mode 13h=16, other modes=3
		mov	ax,cs:[cursorcoord.Y]

@@copyloop:	cmp	ax,cs:[maxcoord.Y]
		jae	@@copynext1		; jump if Y > max || Y < 0

		push	ax cx dx si di
		mov	cx,dx
		shr	dx,1			; X step per byte (mode 13h=1,
		xor	dl,8 xor 1		;  other modes=8)
		mov	ax,cs:[cursorcoord.X]
@@copyloopx:	cmp	ax,cs:[maxcoord.X]
		jae	@@copynextx		; jump if X > max || X < 0
		movsb
		dec	si
		dec	di
@@copynextx:	add	ax,dx
		inc	si
		inc	di
		loop	@@copyloopx
		pop	di si dx cx ax

@@copynext1:	inc	ax
		or	bl,bl
		jnz	@@copynext2
		add	si,cx
		add	di,dx
		j	@@copynext3
@@copynext2:	add	di,cx
		add	si,dx
@@copynext3:	xor	cx,cs:[nextxor]
		dec	bh
		jnz	@@copyloop
		pop	ax
		ret
copysprite	endp

;
;		Return graphic mode video memory offset
;
;
; In:	AX					(Y coordinate in pixels)
;	DI					(X coordinate in pixels)
; Out:	ES:DI					(video memory pointer)
;	CX					(offset to next row)
; Use:	videomode
; Modf: AX, BX, DX, nextxor
; Call: @getoffsret
;
getgroffset	proc
;
;4/5 (320x200x4)   byte offset = (y%2)*2000h + (y/2)*80 + (x*2)/8,
;			low bit (of pair) offset = 6 - (x*2)%8
;  6 (640x200x2)   byte offset = (y%2)*2000h + (y/2)*80 + x/8,
;			bit offset = 7 - (x % 8)
;0Dh (320x200x16)  byte offset = y*40 + x/8, bit offset = 7 - (x % 8)
;0Eh (640x200x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;0Fh (640x350x4)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;10h (640x350x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;11h (640x480x2)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;12h (640x480x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;13h (320x200x256) byte offset = y*320 + x
;HGC (720x348x2)   byte offset = (y%4)*2000h + (y/4)*90 + x/8
;			bit offset = 7 - (x % 8)
;
		mov	bx,320			; length of scanline
		cmp	[videomode],13h-SHIFTHIVMODE
		je	@@interlace
		mov	cl,3
		sar	di,cl			; DI=X/8
		mov	bx,80
		mov	cl,[shift320]
		shr	bx,cl

@@interlace:	mov	cx,bx			; [nextrow]
		xor	dx,dx			; [nextxor]
		cmp	[videomode],6		; is videomode 4-6?
		ja	@@ymul			; jump if not
		mov	cx,2000h
		mov	dx,2000h xor -(2000h-80)
		sar	ax,1			; AX=Y/2
		jnc	@@ymul
		add	di,cx			; DI=X/8+(Y%2)*2000h
		xor	cx,dx

@@ymul:		mov	[nextxor],dx
		mul	bx
		add	di,ax
		j	@getoffsret
getgroffset	endp

;
;		Return text mode video memory offset
;
;
; In:	none
; Out:	ES:DI					(video memory pointer)
; Use:	0:44Ah, 0:44Eh, granpos, granu, cursorptr[2]
; Modf: AX, DX
; Call: getpageoffset
;
gettxtoffset	proc
;
;0/1 (40x25) byte offset = (y*40 + x)*2 = (y*[0:44Ah] + x)*2
;2/3 (80x25) byte offset = (y*80 + x)*2 = (y*[0:44Ah] + x)*2
;  7 (80x25) byte offset = (y*80 + x)*2 = (y*[0:44Ah] + x)*2
;
		mov	ax,[granpos.Y]
		xor	dx,dx
		mov	es,dx
		div	[granu.Y]		; calculate Y in characters
		mul	word ptr es:[44Ah]	; screen width
		mov	di,ax
		mov	ax,[granpos.X]
		;xor	dx,dx
		div	[granu.X]		; calculate X in characters
		add	di,ax
		shl	di,1

@getoffsret:	xor	ax,ax
		mov	es,ax
		add	di,es:[44Eh]		; add page offset
		mov	es,word ptr cursorptr[2]
		ret
gettxtoffset	endp

;
;		Check if cursor in update region
;

checkregion	proc
		IFz	[regionchk?],@@regexit	; exit if not to check region
		cmp	ax,[lowright.Y]
		jg	@@regexit
		cmp	bx,[lowright.X]
		jg	@@regexit
		add	cx,ax
		cmp	cx,[upleft.Y]
		jle	@@regexit
		add	dx,bx
		cmp	dx,[upleft.X]
		jle	@@regexit
		stc
		ret
@@regexit:	clc
		ret
checkregion	endp

;
;		Setup video regs values for current video mode
;
;
; In:	none
; Out:	none
; Use:	0:449h, 0:463h, 0:487h, 0:488h, 0:4A8h; buffer
; Modf: *
; Call: @setcoords
;
setupvideo	proc
		xor	ax,ax
		mov	ds,ax
		mov	ax,ds:[463h]		; CRTC base I/O port address
		mov	cs:@videoregs[0].@port,ax ; 3D4h/3B4h
		add	ax,6
		mov	cs:@videoregs[(SIZE RGROUPDEF)*5].@port,ax
		mov	al,ds:[449h]		; current video mode
		and	al,not 80h
		push	ax

		cmp	al,11h			; VGA videomodes?
		jb	@@checkv1
		add	al,9
		j	@@getshift

@@checkv1:	cmp	al,0Fh			; 0F-10 videomodes?
		jb	@@checkv2
		test	byte ptr ds:[487h],60h	; check RAM size on adapter
		jz	@@getshift
		add	al,2
		j	@@getshift

@@checkv2:	cmp	al,4			; not color text modes?
		jae	@@getshift
		mov	ah,ds:[488h]		; get display combination
		and	ah,1Fh
		cmp	ah,3			; MDA+EGA/ECD?
		je	@@lines350
		cmp	ah,9			; EGA/ECD+MDA?
		jne	@@getshift
@@lines350:	add	al,13h

@@getshift:	les	si,dword ptr ds:[4A8h]
		les	si,dword ptr es:[si]
		mov	ah,0
		mov	cl,6
		shl	ax,cl
		add	al,5			; OPTIMIZE: AL instead AX
		add	si,ax			; SI += AL*64+5

		movSeg	ds,es
		movSeg	es,cs
		mov	di,offset StartDefVRegsArea
		push	di
		mov	al,3
		stosb
		mov	cx,50/2
		rep	movsw			; copy default registers value
		inc	di
		movsw
		movsw
		movsw
		movsw
		movsb

		movSeg	ds,cs
		pop	si
		mov	di,offset StartVRegsArea
		mov	cx,LenVRegsArea/2
		rep	movsw			; copy current registers value

		mov	al,0
		mov	@videoregs[(SIZE RGROUPDEF)*0].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*1].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*2].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*3].rmodify?,al
		mov	[singlemodify?],al
		mov	[reg_GPOS1],al
		mov	[def_GPOS1],al
		inc	ax			; OPTIMIZE: AX instead AL
		mov	[reg_GPOS2],al
		mov	[def_GPOS2],al

;---------- set parameters for current video mode
; mode	 seg   screen  cell  planar
;  0	B800h  640x200 16x8	-
;  1	B800h  640x200 16x8	-
;  2	B800h  640x200	8x8	-
;  3	B800h  640x200	8x8	-
;  4	B800h  640x200	2x1	0
;  5	B800h  640x200	2x1	0
;  6	B800h  640x200	1x1	0
;  7	B000h  640x200	8x8	-
; 0Dh	A000h  320x200	2x1	1
; 0Eh	A000h  640x200	1x1	1
; 0Fh	A000h  640x350	1x1	1
; 10h	A000h  640x350	1x1	1
; 11h	A000h  640x480	1x1	1
; 12h	A000h  640x480	1x1	1
; 13h	A000h  320x200	2x1	0
;
		pop	ax			; current video mode
; mode 0-3
		mov	ah,0B8h			; B800h: 0-6
		mov	[maxcoord.X],640
		mov	di,200			; x200: 0-7, 0Dh-0Eh, 13h
		mov	dx,0810h		; 16x8: 0-1
		mov	cx,300h			; sprite: 3 bytes/row; 640x
		mov	[planar?],cl
		mov	si,cs
		mov	bx,offset buffer
		cmp	al,2
		jb	@@setvmode
		mov	dl,8			; 8x8: 2-3, 7
		cmp	al,4
		jb	@@setvmode
; mode 7
		mov	ah,0B0h			; B000h: 7
		cmp	al,7
		je	@@setvmode
; mode 4-6
		mov	ah,0B8h
		mov	dx,0101h		; 1x1: 6, 0Eh-12h
		cmp	al,6
		je	@@setvmode
		jb	@@set2x1
; mode 13h
; for modes 0Dh-13h we'll be storing the cursor shape and the hided
; screen contents at first free byte in vidmem to save mem
		sub	al,SHIFTHIVMODE
		mov	ah,0A0h			; A000h: 0Dh-
		mov	si,0A000h
		mov	bx,0FA00h
		cmp	al,13h-SHIFTHIVMODE
		jb	@@mode0812
		ja	@@nonstandard
		mov	ch,16
@@set320:	inc	cx			; OPTIMIZE: CX instead CL
@@set2x1:	inc	dx			; OPTIMIZE: DX instead DL
		j	@@setvmode
; mode 8-0Dh
@@mode0812:	cmp	al,0Dh-SHIFTHIVMODE
		jb	@@nonstandard
		mov	bx,3E82h		; 16002: 0Dh-0Eh
		mov	[planar?],bl		; set for modes 0Dh-12h
		je	@@set320
; mode 0Eh-12h
		cmp	al,0Fh-SHIFTHIVMODE
		jb	@@setvmode
		mov	di,350			; x350: 0Fh-10h
		add	bh,40h			; 32386: 0Fh-10h
		cmp	al,11h-SHIFTHIVMODE
		jb	@@setvmode
		mov	di,480			; x480: 11h-12h
		add	bh,20h			; 40578: 11h-12h
		j	@@setvmode

@@nonstandard:	;mov	dx,0101h
		;mov	cl,0
;+++++ for nonstandard textual modes set: ah := 0B8h, dx := 0808h
		mov	al,14h-SHIFTHIVMODE

@@setvmode:	mov	[videomode],al
		mov	byte ptr cursorptr[3],ah
		shr	[maxcoord.X],cl
		mov	[maxcoord.Y],di
		mov	byte ptr granu.X[0],dl
		mov	byte ptr granu.Y[0],dh
		mov	[shift320],cl
		mov	byte ptr spritewidth[0],ch
		saveFAR	[ptr2buffer],si,bx

;---------- set parameters for ranges and cursor position
@@setuprange:	mov	cx,LenInitArea2/2	; clear area 2
		j	@setcoords
setupvideo	endp


; INT 33 HANDLER SERVICES 

_ARG_ES_	equ	word ptr [bp]
_ARG_BP_	equ	word ptr [bp+2]
_ARG_DI_	equ	word ptr [bp+4]
_ARG_SI_	equ	word ptr [bp+6]
_ARG_DX_	equ	word ptr [bp+8]
_ARG_CX_	equ	word ptr [bp+10]
_ARG_BX_	equ	word ptr [bp+12]
_ARG_AX_	equ	word ptr [bp+14]
_ARG_DS_	equ	word ptr [bp+16]

;
; 00 - Reset driver
;
;
; In:	none
; Out:	[AX] = 0/FFFFh				(not installed/installed)
;	[BX] = 2/3/FFFFh			(number of buttons)
; Use:	none
; Modf: none
; Call: enabledriver_20, softreset_21
;
resetdriver_00	proc
		call	enabledriver_20
		;j	softreset_21
resetdriver_00	endp

;
; 21 - Software reset
;
;
; In:	none
; Out:	[AX] = 21h/FFFFh			(not installed/installed)
;	[BX] = 2/3/FFFFh			(number of buttons)
; Use:	shape, maxcoord, shift320, granu
; Modf: *
; Call: @setusershape
;
softreset_21	proc
		db	0B0h			; MOV AL,byte
buttonscnt	db	?			; buttons count (2/3)
		cbw
		mov	[_ARG_BX_],ax
		mov	[_ARG_AX_],0FFFFh

		mov	[cursorhidden],1
		;movSeg	ds,cs
		mov	si,offset shape
		call	@setusershape
		mov	[startscan],77FFh
		mov	[endscan],7700h
		mov	[mickey8.X],8
		mov	[mickey8.Y],16
;;+*		mov	[doublespeed],64
		mov	cx,LenInitArea3/2	; clear area 3
;----------
@setcoords:	mov	ax,[granu.Y]
		dec	ax
		jz	@@setrange		; jump if graphics mode
		xor	ax,ax
		mov	es,ax
		mov	ax,[granu.X]
		mul	word ptr es:[0:44Ah]
		mov	[maxcoord.X],ax
		mov	al,byte ptr es:[0:484h]
		or	al,al
		jnz	@@mulygranu
		mov	al,24
@@mulygranu:	inc	ax			; OPTIMIZE: AX instead AL
		mul	byte ptr [granu.Y]
		mov	[maxcoord.Y],ax

@@setrange:	mov	ax,[maxcoord.Y]		; set lower range
		mov	bx,ax
		dec	ax
		mov	[rangemax.Y],ax
		mov	ax,[maxcoord.X]		; set right range
		dec	ax
		push	cx
		mov	cl,[shift320]
		shl	ax,cl
		pop	cx
		mov	[rangemax.X],ax
		shr	bx,1
		shr	ax,1
		inc	ax
;----------
@setpos:	cli
		mov	[pos.X],ax
		xor	dx,dx
		div	[granu.X]
		mul	[granu.X]
		mov	[granpos.X],ax

		mov	ax,bx
		mov	[pos.Y],ax
		;xor	dx,dx
		div	[granu.Y]
		mul	[granu.Y]
		mov	[granpos.Y],ax

		movSeg	es,cs
		mov	di,offset StartInitArea
		xor	ax,ax
		rep	stosw
		ret
softreset_21	endp

;
; 1F - Disable mouse driver
;
;
; In:	none
; Out:	[AX] = 1Fh/FFFFh			(success/unsuccess)
;	[ES:BX]					(old int33 handler)
; Use:	oldint33, IRQintnum, oldIRQaddr, oldint10
; Modf: AX, DX, ES, disabled?, cursorhidden
; Call: INT 21/25, restorescreen, disableIRQ
;
disabledrv_1F	proc
		les	ax,[oldint33]
		mov	[_ARG_BX_],ax
		mov	[_ARG_ES_],es

		mov	al,1
		cmp	[disabled?],al
		je	@ret
		mov	[disabled?],al
		mov	[cursorhidden],al
		call	disableIRQ
		call	restorescreen
IFNDEF PS2
;---------- restore old IRQ handler
		mov	ax,word ptr [IRQintnum]
		push	ds
		lds	dx,[oldIRQaddr]
		int	21h
		pop	ds
ENDIF
;---------- restore old INT 10 handler
		mov	ax,2510h
		push	ds
		lds	dx,[oldint10]
		int	21h
		pop	ds
;---------- 
@ret:		ret
disabledrv_1F	endp

;
; 20 - Enable mouse driver
;
;
; In:	none
; Out:	[AX] = 20h/FFFFh			(success/unsuccess)
; Use:	IRQintnum
; Modf: AX, BX, DX, ES, disabled?, oldint10, oldIRQaddr
; Call: INT 21/25, setupvideo, enableIRQ
;
enabledriver_20	proc
		cli
		mov	al,0
		cmp	[disabled?],al
		je	@@enabled
		mov	[disabled?],al

;---------- set new INT 10 handler
		mov	ax,3510h
		int	21h
		saveFAR	[oldint10],es,bx

		mov	ax,2510h
		mov	dx,offset int10handler
		int	21h
IFNDEF PS2
;---------- set new IRQ handler
		mov	al,[IRQintnum]
		mov	ah,35h
		int	21h
		saveFAR [oldIRQaddr],es,bx

		mov	ax,word ptr [IRQintnum]
		mov	dx,offset IRQhandler
		int	21h
ENDIF
;----------
@@enabled:	call	setupvideo
		jmp	enableIRQ
enabledriver_20	endp

;
; 01 - Show mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AL, regionchk?, cursorhidden
; Call: showcursor
;
showcursor_01	proc
		mov	al,0
		mov	[regionchk?],al
		cmp	[cursorhidden],al
		je	@ret			; jump if already shown
		dec	[cursorhidden]
		jmp	showcursor
showcursor_01	endp

;
; 02 - Hide mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: cursorhidden
; Call: restorescreen
;
hidecursor_02	proc
		inc	[cursorhidden]
		jnz	@@restore
		dec	[cursorhidden]
@@restore:	jmp	restorescreen
hidecursor_02	endp

;
; 03 - Return position and button status
;
;
; In:	none
; Out:	[BX]					(buttons status)
;	[CX]					(column)
;	[DX]					(row)
; Use:	buttstatus, granpos
; Modf: AX, CX, DX
; Call: none
;
status_03	proc
		mov	ax,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		j	@retBCDX
status_03	endp

;
; 05 - Return button press data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons states)
;	[BX]					(press times)
;	[CX]					(last press column)
;	[DX]					(last press row)
; Use:	buttpress
; Modf: AX
; Call: @retbuttstat
;
butpresdata_05	proc
		mov	ax,offset buttpress
		j	@retbuttstat
butpresdata_05	endp

;
; 06 - Return button release data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons states)
;	[BX]					(release times)
;	[CX]					(last release column)
;	[DX]					(last release row)
; Use:	butrelease, buttstatus
; Modf: AX, BX, CX, DX
; Call: none
;
buttreldata_06	proc
		mov	ax,offset buttrelease
@retbuttstat:	and	bx,7
ERRIF (6 ne SIZE BUTTLASTSTATE) "BUTTLASTSTATE structure size changed!"
		shl	bx,1
		add	ax,bx
		shl	bx,1
		add	bx,ax			; =AX+BX*SIZE BUTTLASTSTATE
		mov	ax,[buttstatus]
		mov	[_ARG_AX_],ax
		xor	ax,ax
		xchg	[bx.counter],ax
		mov	cx,[bx.lastcol]
		mov	dx,[bx.lastrow]
@retBCDX:	mov	[_ARG_CX_],cx
		mov	[_ARG_DX_],dx
@retBX:		mov	[_ARG_BX_],ax
		ret
buttreldata_06	endp

;
; 15 - Return driver storage requirements
;
;
; In:	none
; Out:	[BX]					(buffer size)
; Use:	LenSaveArea
; Modf: AX
; Call: none
;
storagereq_15	proc
		mov	ax,LenSaveArea
		j	@retBX
storagereq_15	endp

;
; 1B - Return mouse sensitivity
;
;
; In:	none
; Out:	[BX]				(number of mickeys per 8 pix
;	[CX]				 horizontally/vertically)
;	[DX]				(threshold speed in mickeys/second)
; Use:	mickey8, /doublespeed/
; Modf: AX, CX, DX
; Call: none
;
getsens_1B	proc
;;+*		mov	dx,[doublespeed]
		mov	dx,64
		mov	ax,[mickey8.X]
		mov	cx,[mickey8.Y]
		j	@retBCDX
getsens_1B	endp

;
; 1E - Return display page number
;
;
; In:	none
; Out:	[BX]					(display page number)
; Use:	0:462h
; Modf: AX, DS
; Call: none
;
getpage_1E	proc
		xor	ax,ax
		mov	ds,ax
		mov	al,ds:[462h]
		j	@retBX
getpage_1E	endp

;
; 07 - Define horizontal cursor range
;
;
; In:	CX					(min column)
;	DX					(max column)
; Out:	none
; Use:	pos
; Modf: CX, DX, rangemin.X, rangemax.X
; Call: setpos_04
;
hrange_07	proc
		cmp	dx,cx
		jb	@@swminmaxx
		xchg	dx,cx
@@swminmaxx:	mov	[rangemin.X],dx
		mov	[rangemax.X],cx
@resetpos:	mov	dx,[pos.Y]
		mov	cx,[pos.X]
		;j	setpos_04
hrange_07	endp

;
; 04 - Position mouse cursor
;
;
; In:	CX					(column)
;	DX					(row)
; Out:	none
; Use:	rangemin, rangemax
; Modf: AX, BX
; Call: cutrangex, cutrangey, @setpos, @show
;
setpos_04	proc
		mov	ax,dx
		call	cutrangey
		mov	bx,ax
		mov	ax,cx
		call	cutrangex
		mov	cx,LenInitArea1/2	; clear area 1
		call	@setpos
		j	@show
setpos_04	endp

;
; 08 - Define vertical cursor range
;
;
; In:	CX					(min row)
;	DX					(max row)
; Out:	none
; Use:	none
; Modf: CX, DX, rangemin.Y, rangemax.Y
; Call: @resetpos
;
vrange_08	proc
		cmp	dx,cx
		jb	@@swminmaxy
		xchg	dx,cx
@@swminmaxy:	mov	[rangemin.Y],dx
		mov	[rangemax.Y],cx
		j	@resetpos
vrange_08	endp

;
; 09 - Define graphics cursor
;
;
; In:	BX					(hot spot column)
;	CX					(hot spot row)
;	ES:DX					(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf: SI, hotspot
; Call: restorescreen, showcursor
;
graphcursor_09	proc
		mov	[hotspot.X],bx
		mov	[hotspot.Y],cx
		movSeg	ds,es
		mov	si,dx

@setusershape:	movSeg	es,cs
		mov	di,offset screenmask
		mov	cx,16+16
		rep	movsw

		movSeg	ds,cs
@redrawscr:	call	restorescreen
@show:		jmp	showcursor
graphcursor_09	endp

;
; 0A - Define text cursor
;
;
; In:	BX					(0=SW, else HW text cursor)
;	CX					(screen mask/start scanline)
;	DX					(cursor mask/end scanline)
; Out:	none
; Use:	none
; Modf: AX, CX, DX, cursortype, startscan, endscan
; Call: INT 10/01, @show
;
textcursor_0A	proc
		mov	[startscan],cx
		mov	[endscan],dx
		mov	dh,cl
		push	bx dx
		call	restorescreen
		pop	cx ax
		mov	[cursortype],al
		or	al,al
		jz	@show			; exit if software cursor
		mov	ah,1
		int	10h			; set cursor shape & size
		ret
textcursor_0A	endp

;
; 0D - Light pen emulation On
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: /nolightpen?/
; Call: lightpenoff_0E
;
;;+*lightpenon_0D	proc
;;+*		mov	[nolightpen?],1
;;+*		;j	lightpenoff_0E
;;+*lightpenon_0D	endp

;
; 0E - Light pen emulation Off
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: /nolightpen?/
; Call: none
;
;;+*lightpenoff_0E	proc
;;+*		dec	[nolightpen?]
;;+*		ret
;;+*lightpenoff_0E	endp

;
; 0F - Define Mickey/Pixel ratio
;
;
; In:	CX					(number of mickeys per 8 pix
;	DX					 horizontally/vertically)
; Out:	none
; Use:	none
; Modf: mickey8
; Call: none
;
micperpixel_0F	proc
		mov	[mickey8.X],cx
		mov	[mickey8.Y],dx
		ret
micperpixel_0F	endp

;
; 10 - Define screen region for updating
;
;
; In:	CX, DX					(X/Y of upper left corner)
;	SI, DI					(X/Y of lower right corner)
; Out:	none
; Use:	none
; Modf: CX, DX, SI, DI, upleft, lowright, regionchk?
; Call: @redrawscr
;
defregion_10	proc
		mov	si,[_ARG_SI_]
		cmp	cx,si
		jbe	@@nosw1
		xchg	cx,si
@@nosw1:	cmp	dx,di
		jbe	@@nosw2
		xchg	dx,di
@@nosw2:	mov	[upleft.X],cx
		mov	[lowright.X],si
		mov	[upleft.Y],dx
		mov	[lowright.Y],di
		mov	[regionchk?],1
		j	@redrawscr
defregion_10	endp

;
; 14 - Exchange interrupt subroutines
;
;
; In:	CX					(new call mask)
;	ES:DX					(new FAR routine)
; Out:	[CX]					(old call mask)
;	[ES:DX]					(old FAR routine)
; Use:	callmask, userproc
; Modf: AX
; Call: intpar_0C
;
exchangeint_14	proc
		mov	ax,[callmask]
		mov	[_ARG_CX_],ax
		mov	ax,word ptr userproc[0]
		mov	[_ARG_DX_],ax
		mov	ax,word ptr userproc[2]
		mov	[_ARG_ES_],ax
		;j	intpar_0C
exchangeint_14	endp

;
; 0C - Define interrupt subroutine parameters
;
;
; In:	CX					(call mask)
;	ES:DX					(FAR routine)
; Out:	none
; Use:	none
; Modf: userproc, callmask
; Call: none
;
intpar_0C	proc
		saveFAR [userproc],es,dx
		mov	[callmask],cx
		ret
intpar_0C	endp

;
; 16 - Save driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(buffer)
; Out:	none
; Use:	StartSaveArea
; Modf: CX, SI, DI
; Call: none
;
savestate_16	proc
		mov	di,dx
		mov	si,offset StartSaveArea
@saveareamove:	cmp	bx,LenSaveArea
		jb	@@saveret
		mov	cx,LenSaveArea/2
		rep	movsw
@@saveret:	ret
savestate_16	endp

;
; 17 - Restore driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(saved state buffer)
; Out:	none
; Use:	none
; Modf: SI, DI, DS, ES, StartSaveArea
; Call: @saveareamove
;
restorestate_17	proc
		push	ds es
		pop	ds es
		mov	si,dx
		mov	di,offset StartSaveArea
		j	@saveareamove
restorestate_17	endp

;
; 1A - Set mouse sensitivity
;
;
; In:	BX				(number of mickeys per 8 pix
;	CX				 horizontally/vertically)
;	DX				(threshold speed in mickeys/second)
; Out:	none
; Use:	none
; Modf: mickey8
; Call: doublespeed_13
;
setsens_1A	proc
		mov	[mickey8.X],bx
		mov	[mickey8.Y],cx
		;j	doublespeed_13
setsens_1A	endp

;
; 13 - Define double speed threshold
;
;
; In:	DX				(threshold speed in mickeys/second)
; Out:	none
; Use:	none
; Modf: /DX/, /doublespeed/
; Call: none
;
doublespeed_13	proc
;;+*		or	dx,dx
;;+*		jnz	@@savespeed
;;+*		mov	dl,64
;;+*@@setspeed:	mov	[doublespeed],dx
		;ret
doublespeed_13	endp

;
; 11 12 18 19 1C - Null function for not implemented calls
;

nullfunc	proc
		ret
nullfunc	endp

;
;				INT 33 handler
;

funcsoffsets	dw	offset resetdriver_00
		dw	offset showcursor_01
		dw	offset hidecursor_02
		dw	offset status_03
		dw	offset setpos_04
		dw	offset butpresdata_05
		dw	offset buttreldata_06
		dw	offset hrange_07
		dw	offset vrange_08
		dw	offset graphcursor_09
		dw	offset textcursor_0A
		dw	offset getmcounter_0B
		dw	offset intpar_0C
		dw	offset nullfunc		;lightpenon_0D
		dw	offset nullfunc		;lightpenoff_0E
		dw	offset micperpixel_0F
		dw	offset defregion_10
		dw	offset nullfunc		;11 - genius driver only
		dw	offset nullfunc		;12 - large graphics cursor
		dw	offset doublespeed_13
		dw	offset exchangeint_14
		dw	offset storagereq_15
		dw	offset savestate_16
		dw	offset restorestate_17
		dw	offset nullfunc		;18 - set alternate handler
		dw	offset nullfunc		;19 - return alternate address
		dw	offset setsens_1A
		dw	offset getsens_1B
		dw	offset nullfunc		;1C - InPort mouse only
		dw	offset nullfunc		;1D - define display page #
		dw	offset getpage_1E
		dw	offset disabledrv_1F
		dw	offset enabledriver_20
		dw	offset softreset_21

handler33	proc	far
		sti
		or	ah,ah
		jnz	@@iret

;
; 32 - Get active advanced functions
; Out:	[AX]					(active functions flag)
;	[BX] = 0
;	[CX] = 0E000h
;	[DX] = 0
;
getactive_32:	cmp	al,32h
		jne	no_getactive_32
		mov	ax,0110010000000100b	; active: 26 27 2A 32
		mov	cx,0E000h
		xor	dx,dx
@@iretBX0:	xor	bx,bx
@@iret:		iret
no_getactive_32:

;
; 23 - Get language for messages
; Out:	[BX]					(language code: 0 - English)
;
getlanguage_23:	cmp	al,23h
		je	@@iretBX0

		cld
		push	ds
		movSeg	ds,cs

;
; 24 - Get software version, mouse type and IRQ
; Out:	[AX] = 24h/FFFFh			(installed/error)
;	[BX]					(version)
;	[CH] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
;	[CL]					(interrupt #/0=PS/2)
; Use:	driverversion
;
getversion_24:	cmp	al,24h
		jne	no_getversion_24
		mov	bx,driverversion
		db	0B9h			; MOV CX,word
IFNDEF PS2
mouseinfo	db	0,2
ELSE
mouseinfo	db	0,4
ENDIF
no_getversion_24:

;
; 26 - Get maximum virtual coordinates
; Out:	[BX]					(mouse disabled flag)
;	[CX]					(maximum virtual X)
;	[DX]					(maximum virtual Y)
; Use:	disabled?, maxcoord
;
getmaxvirt_26:	cmp	al,26h
		jne	no_getmaxvirt_26
		xor	bx,bx
		mov	bl,[disabled?]
		mov	cx,[maxcoord.X]
		dec	cx
		mov	dx,[maxcoord.Y]
		dec	dx
no_getmaxvirt_26:

;
; 2A - Get cursor hot spot
; Out:	[AX]					(cursor visibility counter)
;	[BX]					(hot spot column)
;	[CX]					(hot spot row)
;	[DX] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
; Use:	cursorhidden, hotspot, mouseinfo
;
gethotspot_2A:	cmp	al,2Ah
		jne	no_gethotspot_2A
		mov	al,[cursorhidden]
		cbw
		mov	bx,[hotspot.X]
		mov	cx,[hotspot.Y]
		xor	dx,dx
		mov	dl,mouseinfo[1]
		j	@iret
no_gethotspot_2A:

;
; 4D - Return pointer to copyright string
; Out:	[ES:DI]					(copyright string)
; Use:	IDstring
;
getcopyrght_4D:	cmp	al,4Dh
		jne	no_getcopyrght_4D
		movSeg	es,cs
		mov	di,offset IDstring
no_getcopyrght_4D:

;
; 6D - Return pointer to version string
; Out:	[ES:DI]					(version string)
; Use:	msversion
;
getversion_6D:	cmp	al,6Dh
		jne	no_getversion_6D
		movSeg	es,cs
		mov	di,offset msversion
no_getversion_6D:

		cmp	al,0Bh
		je	getmcounter_0B

;
; 27 - Get screen/cursor masks and mickey counts
; Out:	[AX]			(screen mask/start scanline)
;	[BX]			(cursor mask/end scanline)
;	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Call: getmcounter_0B
;
getscreen_27:	cmp	al,27h
		jne	@@calltable
		mov	ax,[startscan]
		mov	bx,[endscan]
		;j	getmcounter_0B

;
; 0B - Get motion counters
; Out:	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Modf: mickey
;
getmcounter_0B:	xor	cx,cx
		xchg	[mickey.X],cx
		xor	dx,dx
		xchg	[mickey.Y],dx
		j	@iret

;----------
@@calltable:	cmp	al,21h
		ja	@iret
		push	ax bx cx dx si di bp es
		mov	bp,sp
		shl	ax,1
		mov	si,ax
		call	funcsoffsets[si]	; call by calculated offset
@rethandler:	pop	es bp di si dx cx bx ax
@iret:		pop	ds
		iret
handler33	endp

; END OF INT 33 SERVICES 


		evendata
EndTSRpart	label


; INITIALIZATION PART DATA 

paragraphs	dw	0
SaveMemStrat	dw	0
SaveUMBLink	dw	0
XMSentry	dd	0

mousetype	db	?			; 1=MS, 2=LT, 3=MSys
options		db	0
OPT_PS2		equ	00000001b
OPT_SERIAL	equ	00000010b
OPT_COMforced	equ	00000100b
OPT_PS2after	equ	00001000b
OPT_3button	equ	00010000b
OPT_NOMSYS	equ	00100000b
OPT_LEFTHAND	equ	01000000b
OPT_NOUMB	equ	10000000b

Copyright	db	'CuteMouse v1.73 [FreeDOS]',0dh,0ah,'$'
S_already	db	'Mouse driver already loaded',0dh,0ah
S_info		db	'Installed at $'
IFNDEF PS2
S_atCOM		db	'COM '
com_port	db	0,' for $'
S_drvMS		db	'Microsoft mouse',0dh,0ah,'$'
S_drvMouse	db	'Mouse Systems mouse',0dh,0ah,'$'
S_drvLogi	db	'Logitech MouseMan',0dh,0ah,'$'
E_ps2not	db	'Error: PS/2 not supported',0dh,0ah,'$'
ELSE
S_drvPS2	db	'PS/2 port',0dh,0ah,'$'
ENDIF

S_unloaded	db	'Driver successfully unloaded...',0dh,0ah,'$'
S_notinst	db	'Driver is not installed!',0dh,0ah,'$'
E_option	db	0dh,0ah,'Error: Invalid option',0dh,0ah
		db	'Enter /? on command line for help',0dh,0ah,'$'
E_find		db	'Error: device not found',0dh,0ah,'$'

Syntax		db	'Options:',0dh,0ah
IFNDEF PS2
;;+		db	'  /P	   - force PS/2 mouse mode',0dh,0ah
		db	'  /S[c[i]] - force serial mouse mode at COM port c (c=1-4, default is COM1)',0dh,0ah
		db	'	     with IRQ line i (i=2-7, default is IRQ4 for COM1/3 and IRQ3',0dh,0ah
		db	'	     for COM2/4)',0dh,0ah
		db	'  /Y	   - prevent Mouse Systems mode',0dh,0ah
;;+		db	'  /V	   - reverse search: find PS/2 after serial mouse',0dh,0ah
ENDIF
		db	'  /3	   - force 3-buttons mode (for Microsoft and PS/2 mouses only)',0dh,0ah
		db	'  /R[0]    - auto hardware resolution',0dh,0ah
		db	'  /Rh[v]   - horizontal/vertical resolution (h,v=1-9, default is R33)',0dh,0ah
		db	'  /L	   - left hand mode (default is right hand mode)',0dh,0ah
		db	'  /W	   - prevent loading TSR into UMB',0dh,0ah
		db	'  /U	   - uninstall driver',0dh,0ah
		db	'  /?	   - show this help',0dh,0ah,'$'


; Real Start 

@@devnotfound:	mov	dx,offset E_find	; 'Error: device not found'
@@restoredrv:	mov	ax,3533h
		int	21h
		mov	ax,es
		or	ax,bx
		jz	@@exiterrmsg
		mov	ax,20h
		int	33h			; reenable driver
@@exiterrmsg:	jmp	EXITERRMSG

real_start:	cld
		PrintS	Copyright		; 'Cute Mouse Driver'
		call	commandline		; examine command line
IFNDEF PS2
		;mov	mouseinfo[1],2
		call	checkCOMsloop
		jnc	@@mousefound

		mov	bh,3
		PS2serv 0C205h,@@devnotfound	; initalize PS/2 pointing dev
		mov	dx,offset E_ps2not	; 'Error: PS/2 not supported'
		j	@@restoredrv
ELSE
		;mov	word ptr [mouseinfo],400h
		call	checkPS2
		jc	@@devnotfound
ENDIF
@@mousefound:
;---------- save old INT 33h handler
		mov	ax,3533h
		int	21h
		mov	ax,es
		saveFAR [oldint33],ax,bx

;---------- check if CuteMouse driver already installed
		or	ax,bx
		jz	@@nodriver
		push	es bx
		mov	ax,1Fh
		int	33h			; disable driver
		pop	bx es
		call	checkCuteMouse
		mov	dx,offset S_already	; 'Mouse driver already loaded'
		je	@@setupdriver
@@nodriver:
;---------- detect VGA card
		mov	ax,1A00h
		int	10h			; get display type in bx
		cmp	al,1Ah
		jne	@@noVGA
		cmp	bl,7			; monochrome VGA?
		je	@@setATCVGA
		cmp	bl,8			; color VGA?
		jne	@@noVGA
@@setATCVGA:	inc	@videoregs[(SIZE RGROUPDEF)*3].regscnt
@@noVGA:
;---------- copy TSR driver to UMB, if possible, and set INT 33 handler
		call	prepareTSR		; new memory segment in ES
		push	ds
		movSeg	ds,es
		mov	dx,offset handler33
		mov	ax,2533h
		int	21h			; setup new 33 handler
		pop	ds
		mov	dx,offset S_info	; 'Installed at'
;----------
@@setupdriver:	PrintS
IFNDEF PS2
		PrintS	S_atCOM
		mov	dx,offset S_drvMS	; 'Microsoft mouse'
		cmp	[mousetype],2
		jb	@@printmode
		mov	dx,offset S_drvLogi	; 'Logitech MouseMan'
		je	@@printmode
		mov	dx,offset S_drvMouse	; 'Mouse Systems mouse'
ELSE
		mov	dx,offset S_drvPS2	; 'PS/2 port'
ENDIF
@@printmode:	PrintS

		call	setupdriver
		xor	ax,ax
		int	33h			; Reset driver

		cmp	[memtype],2
		je	@@tsrexit
		jmp	EXIT0			; jump if TSR copied to UMB

;---------- TSR exit if no other UMB memory grabbed
@@tsrexit:	mov	es,ds:[2Ch]		; release environment
		mov	ah,49h
		int	21h

		mov	dx,offset EndTSRpart+15
		mov	cl,4
		shr	dx,cl
		mov	ax,3100h
		int	21h			; TSR exit

;
;		Setup resident driver code and parameters
;

setupdriver	proc
;---------- setup left hand mode handling
IFNDEF PS2
		mov	al,0EBh			; JMP SHORT
		test	[options],OPT_LEFTHAND
		jnz	@@setlhand
ELSE
		mov	al,0EBh			; JMP SHORT
		test	[options],OPT_LEFTHAND
		jz	@@setlhand
ENDIF
		mov	al,07Ah			; JPE
@@setlhand:	mov	es:[LEFTHANDJUMP],al
;---------- setup buttons count and mask
		mov	ax,703h
		test	[options],OPT_3button
		jnz	@@setbuttons
IFNDEF PS2
		cmp	[mousetype],1
		jne	@@setbuttons		; jump if no MS mode
ENDIF
		mov	ax,302h
@@setbuttons:	mov	es:[buttonscnt],al
IFDEF PS2
		mov	es:[buttonsmask],ah
ENDIF
;---------- setup serial mouse handler code
IFNDEF PS2
		mov	bx,offset MSMproc-offset mouseproc-2
		cmp	[mousetype],2
		ja	@@setmousep		; jump if Mouse Systems

		mov	bx,offset MSLGproc-offset mouseproc-2
		mov	cx,9090h		; NOP/NOP
		mov	dx,175h			; DL=JNE, DH=0 for MS3
		je	@@setMSLGproc		; jump if Logitech

		mov	cx,0EBh+(offset @@MSLGprocret-offset MSLGJUMP3-2) SHL 8
						; JMP SHORT @@MSLGprocret
		cmp	al,2
		je	@@setMSLGproc		; jump if MS with 2 buttons
		mov	dx,0EBh			; DL=JMP SHORT

@@setMSLGproc:	mov	es:[MSLGJUMP1],dl
		mov	es:[MSLGJUMP2],dh
		mov	es:[MSLGJUMP3],cx
@@setmousep:	mov	es:[mouseproc],bx
ENDIF
;---------- copy (if required) other parameters
		mov	al,[mresolution0]
		mov	es:[mresolution0],al
		mov	al,[mresolution1]
		mov	es:[mresolution1],al
		mov	ax,word ptr [mouseinfo]
		mov	word ptr es:[mouseinfo],ax
IFNDEF PS2
		mov	al,[COMLCR]
		mov	es:[COMLCR],al
		mov	ax,[IO_address]
		mov	es:[IO_address],ax
		mov	al,[PICstate]
		mov	es:[PICstate],al
		mov	al,[IRQintnum]
		mov	es:[IRQintnum],al
ENDIF
		ret
setupdriver	endp

IFNDEF PS2
;
;		Check given or all COM-ports for mouse connection
;

checkCOMsloop	proc
		mov	[COMLCR],2
		mov	cx,1
		push	cx
		test	[options],OPT_COMforced
		jnz	@@searchloop
		pop	cx
		mov	cl,4

@@setCOMloop:	mov	al,'5'
		sub	al,cl
		push	cx
		call	setCOMport
@@searchloop:	call	COMexist?
		jc	@@nextsearch
		call	detectmouse
@@nextsearch:	pop	cx
		jnc	@@searchret
		loop	@@setCOMloop

;---------- set Mouse Systems mouse mode
		test	[options],OPT_NOMSYS
		stc
		jnz	@@searchret
		test	[options],OPT_COMforced
		jnz	@@setMSYS
		mov	al,'1'
		call	setCOMport

@@setMSYS:	mov	al,3
		mov	[COMLCR],al
		mov	[mousetype],al
		call	COMexist?

@@searchret:	ret
checkCOMsloop	endp

;
;			Check if COM port available
;
;
; In:	none
; Out:	Carry flag				(COM port not exist)
; Use:	IO_address
; Modf: AX, CX, DX
; Call: disableIRQ
;
COMexist?	proc
		cli
		mov	cx,[IO_address]
		stc
		jcxz	@@comexret
		push	cx
		call	disableIRQ
		pop	dx
		add	dx,3
		mov	al,0
		out	dx,al			; {3FBh} reset comm params
		inc	dx
		in	al,dx			; {3FCh} modem ctrl reg
		and	al,0E0h			; get reserved bits
		mov	ah,al
		sub	dx,3
		in	al,dx			; {3F9h} int enable reg
		and	al,0F0h			; get reserved bits
		or	al,ah			; AL must be 0 if port exists
		neg	al			; nonzero makes carry flag
@@comexret:	sti
		ret
COMexist?	endp

;
;			Check if mouse available
;
;
; In:	none
; Out:	Carry flag				(no mouse found)
; Use:	IO_address, COMLCR
; Modf: AX, BX, CX, DX, SI, mousetype
; Call: none
;
detectmouse	proc
		mov	si,[IO_address]

;---------- mouse hardware reset
		mov	dx,si
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		mov	dx,si
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		add	dx,3
		mov	al,[COMLCR]
		out	dx,al			; {3FBh} DLAB off, no parity,
						;  stop=1, length=7/8
;---------- pulse (drop and raise) RTS signal
		xor	ax,ax
		mov	es,ax
		inc	dx
		mov	di,es:[46Ch]
@@wait1:	cmp	di,es:[46Ch]
		je	@@wait1			; wait timer tick start
		out	dx,al			; {3FCh} DTR/RTS/OUT2 off

		mov	al,3
		mov	di,es:[46Ch]
@@wait2:	cmp	di,es:[46Ch]
		je	@@wait2			; wait timer tick end
		out	dx,al			; {3FCh} DTR/RTS on, OUT2 off

;---------- detect if Microsoft or Logitech mouse present
		mov	bx,101h			; bl=no MS, bh=mousetype
		mov	cx,4			; scan 4 first bytes
@@detectloop:	call	readCOMbyte
		jc	@@detdone
		cmp	al,'M'
		jne	@@checkLG
		mov	bl,0			; Microsoft mouse found...
@@checkLG:	cmp	al,'3'
		jne	@@detectnext
		mov	bh,2			; ...Logitech mouse found
@@detectnext:	loop	@@detectloop

@@skipdata:	call	readCOMbyte
		jnc	@@skipdata

@@detdone:	mov	[mousetype],bh		; save 1=MS or 2=Logitech
		neg	bl			; nonzero makes carry flag
		ret
detectmouse	endp

;

readCOMbyte	proc
		mov	ah,2			; length of silence in ticks
		mov	dx,si
		add	dx,5
@@timeloop:	mov	di,es:[46Ch]
@@waitloop:	in	al,dx			; {3FDh} line status reg
		test	al,1
		jnz	@@readret		; jump if data ready
		cmp	di,es:[46Ch]
		je	@@waitloop		; jump if same tick
		dec	ah
		jnz	@@timeloop		; wait next tick of silence
		stc
		ret
@@readret:	mov	dx,si
		in	al,dx			; {3F8h} receive byte
		;clc
		ret
readCOMbyte	endp

;
;				Set Mouse Port
;
;
; In:	AL					(COM port letter)
; Out:	none
; Use:	none
; Modf: AL, com_port, IO_address
; Call: setIRQ
;
setCOMport	proc
		mov	[com_port],al
		sub	al,'1'

		xor	bx,bx
		mov	es,bx
		mov	bl,al
		shl	bx,1
		mov	bx,es:400h[bx]
		mov	[IO_address],bx

		shr	al,1
		mov	al,4			; IRQ4 for COM1/3
		jnc	setIRQ
		dec	ax			; OPTIMIZE: AX instead AL
						; IRQ3 for COM2/4
		;j	setIRQ
setCOMport	endp

;
;				Set IRQ number
;
;
; In:	AL					(IRQ number)
; Out:	none
; Use:	none
; Modf: AL, CL, PICstate, IRQintnum
; Call: none
;
setIRQ		proc
		mov	mouseinfo[0],al
		mov	cl,al
		add	al,8
		mov	[IRQintnum],al
		mov	al,1
		shl	al,cl
		mov	[PICstate],al		; PIC interrupt enabler
		ret
setIRQ		endp

ELSE
;
;				Check for PS/2
;
;
; In:	none
; Out:	Carry flag
; Use:	none
; Modf: AX, BX, CX
; Call: none
;
checkPS2	proc
		mov	bh,3
		PS2serv 0C205h,@@PSerror	; initialize mouse, bh=datasize
		movSeg	es,cs
		mov	bx,offset PS2handler
		PS2serv	0C207h,@@PSerror	; mouse, es:bx=ptr to handler
		xor	bx,bx
		mov	es,bx
		PS2serv	0C207h			; mouse, es:bx=ptr to handler
		mov	bh,3
		PS2serv 0C203h,@@PSerror	; set mouse resolution bh
		mov	bh,1
		PS2serv 0C200h,@@PSerror	; set mouse on
		;clc
		ret
@@PSerror:	stc
		ret
checkPS2	endp
ENDIF

;
;		Check if CuteMouse driver is installed
;
;
; In:	ES:BX					(driver address)
; Out:	Zero flag				(zero if installed)
;	ES					(driver segment)
; Use:	none
; Modf: AX, CX, SI, DI
; Call: none
;
checkCuteMouse	proc
		cmp	bx,offset handler33
		jne	@ret2
		mov	si,offset IDstring
		mov	di,si
		mov	cx,IDstringlen
		rep	cmpsb
@ret2:		ret
checkCuteMouse	endp

;
;			Examine Command Line
;

commandline	proc
		mov	si,80h
		lodsb
		mov	ah,0
		mov	di,ax
		add	di,si

@@setoption:	or	[options],ah
newopt:		cmp	si,di
		jae	@ret2
		lodsb
@@newopt2:	cmp	al,' '
		jbe	newopt
		cmp	al,'/'			; option character?
		jne	@badoption
		cmp	si,di			; option itself there?
		jae	@badoption
		lodsb
		cmp	al,'?'			; '/?' -> print help and exit
		je	@helpexit
		mov	ah,OPT_3button
		cmp	al,'3'			; '/3' -> force 3-buttons mode
		je	@@setoption

		and	al,not 20h
;;+		mov	ah,OPT_PS2
;;+		cmp	al,'P'			; '/P' -> force PS/2 mouse
;;+		je	@@setoption
IFNDEF PS2
		cmp	al,'S'
		jne	@@noserial
		or	[options],OPT_SERIAL	; '/S' -> force serial mouse
		cmp	si,di
		jae	@ret2
		lodsb
		cmp	al,'1'
		jb	@@newopt2
		cmp	al,'4'
		ja	@@newopt2
		or	[options],OPT_COMforced
		call	setCOMport		; '/Sc' -> set COM port
		cmp	si,di
		jae	@ret2
		lodsb
		cmp	al,'2'
		jb	@@newopt2
		cmp	al,'7'
		ja	@@newopt2
		sub	al,'0'
		call	setIRQ			; '/Sci' -> set IRQ line
		j	newopt
@@noserial:
ENDIF
		mov	ah,OPT_NOMSYS
		cmp	al,'Y'			; '/Y' -> prohibit Mouse Systems mode
		je	@@setoption
;;+		mov	ah,OPT_PS2after
;;+		cmp	al,'V'			; '/V' -> reverse search
;;+		je	@@setoption
		mov	ah,OPT_LEFTHAND
		cmp	al,'L'			; '/L' -> set left hand mode
		je	@@setoption
		cmp	al,'R'			; '/R' -> set resolution
		je	_resolution
		mov	ah,OPT_NOUMB
		cmp	al,'W'			; '/W' -> don't touch UMB
		je	@@setoption
		cmp	al,'U'			; '/U' -> unload drv and exit
		je	_unload

@badoption:	mov	dx,offset E_option	; 'Invalid parameter'
EXITERRMSG:	PrintS
		mov	ax,4CFFh
		int	21h			; terminate, al=return code

@helpexit:	mov	dx,offset Syntax	; 'Options:'
EXIT0MSG:	PrintS
EXIT0:		mov	ax,4C00h
		int	21h			; terminate, al=return code
commandline	endp

;
;			Set Resolution option
;

_resolution	proc
		mov	ah,'0'
		cmp	si,di
		jae	@@dupres
		lodsb				; first argument
		cmp	al,'0'
		je	@@setres
		jb	@@fixupres
		cmp	al,'9'
		ja	@@fixupres

		mov	ah,al
		cmp	si,di
		jae	@@setres
		lodsb				; second argument
		cmp	al,'1'
		jb	@@fixupres
		cmp	al,'9'
		jbe	@@setres

@@fixupres:	dec	si			; fixup for command line ptr
@@dupres:	mov	al,ah			; drop invalid argument

@@setres:	sub	ax,'00'
		mov	[mresolution0],ah
		mov	[mresolution1],al
		jmp	newopt
_resolution	endp

;
;			Release driver and quit
;

_unload		proc
		mov	ax,3533h
		int	21h			; get INT 33 handler address
		call	checkCuteMouse		; check if this is CTMOUSE
		mov	dx,offset S_notinst	; 'Driver is not installed!'
		jne	EXITERRMSG

		push	ds es
		mov	ax,1Fh
		int	33h			; disable CuteMouse driver
		mov	cx,es
		mov	ds,cx
		mov	dx,bx
		or	cx,dx			; check if old driver exist
		mov	ax,2533h
		int	21h			; restore old int33 handler
		pop	es ds
		call	FreeMem
		mov	dx,offset S_unloaded	; 'Driver successfully unloaded...'
		jcxz	EXIT0MSG
		mov	ax,20h
		int	33h			; enable old driver
		j	EXIT0MSG
_unload		endp


; MEMORY HANDLING 

prepareTSR	proc
		mov	ax,offset EndTSRpart	; get number of bytes
		call	AllocUMB		;  we need memory for
		mov	ax,es
		mov	bx,cs
		cmp	ax,bx
		je	@@prepareret		; exit if TSR "in place"

		dec	ax
		mov	es,ax			; target MCB segment...
		inc	ax
		mov	es:[1],ax		; ...modify owner field
		dec	bx
		mov	ds,bx			; current MCB
		mov	si,8
		mov	di,si
		movsw				; copy process name
		movsw
		movsw
		movsw
		movSeg	ds,cs

		mov	bx,ax
		add	bx,es:[3]		; add target MCB size field
		mov	es,ax			; target PSP
		mov	dx,ax
		mov	ah,26h
		int	21h			; create PSP
		mov	es:[2],bx		; fix upper segment number

		mov	si,offset StartTSRpart	; relocate initialized
		mov	di,si			;  resident portion
		mov	cx,(EndTSRpart-StartTSRpart)/2
		rep	movsw

@@prepareret:	ret
prepareTSR	endp

;
; Get "XMS" handler address (2 words)
;
; In:	none
; Out:	Zero flag
; Use:	none
; Modf: AX, BX, XMSentry
; Call: INT 2F/4310
;
getXMSaddr	proc
		push	es
		xor	bx,bx
		mov	es,bx
		mov	ax,4310h		; XMS: Get Driver Address
		int	2Fh
		mov	ax,es
		saveFAR [XMSentry],ax,bx
		or	ax,bx			; ZF indicates error: JZ error
		pop	es
		ret
getXMSaddr	endp

;
; Save Allocation Srategy
;
; In:	none
; Out:	Carry flag				(no UMB link supported)
; Use:	none
; Modf: AX, SaveMemStrat, SaveUMBLink
; Call: INT 21/5800, INT 21/5802
;
SaveStrategy	proc
		mov	ax,5800h
		int	21h			; get DOS memory alloc strategy
		mov	[SaveMemStrat],ax
		mov	ax,5802h
		int	21h			; get UMB link state
		mov	byte ptr SaveUMBLink[0],al
		ret
SaveStrategy	endp

;
; Restore allocation strategy
;
; In:	none
; Out:	none
; Use:	SaveMemStrat, SaveUMBLink
; Modf: AX, BX
; Call: INT 21/5801, INT 21/5803
;
RestoreStrategy	proc
		mov	bx,[SaveMemStrat]
		mov	ax,5801h
		int	21h			; set DOS memory alloc strategy
		mov	bx,[SaveUMBLink]
		mov	ax,5803h
		int	21h			; set UMB link state
		ret
RestoreStrategy	endp

;
; Allocate high memory
;
; In:	AX					(memory size required)
; Out:	ES					(grabbed memory segment)
; Use:	XMSentry
; Modf:	AX, BX, CX, DX, memtype, paragraphs
; Call:	INT 21/30, INT 21/48, INT 21/49, INT 21/5801, INT 21/5803,
;	INT 2F/4300, SaveStrategy, RestoreStrategy, getXMSaddr
;
AllocUMB	proc
		add	ax,15
		mov	cl,4
		shr	ax,cl
		mov	[paragraphs],ax

		test	[options],OPT_NOUMB
		jnz	@@allocasis		; jump if UMB prohibited
		mov	ax,cs
		cmp	ah,0A0h
		jae	@@allocasis		; jump if already loaded hi

;---------- check if UMB is DOS type
		;mov	[memtype],0		; 0=DOS UMB
		call	SaveStrategy
		mov	bx,1			; add UMB to MCB chain
		mov	ax,5803h
		int	21h			; set UMB link state

; try to set a best strategy to allocate DOS supported UMBs
		mov	bx,41h			; hi mem, best fit
		mov	ax,5801h
		int	21h			; set alloc strategy
		jnc	@@allocDOSUMB

; try a worse one then
		mov	bx,81h			; hi mem then low mem, best fit
		mov	ax,5801h
		int	21h			; set alloc strategy

@@allocDOSUMB:	mov	bx,[paragraphs]
		mov	ah,48h
		int	21h			; allocate UMB memory
		pushf
		push	ax
		call	RestoreStrategy		; restore allocation strategy
		pop	ax
		popf
		jc	@@allocXMSUMB
		mov	es,ax
		cmp	ah,0A0h			; jump if allocated mem is
		jae	@@allocret		;  is above 640k,
		mov	ah,49h			;  else free it
		int	21h

;---------- try XMS driver to allocate UMB
@@allocXMSUMB:	inc	[memtype]		; 1=XMS UMB
		call	getXMSaddr
		jz	@@allocasis
		mov	ah,10h			; XMS: Request UMB
		mov	dx,[paragraphs]
		call	[XMSentry]
		or	ax,ax
		jz	@@allocasis
		mov	es,bx			; memory segment
		cmp	dx,[paragraphs]
		jae	@@allocret
@freeXMSUMB:	mov	dx,es
		mov	ah,11h			; XMS: Release UMB
		call	[XMSentry]

;---------- use current memory segment
@@allocasis:	mov	[memtype],2		; 2=remain "as is"
		movSeg	es,cs

@@allocret:	ret
AllocUMB	endp

;
;
; In:	ES					(segment to free)
; Out:	none
; Use:	memtype
; Modf:	AX
; Call:	INT 21/49, getXMSaddr
;
FreeMem		proc
		cmp	es:[memtype],1
		jne	@@freeDOS
		call	getXMSaddr
		jnz	@freeXMSUMB

@@freeDOS:	mov	ah,49h
		int	21h			; free allocated memory
		ret
FreeMem		endp

;

		end	start
