/***
*diskio.c - drive initialization/read-write module
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Handles low-level DOS Sector reads/writes.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
*/

#ifndef GENERAL_H
#include "general.h"
#endif

#ifndef DOSIO_H
#include "dosio.h"   /* DOS functions and error codes */
#endif

#ifndef INT13_H
#include "int13.h"
#endif

/* diskio() command values */

typedef enum diskio_commands {
   DISK_INIT  = 0x001,        /* initialize all globals */
   DISK_SHOW  = 0x002,        /* display selected sector */
   DISK_SET   = 0x004,        /* set */
   DISK_INC   = 0x008,        /* increment */
   DISK_READ  = 0x010,        /* read */
   DISK_WRITE = 0x020,        /* write */
   DISK_SAVE  = 0x040,        /* store selected sector */
   DISK_REST  = 0x080         /* restore sector */
} DISKIO_COMMAND;

typedef enum diskio_status {
                                    /* non-fatal errors */
   DISK_USE_DPB = -3,               /*  boot sector bad, using DPB */
   DISK_USE_BOOT,                   /*  DPB bad, using boot sector */

   DISK_OK = -1,                    /* no error */

                                    /* fatal errors */
   DISK_BOOT_ERR,                   /*  boot sector bad and no DPB */
   DISK_REMOTE,                     /*  can not log remote drives */
   DISK_BOUNDS,                     /*  sector out out bounds */
   DISK_MEMORY,                     /*  memory allocation error */
   DISK_DOS_ERR,                    /*  DOS function failure */
   DISK_READ_ERR,                   /*  read/write sector error */
   DISK_FUNC_ERR
#ifdef __GNUC__
   ,DISK_SECSIZE
#endif
} DISKIO_STATUS;

typedef enum diskio_functions {
   DISK_ABS,                        /* disk io types */
   DISK_INT13,
   DISK_IOCTL,
   DISK_LOG,                        /* function flow area */
   DISK_DOS
} DISK_FUNC;

/* function prototypes */

extern int diskio(DISKIO_COMMAND command, long arg, BYTE *buffer);
extern UINT16 sectortocluster(UINT32 sector);
extern UINT32 logicalsector(UINT16 track, UINT16 sector, UINT16 head);
extern void physicalsector(UINT32 sec, UINT16 *t, UINT16 *s, UINT16 *h);
extern UINT32 clustertosector(UINT16 cluster);
extern UINT16 clustersector(UINT32 sector);
extern UINT32 ncluster(int);
extern void dumpsector(void);
extern int exterror(void);

extern int diskiochk(void);
extern UINT32 chksum(BYTE *buf, UINT16 siz);

#define logdisk(d)      (diskio(DISK_INIT,(long)d,sec_buf))
#define movesector(n)   (diskio(DISK_INC,(long)n,sec_buf))
#define setsector(n)    (diskio(DISK_SET,(long)n,sec_buf))
#define nextsector()    (diskio(DISK_INC,1L,sec_buf))
#define rootsector()    (diskio(DISK_SET,(long)dir_sector,sec_buf))
#define bootsector()    (diskio(DISK_SET,0L,sec_buf))
#define fatsector()     (diskio(DISK_SET,(long)reserved_secs,sec_buf))
#define lastsector()    (diskio(DISK_SET,(long)(num_sectors-1),sec_buf))
#define savesector()    (diskio(DISK_SAVE,0L,sec_buf))
#define restoresector() (diskio(DISK_REST,0L,sec_buf))
#define readsector()    (diskio(DISK_READ,-1L,sec_buf))
#define writesector()   (diskio(DISK_WRITE,-1L,sec_buf))
#define backsector()    (diskio(DISK_INC,-1L,sec_buf))
#define setphysical(t,s,h) (diskio(DISK_SET,(long)logicalsector(t,s,h),sec_buf))
#define setcluster(n)   (diskio(DISK_SET,(long)(clustertosector(n)),sec_buf))
#define nextcluster()   (diskio(DISK_SET,(long)ncluster(1),sec_buf))
#define backcluster()   (diskio(DISK_SET,(long)ncluster(-1),sec_buf))
#define nexttrack() \
         (diskio(DISK_SET,\
         (long)logicalsector(((phy_track==max_track)?(0):(phy_track+1)),\
         phy_sector,phy_head),sec_buf))
#define backtrack() \
         (diskio(DISK_SET,\
         (long)logicalsector(((phy_track==0)?(max_track):(phy_track-1)),\
         phy_sector,phy_head),sec_buf))
#define nexthead() \
         (diskio(DISK_SET,\
         (long)logicalsector(phy_track,phy_sector,\
         ((phy_head==max_head-1) ? (0) : (phy_head+1)) ),sec_buf))
#define backhead() \
         (diskio(DISK_SET,\
         (long)logicalsector(phy_track,phy_sector,\
         ((phy_head==0) ? (max_head-1) : (phy_head-1)) ),sec_buf))


/* globals for constant drive values */

   /* physical values */

extern UINT32 drive_size;     /* drive size in bytes */
extern UINT16 sec_size;       /* sector size, usually 512 bytes */
extern UINT16 max_head;       /* maximum disk head number */
extern UINT16 max_sector;     /* maximum sector number */
extern UINT16 max_track;      /* maximum track number */
extern UINT32 hidden_secs;    /* number of hidden sectors on a hard drive */
extern UINT16 reserved_secs;  /* number of hidden sectors on a hard drive */

   /* logical values */

extern UINT32 num_sectors;    /* total number of sectors */
extern UINT16 num_clusters;   /* maximum cluster no. */
extern UINT16 data_sector;    /* first data sector */
extern UINT16 dir_sector;     /* start of root directory */
extern UINT16 dir_sectors;    /* number of sectors of root */
extern UINT16 cluster_size;   /* cluster size in bytes */
extern UINT16 secs_cluster;   /* sectors per cluster */
extern UINT16 secs_fat;       /* sectors per FAT */
extern UINT16 num_fats;       /* number of FATs */
extern UINT16 dir_entries;    /* number of directory entries */
extern int max_drive;         /* maximum drive number (statistical) */
extern int fat_size;          /* 12 or 16 */
extern char dsk_format[8];
extern BYTE *sec_buf;         /* main sector buffer */
extern UINT32 sec_bufp;
extern UINT32 sec_bufx;
extern UINT16 avail_clusters;
extern MID mid;               /* media ID structure */

/* globals for variable drive values */

extern int cur_disk;             /* drive number 0+ */
extern UINT32 log_sector;        /* logical sector no. */
extern UINT32 back_log_sector;
extern UINT16 phy_head;          /* physical head no. 0-max_head */
extern UINT16 phy_track;         /* physical track no. 0-max_track */
extern UINT16 phy_sector;        /* physical sector number 1-max_sector */

extern DISKIO_STATUS diskio_status;
extern int diskio_error;
extern int disk_moved;
extern int removable;

extern const char *diskiofunc(void);
