/*
   Surface.c - data area scanning.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <stdio.h>

#include "fte.h"

static BOOL RewriteCluster(RDWRHandle handle, CLUSTER cluster);
static BOOL LenientRelocateCluster(RDWRHandle handle,
                                   CLUSTER source, CLUSTER destination);

BOOL ScanSurface(RDWRHandle handle)
{
   BOOL result;
   unsigned char sectorspercluster, j;
   unsigned long clustersindataarea, i;
   SECTOR datastart, sector;
   CLUSTER label;
   char buffer1[BYTESPERSECTOR], buffer2[BYTESPERSECTOR];

   datastart = GetDataAreaStart(handle);
   if (!datastart) return FALSE;

   sectorspercluster = GetSectorsPerCluster(handle);
   if (!sectorspercluster) return FALSE;

   clustersindataarea = GetClustersInDataArea(handle);
   if (!clustersindataarea) return FALSE;

   for (i = 2; i < clustersindataarea + 2; i++)
   {
       sector = ConvertToDataSector(handle, i);
       if (!sector) return FALSE;

       if (!GetNthCluster(handle, i, &label))
          return FALSE;

       if (!FAT_BAD(label)) /* do not scan area's which are already bad */
       {
          for (j = 0; j < sectorspercluster; j++)
          {
              if ((!ReadDataSectors(handle, 1, sector+j, buffer1))  ||
                  (!WriteDataSectors(handle, 1, sector+j, buffer1)) ||
	          (!ReadDataSectors(handle, 1, sector+j, buffer2))  ||
	          (memcmp(buffer1, buffer2, BYTESPERSECTOR) != 0))
              {
                 printf("Media error at sector %lu, relocating . . .\n", sector+j);

                 result = (RewriteCluster(handle, i) &&
                           WriteFatLabel(handle, i, FAT_BAD_LABEL));

                 SynchronizeFATs(handle);

                 if (!result)
                    return FALSE;
                 
                 break;
              }
          }
       }
   }

   if (!SynchronizeFATs(handle))
      return FALSE;
      
   return TRUE;
}

static BOOL RewriteCluster(RDWRHandle handle, CLUSTER cluster)
{
    CLUSTER freespace;
    unsigned long length;

   /* Search for the first free space. */
   if (!FindFirstFreeSpace(handle, &freespace, &length))
      return FALSE;

   for (;;)
   {
      if (length == 0)
      {
         printf("No free space to relocate cluster!\n");
         return FALSE;
      }

      switch (LenientRelocateCluster(handle, cluster, freespace))
      {
         case TRUE:          /* Recovered as much as possible. */
              return TRUE;
         
         case FALSE:         /* Problem writing to free space. */
              if (!FindNextFreeSpace(handle, &freespace, &length))
                 return FALSE;
              break;
              
         case FAIL:          /* Other media error (like in FAT, dirs, ...) */
              return FALSE;
      }
   }
}

/* This function returns FALSE if the destination is not free. */
static BOOL LenientRelocateCluster(RDWRHandle handle,
                                   CLUSTER source, CLUSTER destination)
{
   int j;
   CLUSTER fatpos, dircluster, label;
   struct DirectoryPosition dirpos;
   struct DirectoryEntry entry;
   BOOL IsInFAT = FALSE;
   SECTOR srcsector, destsector;
   unsigned char sectorspercluster, i;
   CLUSTER clustervalue;
   BOOL found, DOTSprocessed=FALSE, retval = TRUE;
   char* sectbuf;
   
   /* See wether the destination is actually free. */
   if (!GetNthCluster(handle, destination, &label))
      return FAIL;
   if (!FAT_FREE(label)) 
      return FAIL;

   /* See wether we are relocating a filled cluster */
   if (!GetNthCluster(handle, destination, &label))
      return FAIL;
   if (FAT_FREE(label))
      return TRUE;

   /* Do some preliminary calculations. */
   srcsector = ConvertToDataSector(handle, source);
   if (!srcsector)
      return FAIL;
   destsector = ConvertToDataSector(handle, destination);
   if (!destsector)
      return FAIL;
   sectorspercluster = GetSectorsPerCluster(handle);
   if (!sectorspercluster)
      return FAIL;
         
   /* Get the value that is stored at the source position in the FAT */
   if (!ReadFatLabel(handle, source, &clustervalue))
      return FAIL;
   
   /* See where the cluster is refered */
   if (!FindClusterInFAT(handle, source, &fatpos))
      return FAIL;
      
   if (!fatpos)
   {
      if (!FindClusterInDirectories(handle, source, &dirpos, &found))
         return FAIL;
      if (!found)
      {
         return TRUE;                /* Unreferenced cluster, never mind */
      }
      else
      {
         /*
            This is the first cluster of some file. See if it is a directory
            and if it is, adjust the '.' entry of this directory and all
            of the '..' entries of all the (direct) subdirectories to point
            to the new cluster.
         */
         if (!GetDirectory(handle, &dirpos, &entry))
            return FAIL;
            
         if (entry.attribute & FA_DIREC)
         {
            dircluster = GetFirstCluster(&entry);
            if (!AdaptCurrentAndPreviousDirs(handle, dircluster, destination))
            {
               return FAIL;
            }
            DOTSprocessed = TRUE;
         }
      }
   }
   else
   {
      IsInFAT = TRUE;
   }

   /* Copy as much sectors as possible in this cluster to the new position */
   sectbuf = (char*) AllocateSector(handle);
   if (!sectbuf) return FAIL;

   for (i = 0; i < sectorspercluster; i++)
   {
       for (j = 0; j < 3; j++) /* Try 3 times */
           if (ReadDataSectors(handle, 1, srcsector + i, sectbuf))
              break;
              
       if (!WriteDataSectors(handle, 1, destsector + i, sectbuf))
       {
          FreeSectors((SECTOR*) sectbuf);
          return FALSE;
        }
    }
    
    FreeSectors((SECTOR*) sectbuf);
   
   /* Write the entry in the FAT */
   if (!WriteFatLabel(handle, destination, clustervalue))
   {
      if (DOTSprocessed)
         AdaptCurrentAndPreviousDirs(handle, dircluster, source);
      return FAIL;
   }

   /* Adjust the pointer to the relocated cluster */
   if (IsInFAT)
   {
      if (!WriteFatLabel(handle, fatpos, destination))
      {
         return FAIL;
      }
   }
   else
   {
      if (!GetDirectory(handle, &dirpos, &entry))
      {
         if (DOTSprocessed)
            AdaptCurrentAndPreviousDirs(handle, dircluster, source);
      
         return FAIL;
      }
      
      SetFirstCluster(destination, &entry);
      if (!WriteDirectory(handle, &dirpos, &entry))
      {
         if (DOTSprocessed)
            AdaptCurrentAndPreviousDirs(handle, dircluster, source);
      
         return FAIL;
      }
   }

   return retval;
}
