/*
 *  dt.c        d_disk test interface
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 *  29-Nov-1998 greggj  dwrite()
 *  06-Oct-1998 prozac  ds.d_units + 1
 *
 */

/*
    See my program DISKBUG for an enhanced version of the interface
    use here.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "d_disk.h"
#ifndef NDEBUG
#include "d_lib.h"      /* this is not for normal use, but good to */
#endif                  /*  have available for debugging */

#include "iface.h"

char *version = "1.0.1";

typedef struct {
   char *cmd;
   char *hlp;
   int (*func)(char *);
} CMDS;

#define CMDARGS char *

/* command functions -- see parse(), arguments are space delimited */

int dump(CMDARGS);
int disp(CMDARGS);
int dread(CMDARGS);
int dwrite(CMDARGS);
int dseek(CMDARGS);
int dtell(CMDARGS);
int dopen(CMDARGS);
int dclose(CMDARGS);
int quit(CMDARGS);
int list(CMDARGS);
int help(CMDARGS);

/* other functions */

int get(char *);
void parse(char *);
int output(int c);
int sstrcmp(const char *a, const char *b);
int isalphas(const char *s);
int aton(const char *s);
void error(char *s);

CMDS commands[] = {
   {"open","d: [0|1|2|3]",dopen},
   {"close","",dclose},
   {"read","",dread},
   {"write","",dwrite},
   {"dump","",dump},
   {"disp","",disp},
   {"seek","[+-]n",dseek},
   {"tell","",dtell},
   {"quit","",quit},
   {"help","",help},
   {"?",NULL,list},
   {"q","",quit},
   {0}
};

int diskhandle;
unsigned char *diskbuffer;
int noread;

int main(int argc, char *argv[])
{
char inp[BUFSIZ];

#if defined __WATCOMC__ || defined __GNUC__
    setvbuf(stdout,NULL,_IONBF,0);
#endif

    diskhandle = -1;        /* no disk open */
    noread = 1;             /* no sector read */

    if (argc > 1)
        parse(argv[1]);

    for (;;)
    {
        d_errno = 0;

        if (get(inp) == -1)         /* this won't happen; it's to */
            break;                  /*  fool the compiler */
        parse(inp);
    }
    /* NOT REACHED */
    return 0;
}

#ifdef _MSC_VER
#pragma warning(disable:4100)       /* some arguments are unused */
#endif

int dopen(char *a)
{
char *m;
int bsize,mode;

    if (!a) {
        printf("\nmust supply a drive letter");
        return 1;
    }

    if (diskhandle != -1) {
        printf("\ndemo supports only one disk handle; use close first");
        return 1;
    }

    if ((m = strchr(a,' ')) != NULL)
        mode = atoi((char*)m+1);
    else
        mode = 0;

    if ((diskhandle = d_open(a,mode)) == -1) {
        error("d_open");
        return 1;
    }

    bsize = d_bsize(diskhandle);

    if ((diskbuffer = malloc((size_t)bsize)) == NULL) {
        printf("\nmalloc(%d) == NULL\n",bsize);
        exit(1);
    }
    return 0;
}

int dclose(char *a)
{
    if (diskhandle == -1)
        return 1;

    if (d_close(diskhandle) == -1) {
        error("d_close");
        return 1;
    }
    diskhandle = -1;
    noread = 1;
    free(diskbuffer);
    return 0;
}

int dread(char *a)
{
    if (diskhandle == -1)
        return 1;

    if (d_read(diskhandle,diskbuffer,1) == -1) {
        error("d_read");
        return 1;
    }

    noread = 0;

    return 0;
}

int dwrite(char *a)
{
    if (diskhandle == -1)
        return 1;

    if (noread) {
        printf("\nmust read first");
        return 1;
    }

    if (d_write(diskhandle,diskbuffer,1) == -1) {
        error("d_write");
        return 1;
    }

    return 0;
}

int dtell(char *a)
{
    if (diskhandle == -1)
        return 1;

    printf("\n%ld",d_tell(diskhandle));
    return 1;
}

int dseek(char *a)
{
long sec;
int i,sign = 0;

    if (diskhandle == -1)
        return 1;

    if (a) {
        if (*a == '-' || *a == '+') {
            sign = *a;
            ++a;
        }

        sec = strtol(a,NULL,10);

        i = (int)d_lseek(diskhandle,sec,(sign) ? SEEK_CUR : SEEK_SET);

        if (i == -1) {
            error("d_lseek");
            return 1;
        }
    }
    return 0;
}

int disp(char *a)
{
struct d_stat dstat;

    if (diskhandle == -1)
        return 1;

    d_hstat(diskhandle,&dstat);

    printf("\n");
    printf("file system:  %s\n", dstat.d_fsid);
    printf("disk id:      %s\n", dstat.d_dev);
    printf("size:         %ld\n",dstat.d_size);
    printf("sectors:      %ld\n",dstat.d_bnum);
    printf("bytes/sec:    %ld\n",dstat.d_bsize);
    printf("sec/unit:     %ld\n",dstat.d_usize);
    printf("units:        %ld\n",dstat.d_units+1);
    printf("free units:   %ld\n",dstat.d_ufree);
    printf("alloc tables: %ld\n",dstat.d_ftables);
    printf("secs/table:   %ld\n",dstat.d_ftsize);
    printf("root dirs:    %ld\n",dstat.d_ndir);
    printf("tracks:       %ld\n",dstat.d_ptracks);
    printf("heads:        %ld\n",dstat.d_pheads);
    printf("sectors:      %ld",dstat.d_psecs);
#ifndef NDEBUG

    {   struct disk *dd;
        dd = handle_todisk(diskhandle);

        printf("\n");
        printf("I/O:          %s\n",
                dd->iotype == INT25 ?    "INT25" :
                dd->iotype == INT25X ?   "INT25X" :
                dd->iotype == INT13 ?    "INT13" :
                dd->iotype == IOCTL ?    "IOCTL" :
                "INT7305");
    }
#endif
    return 0;
}

int dump(char *a)
{
int i,j;
int bufptr,n;

    if (diskhandle == -1)
        return 1;

#ifndef NDEBUG
    {
        struct disk *d;
        d = handle_todisk(diskhandle);
        printf("disk %d (%d), sector %ld, ",d->drv,d->phys,d->sec-1);
        printf("(t%u, h%u, s%u)\n",d->t,d->h,d->s);
    }
#endif

    bufptr = 0;
#if 0                                               /* short display */
    for (i = 0; i < 128; i+=16)
    {
        printf("\n%03x: ",i);
        n = bufptr;
        for (j = 0; j < 16; j++)
            printf("%02x ",diskbuffer[bufptr++]);
        for (j = 0; j < 16; j++)
            printf("%c",output(diskbuffer[n++]));
    }
#else                                               /* full display */
    for (i = 0; i < 512; i += 24)
    {
        printf("%03x: ",i);
        n = bufptr;
        for (j = 1; j <= 24; j++)
        {
            printf("%02x",diskbuffer[bufptr++]);
            if (bufptr >= 512)
            {
                putchar(' ');
                j = 9;
                break;
            }
            if (j%8 == 0)
                putchar(' ');
        }
        for (j = 1; j <= 24; j++)
        {
            printf("%c",output(diskbuffer[n++]));
            if (n >= 512)
                break;
        }
    }
#endif
    return i;
}

int output(int c)
{
    return isprint(c) ? c : '.';
}


/* IFACE (interface) stuff */

int get(char *b)
{
char buf[INSZ+3];

    _outtext("\n\r");
    _settextcolor(Yellow);
    _outtext(">>");
    _settextcolor(White);
    buf[0] = INSZ;
    _cgets(buf);
    memcpy(b,buf+2,buf[1]);
    b[buf[1]] = '\0';
    return buf[1];

    /*
        there is a bug in the DJGPP _cgets() function: extended
        keyboard input causes and infinite loop
    */

    /*
        there is a bug of unknown cause with Windows in that the
        colors get screwed up
    */

}

void parse(char *b)
{
char buf[INSZ],tbuf[INSZ];
char *cmd,*args;
static int bad;
CMDS *cmds;

    strcpy(buf,b);                  /* stash command line */
    cmd = strtok(buf,";");          /* get first argument */

    while (cmd)                     /* while there are comamnds */
    {
        for (cmds = commands; cmds->cmd ;++cmds)
        {
            if (sstrcmp(cmd,cmds->cmd))
            {
                strcpy(tbuf,cmd);           /* copy command */
                args = strchr(tbuf,' ');    /* point to arguments (if any) */
                if (args) ++args;           /* skip the ' ' */
                (cmds->func)(args);         /* call the command function */
                break;                      /* go on to next */
            }
        }

        if (!cmds->cmd)             /* command not found */
        {
            if (++bad == 5) {
                printf(" try '?'");
                bad = 0;
            }
            else
                printf(BADSTR,cmd);
        }

        cmd = strtok(NULL,";");     /* get next command */
    }
}

/* start of command functions */

/* NO RETURN */
int quit(char *a)
{
   exit((a) ? atoi(a) : 0);
   /* NOT REACHED */
   return 0;
}

int list(char *a)
{
CMDS *cmds;

   for (cmds = commands; cmds->cmd ;++cmds)
      if (cmds->hlp)
         printf("\n%s\t%s",cmds->cmd,cmds->hlp);

   return 0;
}

int help(char *a)
{
    printf("\nd_disk %s - library demo\n",version);
    printf("\n"
           "open d: [0|1|2|3]   open drive in [mode n]\n"
           "close               close drive\n"
           "read                read current position\n"
           "write               write current position\n"
           "tell                tell current position\n"
           "seek [-+]n          lseek position [from current]\n"
           "dump                dump sector\n"
           "disp                disply drive stats\n"
           "\n"
           "multiple commands may be seperated by ';'"
           "\n"
           "modes bits are: 1 read/write, 2 binary\n"
           "\n"
           );

    return 0;
}

int sstrcmp(const char *a, const char *b)
{
char *c = strchr(a,' ');

   if (c)
      return !strncmp(a,b,c-a);
   return !strcmp(a,b);
}

int isalphas(const char *s)
{
   while (*s && (isalpha(*s) || *s == ' '))
      s++;
   return *s == '\0';
}

int aton(const char *s)
{
int n = 0;

   if (isalphas(s))
      printf(" digits required");
   else if ((n = atoi((char *)s)) < 1)
      printf(" range error");

   return n;
}

void error(char *s)
{
   _outtext("\n\r");
   _settextcolor(Red);
   _outtext(s);
   _settextcolor(White);
   _outtext(" : ");
   d_perror("");
}
