/*
   LblValid.c - label validity checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include "stdio.h"

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL IsLabelValid(RDWRHandle handle, CLUSTER label);

static BOOL ValidityChecker(RDWRHandle handle, CLUSTER label,
                            SECTOR datasector, void** structure);

static BOOL ValidityFixer(RDWRHandle handle, CLUSTER label,
                          SECTOR datasector, void** structure);


/* CHECKING */
RETVAL CheckFatLabelValidity(RDWRHandle handle)
{
   BOOL invalidfound = FALSE, *pinvalidfound = &invalidfound;

   if (!LinearTraverseFat(handle, ValidityChecker, (void**) &pinvalidfound))
      return ERROR;

   return (invalidfound) ? FAILED : SUCCESS;
}

static BOOL ValidityChecker(RDWRHandle handle, CLUSTER label,
                            SECTOR datasector, void** structure)
{
   CLUSTER thisCluster;
   BOOL* invalid = *((BOOL**) structure);

   thisCluster = DataSectorToCluster(handle, datasector);
   if (!thisCluster) return FAIL;

   /* Not checking the descriptors in the first two fat labels */
   if (thisCluster < 2) return TRUE; /* Unecesary check */
   
   if (!IsLabelValid(handle, label))
   {
      printf("Invalid cluster found at: %lu\n", thisCluster);
      *invalid = TRUE;
   }

   return TRUE;
}

/* FIXING */
RETVAL MakeFatLabelsValid(RDWRHandle handle)
{
   return (LinearTraverseFat(handle, ValidityFixer, NULL)) ?
          SUCCESS : ERROR;
}

static BOOL ValidityFixer(RDWRHandle handle, CLUSTER label,
                          SECTOR datasector, void** structure)
{
   CLUSTER thisCluster;

   structure = structure;

   thisCluster = DataSectorToCluster(handle, datasector);
   if (!thisCluster) return FAIL;

   /* Not checking the descriptors in the first two fat labels */
   if (thisCluster < 2) return TRUE; /* Unecesary check */
   
   if (!IsLabelValid(handle, label))
   {
      printf("Invalid cluster found at: %lu\n", thisCluster);
      if (!WriteFatLabel(handle, thisCluster, FAT_LAST_LABEL))
         return FAIL;
   }

   return TRUE;
}

/* Common function to check wether a label is valid. */
static BOOL IsLabelValid(RDWRHandle handle, CLUSTER label)
{
   unsigned long labelsinfat;

   /* Only normal FAT labels can be detected as wrong. */
   if (!FAT_NORMAL(label)) return TRUE;
   
   /* label 1 can never be a FAT label */
   if (label == 1) return FALSE;

   /* Check wether the label is in the right range */
   labelsinfat = GetLabelsInFat(handle);
   if (!labelsinfat) return FAIL;

   return label < labelsinfat;
}
