\LIB2.XPL	23-Jul-2011
\Library of Non-Standard Routines

proc	Fatal(Str);		\Display error message and exit program
func	ExtWord(N);		\Sign-extend 16-bit word to 32-bit double word
func	Max3(A, B, C);		\Return the largest of the arguments
func	Clip(X, L);		\Clip +/-X to +/-L
func	ILn2(N);		\Integer log to base 2 (or bit number of MSB)
func	SAR(N, C);		\Shift N Arithmetic Right C bits
func	ISqrt(N);		\Return the integer square root of N
func	GCD(N, D);		\Return the Greatest Common Divisor of N and D
func real Mod(X, Y);		\Alternative to Mod intrinsic
func	WordIn;			\Input a 2-byte binary value from disk file
func	LongIn;			\Input a 4-byte binary value from disk file
func	KeyIn;			\Input a keystroke; (Ctrl+C aborts)
func	IntIn(Dev);		\Input a decimal integer from specified device
func	GetInt(N0, Size, X, Y); \Input an integer; scroll calculator-style
func	GetDec;			\Read a decimal string from GetCh
func	GetHex;			\Read a hex string from GetCh, e.g: $1AC4
func	Hex1In;			\Read in one ASCII hex digit from disk file
func	Hex2In;			\Read in two ASCII hex digits from disk file
func	Hex4In;			\Read in four ASCII hex digits from disk file
proc	WordOut(N);		\Output a 2-byte binary value to disk file
proc	LongOut(N);		\Output a 4-byte binary value to disk file
proc	IntOut(Dev, Num);	\Output a positive integer to Dev
proc	NumOut(N);		\Display a positive integer
proc	NumOut(Dev, N);		\Output a positive integer with leading zeros
proc	JustOut(Dev, N, Places); \Output a right-justified integer
proc	IntOut(Dev, N);		\Output decimal integer N to device Dev
proc	Hex3Out(Dev, N);	\Output integer N as three ASCII hex digits
proc	HexNOut(Dev, N, Digits); \Output the specified number of hex digits
proc	BinOut(Dev, N, Places, Fill); \Output N in binary (right-justified)
proc	ProseNo(Dev, Num);	\Output integer Num in prose to device Dev
proc	BigNumOut(X, Y, Num, W, H); \Display a big right-justified number
proc	DoTab(Col);		\Move cursor to column Col on the screen
func	OpenInFile(FN);		\Open file for input; return 'true' if success
func	OpenInFile;		\Open for input the file typed on command line
func	OpenOutFile(FN);	\Open for output the named file
func	OpenIO;			\Open file named on command line for I/O
func	GetFileSize(FN);	\Return the size of a file (in bytes)
proc	TermSim;		\Serial terminal simulator
func	StrLen(Str);		\Returns number of chars in an ASCIIZ string
func	StrEqual(S1, S2);	\Compare strings, return 'true' if they're equal
func	StrFind(A, B);		\Search for string A in string B, returns index
func	Concat(S1, S2, S3);	\Concatenate strings: S3:= S1 + S2
proc	StrCat(S0, S1, S2);	\Concatenate strings S0 and S1 and output to S2
func	AtoIN(Str, N);		\Convert string to positive integer
proc	AToI(Str);		\Convert string to signed integer
proc	ItoAN(I, Str, N);	\Convert positive integer I to ASCII digits
proc	ItoAN(I, Str, N);	\Convert positive integer I to ASCII digits
proc	ItoAN2(I, Str, N);	\Convert signed integer I to ASCII digits
func real AtoFN(Str, N);	\Convert ASCII string to real
func real AtoFN2(Str, Size);	\Convert ASCII string with exponent to real
proc	FtoAN(X, Str, N, M);	\Convert real to ASCII string with exponent
proc	FlOut(Dev, X, N, M);	\Output a right justified real number
proc	StrNDiv(A, B, C, N);	\Multiprecision divide A / B --> C
proc	OpenTimer;		\Initialize the (8253) system timer
proc	CloseTimer;		\Restore normal timer mode (3)
func	GetTime;		\Get time from the 8253 counter
proc	Delay(TD);		\Time delay in microseconds
func	ElapsedTime;		\Return elapsed time in microseconds
func	GetTimeLo;		\Get time from the 8253 counter
func	GetTimeHi;		\Get 16 bits of BIOS's system time
func	GetTime;		\Get 32 bits of time with 838ns resolution
proc	TimeOut(Dev, Time);	\Output specified time, e.g: 14:25:58
func	GetTime;		\Get current time-of-day in DOS packed format
func real HmsIn(Dev);		\Input HH:MM:SS.S and return equivalent hours
proc	HmsOut(Dev, HMS, Tenths); \Output HMS as HH:MM:SS.S
func real Hms2Rl(Str, Size);	\Convert HH:MM:SS.S string to equivalent hours
proc	GetSysTime;		\Get system time of day
proc	ShowTime(H, M, S, C);	\Display time
proc	ShowElapsedTime;	\Display elapsed time
proc	DateOut(Dev);		\Output current date, e.g: JUN-12-06
proc	DateOut(Dev, Date);	\Output specified date, e.g: 10-Feb-2005
proc	WaitVB;			\Wait for start of vertical blank
proc	SetBkgndColor(X, Y, C);	\Set background color for a character
proc	Backlight(X0, Y0, X1, Y1, C); \Hilight but don't change foreground color
proc	SetColorReg(N, RGB);	\Set color register to values in RGB
proc	GetColorReg(N, RGB);	\Return color register values in array RGB
func	OpenMouse;		\Initializes mouse; returns 'false' if it fails
func	MouseButton;		\Return 'true' if a mouse button is down
proc	EraseImage;		\Erase the Image array
proc	Erase(Color);		\Fill 320x200 graphic screen with given color
proc	FillScreen(Col);	\Fill 320x200 graphic screen with color
proc	DrawLine(X, Y, C); \Draw line on Image from PenX,PenY to X,Y in color C
proc	ShowLine(X1, Y1, X2, Y2, Type);	\Display line clipped to a window
proc	DrawTriangle(Tri);	\Draw a filled triangle onto Image array
proc	DrawQuad(Quad);		\Draw a filled quadrilateral onto Image array
proc	Rectangle(X0, Y0, X1, Y1, C, F); \Display a rectangle
proc	ShowRectangle(X, Y, W, H, C); \Display a rectangle
proc	ShowFilledRectangle(X, Y, W, H, C); \Display a filled rectangle
proc	ShowFilledCircle(X0, Y0, R, C);	\Display a filled circle
proc	Circle(X0, Y0, Radius, Color, Segs); \Display a circle
proc	ShowCircle(X0, Y0, Radius, Color, Filled);	\Display a circle
proc	ShowEllipse(X0, Y0, A, B, Color, Filled);	\Display an ellipse
proc	XFlood(X, Y, C, C0);	\Fill an area of color C0 with color C
proc	Flood(X, Y, C, C0);	\Fill an area of color C0 with color C
proc	GrabSprite(X0, Y0, W, H, Spr);	\Load sprite data array from Screen
proc	ShowSprite(X0, Y0, Spr); \Display a sprite
proc	VectAdd(V1, V2, V3);	\Add two 3D vectors
proc	VectSub(V1, V2, V3);	\Subtract two 3D vectors
proc	VectMul(V1, S, V2);	\Multiply 3D vector by a scaler
func real DotProd(V1, V2);	\Return the dot product of two 3D vectors
proc	CrossProd(V1, V2, V3);	\Calculate the cross product of two 3D vectors
func real VectMag(V);		\Return the magnitude (length) of a 3D vector
proc	Unitize(V);		\Convert a 3D vector to its unit vector
proc	Rotate(V, W, P, R);	\3D rotate vector V
proc	Unrotate(V, W, P, R);	\3D unrotate vector V
proc	Mat4x1Mul(M, V);	\Multiply matrix M times column vector V
proc	Mat4x4Mul(M, N);	\Multiply matrix M times matrix N
proc	Beep;			\A not-too-obnoxious beep
proc	Snd(Freq);		\Like Sound intrinsic but it doesn't wait
proc	Noise(Vol, Cy, Per);	\Emit sound on beeper speaker (like on Apple)
proc	Sort(A, N);		\Shell sort array in ascending order
proc	QSort(Array, Num, Size); \Quicksort Array into ascending order
func	Lookup(Class);		\Lookup current identifier and return its index
proc	Insert(Class, Val, Type); \Insert identifier into symbol table
proc	GetCmdLine(Str, Ext);	\Read file name from command line
proc	ShowEnvironment;	\Display Environment Block
proc	GetSBEnv;		\Get Sound Blaster's environment parameters
proc	EditWindow;		\Type text into a window and edit it
proc	FitLine(N, X, Y);	\Fit a straight line to N number of X,Y coords
proc	ColumnOut(Dev, N);	\Output 6 columns of numbers
proc	TextIn(Dev, Str, N);	\Input a string ending with a line feed
proc	MemCpy(Dst, Src, Size);	\Copy block of Size bytes from Src to Dst
proc	MemSet(Array, Value, Size); \Set Size many bytes in Array to Value
func	Sqr(N);			\Square integer (Pascal)
func	AllocMem(Bytes);	\Reserve memory permanently
func	OneBits(N);		\Return the number of 1 bits in N
func	RevBits(N);		\Reverse the order of the bits
func	RevBytes(N);		\Reverse the order of the bytes in a 32-bit word
proc	SetFont;		\Set interrupt vector $43 to point to font table



proc	Fatal(Str);		\Display error message and exit program
char	Str;
begin
SetVid($03);			\make sure screen is in normal text mode
Text(0, Str);
CrLf(0);
CrLf(0);
Text(0, "Press any key to exit program"); \ deal with windows that slam shut
OpenI(1);			\wait for new keystroke
if ChIn(1) then [];
exit;
end;	\Fatal

\------------------------------ INTEGER ROUTINES -------------------------------

func	ExtWord(N);		\Sign-extend 16-bit word to 32-bit double word
int	N;
return if N & $8000 then N ! $FFFF0000 else N & $0000FFFF;



func	Max3(A, B, C);		\Return the largest of the arguments
int	A, B, C;
return Max(Max(A, B), C);



func	Clip(X, L);		\Clip +/-X to +/-L
int	X, L;
begin
if X > L then X:= L
else if X < -L then X:= -L;
return X;
end;	\Clip



func	ILn2(N);		\Integer log to base 2 (or bit number of MSB)
int	N;			\ (this truncates, e.g: ILn2(31)=4)
int	I;			\Returns -1 if N=0 (which is an error)
begin
for I:= -1, 31 do
	begin
	if N = 0 then return I;
	N:= N >> 1;
	end;
end;	\ILn2



func	SAR(N, C);		\Shift N arithmetic right C bits
int	N, C;
begin
C:= C & $1F;	\only the low 5 bits of the shift count are used (8088 uses 8)
return if N >= 0 then N>>C else ~((~N)>>C);
end;	\SAR



func	ISqrt(N);		\Return the integer square root of N
int	N;			\ without using division
int	Min, Max,		\converging values for square root
	Root;			\best guess
begin
Min:= 0;
Max:= $7FFF;			\15 bits of precision require 15 loops, etc.
repeat	Root:= (Min+Max)>>1;	\average
	if Root*Root >= N then Max:= Root
	else Min:= Root;
until Max = Min+1;
\Determine whether Max or Min is closer
return if Max*Min >= N then Min else Max;
end;	\ISqrt



func	GCD(N, D);	\Return the greatest common divisor of N and D
int	N, D;
int	R;
begin
if D > N then
	[R:=D; D:=N; N:=R];	\swap D and N
while D > 0 do			\Eculid's method
	begin
	R:= rem(N/D);
	N:= D;
	D:= R;
	end;
return N;
end;	\GCD

\------------------------------- REAL ROUTINES ---------------------------------

func real Mod(X, Y);		\Alternative to Mod intrinsic
real	X, Y, Z;
begin
Z:= X/Y;
return X - Float(Fix(Z-0.5))*Y;
end;	\Mod

\----------------------------- CHARACTER ROUTINES ------------------------------

\------------------------------- INPUT ROUTINES --------------------------------

func	WordIn;			\Input a 2-byte binary value from disk file
begin
return ChIn(3) + ChIn(3)<<8;
end;	\WordIn



func	LongIn;			\Input a 4-byte binary value from disk file
begin
return ChIn(3) + ChIn(3)<<8 + ChIn(3)<<16 + ChIn(3)<<24;
end;	\LongIn



func	KeyIn;			\Input a keystroke; (Ctrl+C aborts)
int	Key;
begin
ShowCursor(true);		\turn on flashing cursor
Key:= ChIn(1);			\no echo to display (except ^C)
ShowCursor(false);		\turn off flashing cursor
if Key = 0 then
	Key:= -ChIn(1);		\get scan code for non-ASCII keys
return Key;			\ and return it as a negative number
end;	\KeyIn



func	IntIn(Dev);		\Input a decimal integer from specified device
int	Dev;			\ (Does not handle EOF like intrinsic does)
int	NF, Neg, Ch, N;
begin
NF:= false;
repeat	begin
	Ch:= ChIn(Dev);
	Neg:= false;
	if Ch = ^- then
		begin
		Neg:= true;
		Ch:= ChIn(Dev);
		end;
	N:= 0;
	loop	begin
		if Ch<^0 ! Ch>^9 then quit;
		NF:= true;		\indicate a digit was read in
		N:= N*10 + Ch-^0;
		Ch:= ChIn(Dev);
		end;
	end;
until NF;
return if Neg then -N else N;
end;	\IntIn



func	GetInt(N0, Size, X, Y); \Input an integer; scroll calculator-style
int	N0,	\initial value
	Size,	\number of digits in field
	X, Y;	\display position on screen
int	N,	\number
	M,	\maximum number +1
	Ch,	\digit (character) from keyboard
	First;	\flag: first digit clears display
begin
M:= 1;						\create maximum number +1
for N:= 1, Size-1 do M:= M*10;

N:= N0;						\get initial value
First:= true;
loop	begin
	Cursor(X, Y);				\display our number
	JustOut(N, Size);

	Ch:= GetKey;
	case Ch of
	  BS, -Delete: [N:= 0;   First:= true];
	  Esc:	return N0;			\return original value
	  CR:	return N			\return new value
	other
	    if Ch>=^0 & Ch<=^9 then		\ignore other characters
		begin
		if First then N:= 0		\first digit replaces number
		else	if N < M then N:= N * 10;
		N:= N/10*10 + Ch-^0;
		First:= false;
		end
	end;	\loop
end;	\GetInt



func	GetDec;			\Read a decimal string from GetCh
\The first digit is already read in, the first non-digit character (terminator)
\ will be in Char
int	Val;
begin
Val:= Char - ^0;
GetCh;
loop	begin
	if Char<^0 ! Char>^9 then quit;
	Val:= Val*10 + Char - ^0;
	GetCh;
	end;
return Val;
end;	\GetDec



func	GetHex;			\Read a hex string from GetCh, e.g: $1AC4
int	Val, I;
begin
GetCh;				\skip the "$"
case of
  Char>=^0 & Char<=^9: Val:= Char-^0;
  Char>=^A & Char<=^F: Val:= Char-$37
other [Error("HEX DIGIT EXPECTED");   return 0];
loop	begin
	GetCh;
	case of
	  Char>=^0 & Char<=^9: I:= Char-^0;
	  Char>=^A & Char<=^F: I:= Char-$37
	other quit;
	if Val > $FFF then Error("OVERFLOW");

	Val:= Val<<4 +I;
	end;
return Val;
end;	\GetHex



func	Hex1In;			\Read in one ASCII hex digit from disk file
int	Ch;			\ (for .HEX and 6502 .BIN files)
begin
Ch:= ChIn(3);
if Ch>=^0 & Ch<=^9 then return Ch - ^0;
if Ch>=^A & Ch<=^F then return Ch - ^A + 10;
return -1;			\error--hex digit expected
end;	\Hex1In



func	Hex2In;			\Read in two ASCII hex digits from disk file
return Hex1In<<4 +Hex1In;



func	Hex4In;			\Read in four ASCII hex digits from disk file
return Hex2In<<8 +Hex2In;

\------------------------------- OUTPUT ROUTINES -------------------------------

proc	WordOut(N);		\Output a 2-byte binary value to disk file
int	N;
begin
ChOut(3, N);
ChOut(3, N>>8);
end;	\WordOut



proc	LongOut(N);		\Output a 4-byte binary value to disk file
int	N;
begin
ChOut(3, N);
ChOut(3, N>>8);
ChOut(3, N>>16);
ChOut(3, N>>24);
end;	\LongOut



proc	IntOut(Dev, Num);	\Output a positive integer to Dev
int	Dev, Num;		\ (when IntOut is not available)
int	Div, Flag, Digit;
begin
Div:= 1000000000;		\for 32-bit XPL
Flag:= false;
repeat	Digit:= Num / Div;
	Num:= Rem(0);
	Flag:= Flag ! (Digit # 0);
	if Flag then ChOut(Dev, Digit+^0);
	Div:= Div / 10;
until	Div = 1;
ChOut(Dev, Num+^0);
end;	\IntOut



proc	NumOut(N);		\Display a positive integer
int	N;			\ (when IntOut is not available)
int	Q, R;
begin
Q:= N/10;
R:= rem(0);
if Q \>0\ then NumOut(Q);	\recurse
ChOut(0, R+^0);
end;	\NumOut



proc	NumOut(Dev, N);		\Output a positive integer with leading zeros
int	Dev, N;
begin
if N < 10000 then ChOut(Dev, ^0);
if N <  1000 then ChOut(Dev, ^0);
if N <   100 then ChOut(Dev, ^0);
if N <    10 then ChOut(Dev, ^0);
IntOut(Dev, N);
end;	\NumOut



proc	JustOut(Dev, N, Places); \Output a right-justified integer
int	Dev, N, Places;
begin
Format(Places, 0);
RlOut(Dev, Float(N));		\(no decimal point is displayed)
end;	\JustOut



proc	IntOut(Dev, N);		\Output decimal integer N to device Dev
int	Dev, N;
int	P, LZ, D;
begin
if N < 0 then
	begin
	N:= -N;
	ChOut(Dev, ^-);
	end;
P:= 10000;
LZ:= false;
repeat	D:= N /P;
	N:= Rem(0);
	if LZ ! D \#0\ then
		begin
		ChOut(Dev, D+^0);
		LZ:= true;
		end;
	P:= P /10;
until	P = 1;
ChOut(Dev, N+^0);
end;	\IntOut



proc	Hex3Out(Dev, N);	\Output integer N as three ASCII hex digits
int	Dev, N;
begin
Hex1Out(Dev, N >>8);
Hex2Out(Dev, N);
end;	\Hex3Out



proc	HexNOut(Dev, N, Digits); \Output the specified number of hex digits
int	Dev, N, Digits;
int	I;
	for I:= 1, Digits do Hex1Out(Dev, N>>((Digits-I)*4));



proc	BinOut(Dev, N, Places, Fill);	\Output N in binary (right-justified)
\Always outputs correct value of N regardless of Places
int	Dev,	\output device number
	N,	\16-bit unsigned integer
	Places,	\size of field in characters (right-justifies)
	Fill;	\character to fill background of field (Usually Sp or ^0)
int	I,	\index (bits are counted backwards)
	J,	\index to first bit in the field
	Mask,	\mask used to test a bit in N
	Flag;	\flag: output zeros, used to suppress leading zeros
begin
Mask:= $8000;					\initialize mask to first bit
J:= 16 -Places;
Flag:= false;
for I:= 0, 14 do
	begin
	if Mask & N then			\bit is set
		begin
		ChOut(Dev, ^1);
		Flag:= true;			\don't suppress zeros anymore
		end
	else	begin
		if Flag then ChOut(Dev, ^0)
		else	begin
			if I >= J then		\leading zero within field size
			ChOut(Dev, Fill);	\replace it with background
			end;
		end;
	Mask:= Mask >> 1;			\shift mask to next bit
	end;
ChOut(Dev, if Mask & N then ^1 else ^0);	\last bit is always sent out
end;	\BinOut



proc	ProseNo(Dev, Num);	\Output integer Num in prose to device Dev
int	Dev, Num;
int	OneTbl, TenTbl, ThoTbl,
	ThoPwr, I, Quot;

	proc	Out999(N);
	\Output number in the range 0..999 (0 does nothing)
	int	N;
	int	Huns, Tens, Ones;
	begin
	Huns:= N /100;				\0..9
	N:= Rem(0);				\0..99
	Tens:= N /10;				\0..9
	Ones:= Rem(0);				\0..9

	if Huns # 0 then
		[Text(Dev, OneTbl(Huns));	\1..9
		Text(Dev, " hundred ")];

	if Tens >= 2 then
		begin
		Text(Dev, TenTbl(Tens));
		if Ones # 0 then
			[ChOut(Dev, ^-);   Text(Dev, OneTbl(Ones))];
		end
	else	if N # 0 then Text(Dev, OneTbl(N));	\N = 1..19
	end;	\Out999

begin	\ProseNo
if Num = 0 then [Text(Dev, "zero");   return];

if Num < 0 then [Num:= -Num;   Text(Dev, "minus ")];

OneTbl:=[0, "one", "two", "three", "four",
	"five", "six", "seven", "eight", "nine",
	"ten", "eleven", "twelve", "thirteen", "fourteen",
	"fifteen", "sixteen", "seventeen", "eighteen", "nineteen"];

TenTbl:=[0, 0, "twenty", "thirty", "forty",
	"fifty", "sixty", "seventy", "eighty", "ninety"];

ThoTbl:=[" billion ", " million ", " thousand "];

\ThoPwr:= 1000000000;			\\32-bit XPL
ThoPwr:= 1000;				\16-bit XPL
for I:= 0, 2 do				\(I:= 2, 2 if 16 bits)
	begin
	Quot:= Num/ThoPwr;
	Num:= Rem(0);
	if Quot # 0 then
		[Out999(Quot);
		Text(Dev, ThoTbl(I))];
	ThoPwr:= ThoPwr/1000;
	end;
Out999(Num);
end;	\ProseNo



proc	BigNumOut(X, Y, Num, W, H);	\Display a big right-justified number
\ using the format: -XXX.XXX
\ Use large, seven-segment digits if W and H are not both zero.
int	X, Y;	\initial cursor position
real	Num,	\number to display
int	W, H;	\width and height of large digits
int	LZ, Neg, Dig, Ch;


	proc	BigDigOut(Ch);	\Display a digit
	int	Ch;
	int	JctTbl, SegTbl;
	addr	Jct, Seg;


		proc	HSegOut(N);	\Display a horizontal segment
		int	N;		\segment number (0-6)
		int	I;
		for I:= 1, W do ChOut(0, Seg(N));


		proc	VSegOut(N);	\Display a vertical segment
		int	N;		\segment number (0-6)
		int	I;
		for I:= 1, H do
			begin
			ChOut(0, Seg(N));
			ChOut(0, LF);		\move down
			ChOut(0, BS);		\ and under
			end;


	begin	\BigDigOut
	if H=0 & W=0 then
		[ChOut(0, Ch);   return];	\display at normal size
	\Otherwise display a large, seven-segment digit

	\This table contains the graphic codes which describe the junctions
	\ for each digit.
	\	 01234567	Digit		Order of the Junctions
	JctTbl:=[
		"ڿ  ",	\0			01
		"     ",	\1			   
		"ڿ  ",	\2			23
		"ڿ   ",	\3			   
		"   ",	\4			45
		"  ",	\5
		" ÿ  ",	\6
		"ڿ    ",	\7
		"ڿô  ",	\8
		"ڿ   ",	\9
		"        ",	\space
		"        "];	\minus sign

	\This table contains the graphic codes which describe the segments
	\ for each digit.
	\	 01234567	Digit		Order of the Segments
	SegTbl:=[
		"ĳ  ",	\0			 0 
		"      ",	\1			1   2
		" ĳ  ",	\2			 3 
		"   ",	\3			4   5
		"    ",	\4			 6 
		"ĳ   ",	\5
		"  ĳ ",	\6
		"     ",	\7
		"ĳĳ ",	\8
		"ĳ   ",	\9
		"        ",	\space
		"       "];	\minus sign

	case Ch of				\convert ASCII to table index
	  Sp:	Ch:= 10;
	  ^-:	Ch:= 11
	other Ch:= Ch -^0;

	if Ch>=0 & Ch<=11 then			\decimal point just skips
		begin
		Jct:= JctTbl(Ch);		\get string for digit
		Seg:= SegTbl(Ch);

		\Display a seven-segment digit
		ChOut(0, Jct(0));	HSegOut(0);	ChOut(0, Jct(1));
		ChOut(0, LF);		ChOut(0, BS);	VSegOut(2);
		Cursor(X, Y+1);		VSegOut(1);
		ChOut(0, Jct(2));	HSegOut(3);	ChOut(0, Jct(3));
		ChOut(0, LF);		ChOut(0, BS);	VSegOut(5);
		Cursor(X, Y+2+H);	VSegOut(4);
		ChOut(0, Jct(4));	HSegOut(6);	ChOut(0, Jct(5));
		end;
	X:= X +2 +W +3;				\move to start of next digit
	Cursor(X, Y);
	end;	\BigDigOut


begin	\BigNumOut
OpenO(8);
if Num >=0.0 then ChOut(8, Sp);
Format(3, 3);
RlOut(8, Num);
ChOut(8, EOF);

OpenI(8);
loop	begin
	Ch:= ChIn(8);
	if Ch=EOF then quit;
	BigDigOut(Ch);
	end;
end;	\BigNumOut



proc	DoTab(Col);		\Move cursor to column Col on the screen
int	Col;
int	I;
begin
I:= 0;
repeat	begin
	ChOut(TV, ^ );
	CpuReg(0):= $0300;	\function $03
	CpuReg(1):= $0000;
	SoftInt($10);
	I:= I+1;		\prevent potiential infinite loop
	end;
until	(CpuReg(3)&$00FF)>=Col ! I>=Col;
end;	\DoTab

\-------------------------------- FILE ROUTINES --------------------------------

func	OpenInFile(FN);		\Open file for input; return 'true' if success
char	FN;	\file name string
int	H;	\handle
begin
Trap(false);
H:= FOpen(FN, 0);
FSet(H, ^i);			\small buffer allows multiple files
OpenI(3);
if GetErr then return false;
Trap(true);
return true;
end;	\OpenInFile



func	OpenInFile;		\Open for input the file typed on command line
int	CpuReg, Hand;
char	CmdTail($80);
begin
CpuReg:= GetReg;
Blit(CpuReg(11), $81, CpuReg(12), CmdTail, $7F);       \get copy of command line
Trap(false);
Hand:= FOpen(CmdTail, 0);	\open named file for input
FSet(Hand, ^I);
OpenI(3);
if GetErr then return false;
Trap(true);
return true;
end;	\OpenInFile



func	OpenOutFile(FN);	\Open for output the named file
char	FN;	\file name string
int	H;	\handle
begin
Trap(false);
H:= FOpen(FN, 1);
FSet(H, ^o);			\small buffer allows multiple files and it is
OpenO(3);			\ closed automatically when the program exits
if GetErr then return false;
Trap(true);
return true;
end;	\OpenOutFile



func	OpenIO;			\Open file named on command line for I/O
int	CpuReg, PspSeg, DataSeg, Hand;	\return 'true' if successful
int	J, J0, ExtFlag, I;
char	ExtIn, ExtOut;
char	CmdTail($84);
begin
ExtIn:= ".TXT";		\default input extension
ExtOut:= ".SRT";	\output extension

CpuReg:= GetReg;	\get file name from PSP to our data segment
PspSeg:= CpuReg(11);
DataSeg:= CpuReg(12);
Blit(PspSeg, $80, DataSeg, CmdTail, $80);

ExtFlag:= false;	\assume there is no extension on command line
J:= 1;
loop	begin
	if CmdTail(J) = ^. then		\extension is on command line
		[ExtFlag:= true;  quit];
	if J > CmdTail(0) then quit;	\if quit then J points to CR
	J:= J+1;
	end;
if not ExtFlag then	\tack on extension
	for I:= 0, 3 do CmdTail(J+I):= ExtIn(I);
Trap(false);
Hand:= FOpen(CmdTail+1, 0);
FSet(Hand, ^I);
OpenI(3);

for I:= 0, 3 do CmdTail(J+I):= ExtOut(I); \output file extension
Hand:= FOpen(CmdTail+1, 1);
FSet(Hand, ^o);		\small buffer allows multiple files and it's
OpenO(3);		\ closed automatically when the program exits

if GetErr then return false;
Trap(true);
return true;
end;	\OpenIO



func	GetFileSize(FN);	\Return the size of a file (in bytes)
\(Note: Trapping the "read beyond EOF" error is a simpler way to detect EOF.)
char	FN;		\file name
int	PspSeg,		\Program Segment Prefix segment, holds command line
	CpuReg,		\address of CPU registers from GetReg
	DataSeg,	\data segment address for this program
	T;		\temporary scratch
char	DTA;		\Disk Transfer Access area for DOS calls
begin
\WARNING: must be called at start of program
CpuReg:= GetReg;		\get address of CPU registers
PspSeg:= CpuReg(11);
DataSeg:= CpuReg(12);

DTA:= Reserve($80);		\set up disk transfer access (DTA) area
				\use 128 bytes even though 34 is enough
CallInt($21, $1A00, 0, 0, DTA, 0, DataSeg);

\Look up first file name, including everything except volume label
T:= CallInt($21, $4E00, 0, $0037, FN, 0, DataSeg);

return	if (T&$FF) \#0\ then 0 else DTA($1A) + Swap(DTA($1B));
end;	\GetFileSize

FN:= "DI.EXE ";   FN(6):= 0;	\(yuck!)
GetFileSize(FN);

\---------------------------- SERIAL I/O ROUTINES ------------------------------

proc	TermSim;		\Serial terminal simulator
int	SerData,	\serial port data register
	SerStat;	\serial port status register
int	Char;



proc	SerOpen(Comm, Baud);	\Initialize communications port with baud rate
int	Comm,		\comm port (1, 2, 3, 4)
	Baud;		\baud rate (unsigned)
int	N, Base;
begin
if Comm<1 ! Comm>4 then 
	begin
	Text(0, " ILLEGAL COMM PORT NUMBER ");
	OpenI(1);
	repeat until ChkKey;
	end;
Comm:= (Comm-1)*2;
Base:= Peek($40, Comm) + Swap(Peek($40, Comm+1));
if Base = 0 then 
	begin
	Text(0, " COMM PORT IS NOT INSTALLED ");
	OpenI(1);
	repeat until ChkKey;
	end;
SerData:= Base;
SerStat:= Base+5;

POut($80, Base+3, 0);		\set up baud rate generator
N:= 1152 / (Baud>>1/50);	\first divide unsigned baud rate by 100
POut(Swap(N), Base+1, 0);
POut(N, Base, 0);

\   BITS:	765		43		2		10
\		NOT USED	PARITY		STOP BITS	WORD LENGTH
\		XXX		X0 = none	0 = 1 bit	10 = 7 bits
\				01 = odd	1 = 2 bits	11 = 8 bits
\				11 = even
POut($03, Base+3, 0);		\no parity, 1 stop, 8 bits

POut($00, Base+1, 0);		\no interrupts
POut($03, Base+4, 0);		\turn on RTS and DTR

N:= PIn(Base, 0);		\"kick some life into it"
N:= PIn(Base, 0);
end;	\SerOpen



proc	SerOut(Ch);		\Output a character to the selected COMM port
int	Ch;
begin
repeat until (PIn(SerStat,0) & $20) \#0\;	\transmitter holding reg empty
POut(Ch, SerData, 0);
end;	\SerOut



func	SerIn;			\Receive a character from serial port
begin
repeat until (PIn(SerStat,0) & $01) \#0\;	\data ready
return PIn(SerData, 0);
end;	\SerIn



func	SerChk;			\Check if input character is available
return (PIn(SerStat,0) & $01) # 0;



begin	\TermSim
Text(0, "SEND AND RECEIVE FROM SERIAL PORT
   ESC TO QUIT

");

SerOpen(1, 9600);

loop	begin
	if SerChk then
		begin
		Char:= SerIn;
		ChOut(0, Char);
		end;
	if ChkKey then 
		begin
		Char:= ChIn(1);
		if Char = $1B \Esc\ then quit;
		SerOut(Char);
		end;
	end;
end;	\TermSim

\------------------------------- STRING ROUTINES -------------------------------

func	StrLen(Str);		\Returns number of chars in an ASCIIZ string
char	Str;
int	I;
begin
I:= 0;
while Str(I) \#0\ do I:= I+1;
return I;
end;	\StrLen



func	StrEqual(S1, S2);	\Compare strings, return 'true' if they're equal
char	S1, S2;
int	I;
begin
for I:= 0, 32000 do
	begin
	if S1(I) # S2(I) then return false;
	if S1(I) > $7F then return true;
	end;
end;	\StrEqual



func	StrFind(A, B);		\Search for string A in string B
\Returns the index of first occurance of string A in B or -1 if A is not found
char	A, B;			\strings to be compared
int	LenA, LenB, I, J;
begin
LenA:= StrLen(A);
LenB:= StrLen(B);
for I:= 0, LenB-LenA do
	begin
	for J:= 0, LenA-1 do
		if (A(J)&$7F) # (B(J+I)&$7F) then J:= LenA+1;
	if J = LenA then return I;	\found
	end;
return -1;
end;	\StrFind



func	Concat(S1, S2, S3);	\Concatenate strings: S3:= S1 + S2
char	S1, S2, S3;
int	C, I, J;
begin
I:= 0;
repeat	C:= S1(I);
	S3(I):= C & $7F;
	I:= I+1;
until	C >= $80;
J:= 0;
repeat	C:= S2(J);
	S3(I+J):= C;
	J:= J+1;
until	C >= $80;
return S3;
end;	\Concat



proc	StrCat(S0, S1, S2);	\Concatenate strings S0 and S1 and output to S2
char	S0, S1, S2;
int	I, J;
begin
I:= 0;
while S0(I) \#0\ do [S2(I):= S0(I);  I:= I+1];
J:= 0;
loop	begin
	S2(I):= S1(J);
	if S1(J) = 0 then quit;
	I:= I+1;
	J:= J+1;
	end;
end;	\StrCat



func	AtoIN(Str, N);		\Convert string to positive integer
\This ignores leading blanks, minus signs, decimal points, commas, etc.
char	Str;
int	N;
int	I, Num, Digit;
begin
Num:= 0;
for I:= 0, N-1 do
	begin
	Digit:= (Str(I) & $7F) - ^0;
	if Digit>=0 & Digit<=9 then
		Num:= Num*10 + Digit
	end;
return Num;
end;	\AtoIN



proc	AToI(Str);		\Convert string to signed integer
\String must be terminated with a non-numeric character
char	Str;
int	I, Ch, Num, Neg;
begin
I:= 0;
repeat	Ch:= Str(I);		\skip leading garbage such as spaces or '+'
	I:= I+1;
until Ch>=^0 & Ch<=^9 ! Ch=^- ! I>=3;  \limit number of chars of leading garbage
				       \ to prevent a possible infinite loop
Neg:= false;
if Ch=^- then
	[Neg:= true; Ch:= Str(I); I:= I+1];

Num:= 0;
while Ch>=^0 & Ch<=^9 do
	begin
	Num:= Num*10 + Ch-^0;
	Ch:= Str(I);
	I:= I+1;
	end;
return if Neg then -Num else Num;
end;	\AToI



proc	ItoAN(I, Str, N);	\Convert positive integer I to ASCII digits
\This produces a string containing the ASCII representation of an integer.
\ The output string (Str) is assumed to be at least N bytes long. The last
\ byte does not have the MSB set. The number is right rather than left
\ justified. N must be at least 1.
int	I;	\input integer
char	Str;	\output string
int	N;	\number of digits allowed for (size of Str)
int	J;
begin
for J:= 0, N-1 do
	begin
	I:= I/10;
	Str(N-1-J):= Rem(0) + ^0;
	end;
if I \#0\ then [Error("ITOAN OVERFLOW");   return];
J:= 0;				\replace leading zeros with spaces
while J<N & Str(J)=^0 do [Str(J):= Sp;   J:= J+1];
if J = N then Str(J-1):= ^0;	\make sure at least one zero remains
end;	\ItoAN



proc	ItoAN(I, Str, N);	\Convert positive integer I to ASCII digits
\This produces a string containing the ASCII representation of an integer.
\ The output string (Str) is assumed to be at least N bytes long. The last
\ byte does not have the MSB set. The number is right rather than left
\ justified. N must be at least 1.
int	I;	\positive integer to convert
char	Str;	\string to output (right-justified)
int	N;	\number of digits to convert (= size of Str)
int	Ch;
begin
Ch:= ^0;
for N:= 1-N, 0 do
	begin
	I:= I/10;
	Str(-N):= rem(0) + Ch;
	if I=0 then Ch:= ^ ;
	end;
end;	\IToAN



proc	ItoAN2(I, Str, N);	\Convert signed integer I to ASCII digits
\This routine produces a string containing the ASCII representation of an
\ integer number. The string is assumed to be at least N bytes long. The last
\ byte does NOT have the high bit set. The number is right rather than left
\ justified. If the quantity will not fit then the first byte of the string is
\ set to "*" and the remainder of the string is undefined. N must be at least
\ one.
int	I;
char	Str;
int	N;
int	J, Neg;
begin
Neg:= I < 0;
if Neg then I:= -I;
for J:= 0, N-1 do
	begin
	I:= I/10;
	Str(N-1-J):= Rem(0) + ^0;
	end;
if I >= 1 then [Str(0):= ^*;   return];
J:= 0;				\replace leading zeros with spaces
while J<N & Str(J)=^0 do [Str(J):= Sp;   J:= J+1];
if J = N then Str(J-1):= ^0;	\make sure at least one zero remains
if Neg then
	if J >= 1 then Str(J-1):= ^-
	else Str(0):= ^*;
end;	\ItoAN2



func real AtoFN(Str, N);	\Convert ASCII string to real
char	Str;
int	N;
int	I, C;
real	X, Y;
begin
X:= 0.0;
Y:= 0.0;
for I:= 0, N-1 do
	begin
	C:= Str(I) & $7F;
	if C>=^0 & C<=^9 ! C=^. then
		begin
		if C=^. then 
			[if Y=0.0 then Y:= 1.0]
		else	begin
			X:= X*10.0 + Float(C-^0);
			Y:= Y*10.0;
			end;
		end;
	end;
if Y > 0.0 then X:= X/Y;
return X;
end;	\AtoFN



func real AtoFN2(Str, Size);	\Convert ASCII string with exponent to real
char	Str;	\string
int	Size;	\number of characters in Str
int	Ch,	\character
	Inx,	\index
	Ex,	\power-of-ten exponent, total effective value
	N,	\exponent as specified by input
	Neg,	\flag: negative real number
	ENeg,	\flag: negative exponent
	Digit;	\flag: last character is a digit (0 thru 9)
real	X,	\value of real number
	Ten;	\10.0; avoids use of real constants which are not easily
		\ ported from one floating-point representation to another

	proc	GetCh;		\Get character from Str
	begin
	if Inx < Size then
		[Ch:= Str(Inx);   Inx:= Inx+1]
	else Ch:= ^0;

	Digit:= Ch>=^0 & Ch<=^9;	\is it a digit?
	end;	\GetCh

begin	\AtoFN2
Inx:= 0;
Ten:= Float(10);
Neg:= false;
loop	begin
	GetCh;			\ignore any leading garbage
	if Ch =^- then Neg:= ~Neg;
	if Digit then
		begin
		X:= Float(Ch -^0);
		loop	begin
			if Inx >= Size then quit;
			GetCh;
			if ~Digit then quit;
			X:= X *Ten + Float(Ch -^0);
			end;
		quit;
		end;
	if Ch=^. then [X:= Float(0);   quit];
	end;
Ex:= 0;
if Ch = ^. then
	loop	begin
		if Inx >= Size then quit;
		GetCh;
		if ~Digit then quit;
		X:= X*Ten + Float(Ch-^0);
		Ex:= Ex-1;	\if X gets bigger, the exponent
				\gets smaller
		end;
if Ch=^E ! Ch=^e then
	begin
	N:=0;
	GetCh;
	if Ch = ^- then [ENeg:= true;   GetCh] else ENeg:= false;
	if Ch = ^+ then GetCh;
	while Digit do [N:= N *10 +(Ch -^0);   GetCh];
	Ex:= Ex + (if ENeg then -N else N);
	end;
while Ex < 0 do [X:= X/Ten;   Ex:= Ex+1];
while Ex > 0 do [X:= X*Ten;   Ex:= Ex-1];
return if Neg then -X else X;
end;	\AtoFN2



proc	FtoAN(X, Str, N, M);	\Convert real to ASCII string with exponent
real	X;
char	Str;
int	N, M;
\This routine produces a string containing the ASCII representation of a
\ floating point number, X. The string is assumed to be at least N+M+1 bytes
\ long. It produces N bytes before the decimal and M bytes after it. The
\ last byte does NOT have the high bit set. The number is right rather than
\ left justified. If the quantity will not fit then the first byte of the
\ string is set to "*" and the remainder of the string is undefined. The
\ decimal itself is always present. It is in the last byte if M=0 (which is
\ legal). N must be at least one. exponential and other notations are not
\ supported.
int	I, Neg, Digit, Len;
real	Hold, Expo, Comp, Round;

	proc	Fill;
	int	I;
	for I:= 0, Len-1 do Str(I):= ^*;

begin
Len:= if M = 0 then N else N+M+1;
Neg:= X < 0.0;
if Neg then X:= -X;
Round:= 0.5;
for I:= 0, M-1 do Round:= Round * 0.1;
X:= X + Round;
Hold:= X;
Expo:= 0.0;
Comp:= 1.0;
for I:= 0, N-1 do
	begin
	Digit:= Fix(Mod(X,10.0) - 0.5);
	Expo:= Expo + Comp*Float(Digit);
	Comp:= Comp * 10.0;
	Str(N-I-1):= Digit + ^0;
	X:= X / 10.0;
	end;
if X >= 1.0 then [Fill; return];

I:= 0;
while I<N & Str(I)=^0 do [Str(I):= ^ ;   I:= I+1];
if I = N then [Str(I-1):= ^0;   I:= I-1];
if Neg then
	begin
	if I <= 0 then
		[Fill; return]
	else	Str(I-1):= ^-;
	end;
X:= Hold - Expo;
Str(N):= ^.;
for I:= 1, M do
	begin
	X:= X * 10.0;
	Digit:= Fix(X-0.5);
	Str(N+I):= Digit + ^0;
	X:= X - Float(Digit);
	end;
end;	\FtoAN



proc	FlOut(Dev, X, N, M);	\Output a right justified real number
int	Dev;
real	X;
int	N, M;
char	Str;
int	I, Len;
begin
Str:= Reserve(N+M+1);
FtoAN(X, Str, N, M);
I:= 0;
while Str(I) = ^  do I:= I+1;
Len:= if M=0 then N else N+M+1;
for I:= I, Len-1 do ChOut(Dev, Str(I));
end;	\FlOut



proc	StrNDiv(A, B, C, N);	\Multiprecision divide A / B --> C
char	A;	\ASCII string containing dividend
		\Must be right-justified. Fill with leading zeros if necessary.
int	B;	\integer divisor (1..3276) 
char	C;	\ASCII string for quotient (result)
int	N;	\number of ASCII digits in string A
int	D, I;
begin
D:= 0;
for I:= 0, N-1 do
	begin
	D:= D + A(I) - ^0;	\get digit (converted to binary)
	C(I):= D/B + ^0;	\divide and convert to ASCII
	D:= Rem(0)*10;
	end;
end;	\StrNDiv

\------------------------------- TIME ROUTINES ---------------------------------

proc	OpenTimer;		\Initialize the (8253) system timer
\The system timer (timer 0) is normally set to mode 3, as defined by IBM.
\ Some BIOSes, Win3.1 and WinXP have a bug that instead sets this timer to
\ mode 2. This does not change the interrupt rate, but it does change the
\ rate that the internal counter decrements. Mode 3 decrements by 2 (and
\ makes two pases), and mode 2 decrements by 1. Since the internal counter
\ is read by this code, it is essential that its rate be correct. Of course
\ WinXP (in its infinite wisdom) does not allow reprogramming the mode, so
\ the non-standard (but more logical) mode 2 is used here.
begin
port($43):= $34;		\set timer 0 to mode 2
port($40):= $FF;		\set 16-bit countdown timer to maximum count
port($40):= $FF;		\ (low byte first)
end;	\OpenTimer;



proc	CloseTimer;		\Restore normal timer mode (3)
begin
port($43):= $36;		\set timer 0 to mode 3
port($40):= $FF;		\set to maximum count
port($40):= $FF;		\ (low byte first)
end;	\CloseTimer



func	GetTime;		\Get time from the 8253 counter
int	T;
begin				\each count is 838 ns (= 1 / 1.19e6 Hz)
port($43):= 0;			\latch counter 0
T:= port($40) + port($40)<<8;	\read 2 bytes
return T | $0000FFFF;		\reinvert the bits
end;	\GetTime



proc	Delay(TD);		\Time delay in microseconds
int	TD;	\(maximum is 26000, even for 32-bit XPL)
int	Cnt, T0;
begin
Cnt:= TD*1000/838;
T0:= GetTime;
repeat until ((GetTime-T0) & $FFFF) >= Cnt;  \25ms = 25e6ns; 25e6ns/838 = 29833
end;	\Delay



	proc	Delay(D);
	int	D;  \number of microseconds to delay
	int	T;
	begin
	T:= GetTime;	\intrinsic
	repeat until GetTime-T >= D;
	end;



func	ElapsedTime;		\Return elapsed time in microseconds
int	T;			\ StartTime:= GetTime
begin				\ (must be <= 54,919 microseconds)
T:= GetTime;
T:= T - StartTime;
if T < 0 then T:= T + 65536;
return T*838/1000;
end;	\ElapsedTime



func	GetTimeLo;		\Get time from the 8253 counter-timer
int	T;
begin				\each count is 838 ns (= 1 / 1.19E6 Hz)
port($43):= 0;			\latch counter 0
T:= port($40) + port($40)<<8;	\read 2 bytes
return T | $0000FFFF;		\reinvert the bits, so they're right side up
end;	\GetTimeLo



func	GetTimeHi;		\Get 16 bits of BIOS's system time
int	L, H0, H;
begin
H0:= Peek($40, $6D);		\read high byte
L:=  Peek($40, $6C);		\read low byte

\WinXP is awkward at best about turning off interrupts so the following
\ determines if a system timer interrupt occurred that makes the H0 and
\ L bytes inconsistent with each other.
H:= Peek($40, $6D);
if H # H0 then			\if an interrupt caused a carry then
	L:= Peek($40, $6C);	\ reread the low byte
return H<<8 + L;
end;	\GetTimeHi



func	GetTime;		\Get 32 bits of time with 838ns resolution
int	L, H0, H;
begin
H0:= GetTimeHi;
L:= GetTimeLo;

H:= GetTimeHi;
if H # H0 then
	L:= GetTimeLo;
return H<<16 + L;
end;	\GetTime



proc	TimeOut(Dev, Time);	\Output specified time, e.g: 14:25:58
int	Dev, Time;	\time in DOS packed format
int	H, M, S;	\hours, minutes, seconds
begin
S:= Time<<1 & $003E;		\0..58
M:= Time>>5 & $003F;		\0..59
H:= (Time>>11 & $001F);		\0..23

if H < 10 then ChOut(Dev, ^0);
IntOut(Dev, H);
ChOut(Dev, ^:);
if M < 10 then ChOut(Dev, ^0);
IntOut(Dev, M);

ChOut(Dev, ^:);
if S < 10 then ChOut(Dev, ^0);
IntOut(Dev, S);
end;	\TimeOut



func	GetTime;		\Get current time-of-day in DOS packed format
begin
CallInt($21, $2C00);
return (CpuReg(2)&$1F00)<<3 ! (CpuReg(2)&$003F)<<5 ! (CpuReg(3)&$3F00)>>9;
end;	\GetTime



func real HmsIn(Dev);		\Input HH:MM:SS.S and return equivalent hours
int	Dev;	\device number
int	Neg,	\flag: minus sign seen
	CrFlag,	\flag: carriage return seen
	N,	\integer value after the decimal point (if any)
	Dem;	\denominator for correcting fractional values
real	X;	\value that is returned

	func	ChInX(Dev);
	int	Dev;
	int	Ch;
	begin
	if CrFlag then return ^0;
	Ch:= ChIn(KB);
	if Ch=CR then CrFlag:= true;
	return Ch;
	end;	\ChInX


	func	GetNum;		\Get number from device, look for minus signs
	int	Ch, Num;
	begin
	loop	begin		\ignore leading garbage (non-numerics)
		Ch:= ChInX(Dev);
		if Ch>=^0 & Ch<=^9 ! Ch=^- then quit;
		end;
	if Ch=^- then [Neg:= true;   Ch:= ChInX(Dev)];

	Num:= Ch -^0;
	loop	begin
		Ch:= ChInX(Dev);
		if Ch<^0 ! Ch>^9 ! CrFlag then quit;
		Num:= Num*10 + Ch-^0;
		end;

	return Num;
	end;	\GetNum

begin	\HmsIn
Neg:= false;
CrFlag:= false;
X:= Float(GetNum) + Float(GetNum)/60.0 + Float(GetNum)/3600.0;
N:= GetNum;	\deal with wise guys who put more than one digit after the D.P.
Dem:= 1;
while N/Dem >= 10 do Dem:= Dem*10;
X:= X + Float(N)/36000.0/Float(Dem);
return if Neg then -X else X;
end;	\HmsIn



proc	HmsOut(Dev, HMS, Tenths); \Output HMS as HH:MM:SS.S
\HMS must be < 100.
int	Dev;	\output device number
real	HMS;	\value to output
int	Tenths;	\flag to display tenths of a second
char	A;
int	HH, MM, SS;

	func	Trunc(X);	\Truncate X to integer
	real	X;
	int	I;
	begin			\(beware of numbers like 51.9999999)
	I:= Fix(X-.5);
	if I < 0 then I:= 0;	\(beware of 0.999999999)
	return I;
	end;	\Trunc


	proc	NumOut(N);
	\Output a positive 2-digit integer with leading zeros
	int	N;
	begin
	ChOut(Dev, N/10 + ^0);
	ChOut(Dev, Rem(0) + ^0);
	end;	\NumOut

begin	\HmsOut
if HMS < 0.0 then
	[ChOut(Dev, ^-); HMS:= -HMS]
else ChOut(Dev, ^ );

HMS:= HMS + (if Tenths then 1.39E-5 else 1.39E-4);	\round

A:= addr HMS;			\make HMS a nice number that won't
A(4):= 0;			\ give rounding problems

HH:= Trunc(HMS);
NumOut(HH);
ChOut(Dev, ^:);

HMS:= (HMS - Float(HH)) * 60.0;
MM:= Trunc(HMS);
NumOut(MM);
ChOut(Dev, ^:);

HMS:= (HMS - Float(MM)) * 60.0;
SS:= Trunc(HMS);
NumOut(SS);

if Tenths then			\show tenths of seconds
	begin
	ChOut(Dev, ^.);
	HMS:= (HMS - Float(SS)) * 10.0;
	ChOut(Dev, Trunc(HMS) +^0);
	end;
ChOut(Dev, CR);
end;	\HmsOut



func real Hms2Rl(Str, Size);	\Convert HH:MM:SS.S string to equivalent hours
char	Str;	\string containing HH MM SS.S
int	Size;	\length of Str
int	Inx,	\index into Str array
	Neg; 	\flag: minus sign seen
real	Num;	\hours * 360000.0


	func	GetCh;		\Get next character from Str
	int	Ch;
	begin
	if Inx >= Size then return ^0;
	Ch:= Str(Inx);
	Inx:= Inx+1;
	return Ch;
	end;	\GetCh



	func real GetNum;	\Get number, look for minus signs
	int	Ch, N;
	begin
	\Ignore leading garbage (i.e: non-numerics)
	loop	begin
		Ch:= GetCh;
		if Ch>=^0 & Ch<=^9 ! Ch=^- then quit;
		if Ch=^. ! Ch=^: then return 0;
		end;
	if Ch=^- then [Neg:= true;   Ch:= GetCh];

	N:= Ch - ^0;
	loop	begin
		if Inx >= Size then quit;
		Ch:= GetCh;
		if Ch<^0 ! Ch>^9 then quit;
		N:= N*10 + Ch - ^0;
		end;

	return Float(N);
	end;	\GetNum


begin	\Hms2Rl
Inx:= 0;
Neg:= false;
Num:= GetNum*36000.0 + GetNum*600.0 + GetNum*10.0 + GetNum;
if Neg then Num:= -Num;
return Num / 36000.0;
end;	\Hms2Rl

\-------------------------------------------------------------------------------

int	Hour, Minute, Second, Hundreth,	\for measuring elapsed time
	HR1, MN1, SC1, HN1;

proc	GetSysTime;		\Get system time of day
begin
CpuReg(0):= $2C00;
SoftInt($21);
Hour:= CpuReg(2)>>8 & $00FF;
Minute:= CpuReg(2) & $00FF;
Second:= CpuReg(3)>>8 & $00FF;
Hundreth:= CpuReg(3) & $00FF;
end;	\GetSysTime



proc	ShowTime(H, M, S, C);	\Display time
int	H, M, S, C;
begin
IntOut(0, H); Text(0, ":");
if M<10 then ChOut(0, ^0);
IntOut(0, M); Text(0, ":");
if S<10 then ChOut(0, ^0);
IntOut(0, S); Text(0, ".");
if C<10 then ChOut(0, ^0);
IntOut(0, C);
end;	\ShowTime



proc	ShowElapsedTime;	\Display elapsed time
int	HR2, MN2, SC2, HN2;
begin
Text(0, "Starting time: ");
ShowTime(HR1, MN1, SC1, HN1);
CrLf(0);
Text(0, "Ending time:   ");
ShowTime(Hour, Minute, Second, Hundreth);
CrLf(0);

if HN1 > Hundreth then		\deal with borrows (Larry's method)
	[SC1:= SC1+1; Hundreth:= Hundreth+100];
HN2:= Hundreth - HN1;
if SC1 > Second then
	[MN1:= MN1+1; Second:= Second+60];
SC2:= Second - SC1;
if MN1 > Minute then
	[HR1:= HR1+1; Minute:= Minute+60];
MN2:= Minute - MN1;
if HR1 > Hour then Hour:= Hour+24;
HR2:= Hour - HR1;

Text(0, "Elapsed time:  ");
ShowTime(HR2, MN2, SC2, HN2);
CrLf(0);
end;	\ShowElapsedTime

\-------------------------------- DATE ROUTINES --------------------------------

proc	DateOut(Dev);		\Output current date, e.g: JUN-12-06
int	Dev;			\I/O device number
int	Year, Mo, Day, I;
char	MoStr;


	proc	NumOut(N);	\Output 2-digit number with leading 0
	int	N;
	begin
	if N < 10 then ChOut(Dev, ^0);
	IntOut(Dev, N);
	end;


begin	\DateOut
CpuReg(AX):= $2A00;
SoftInt($21);
Day:= CpuReg(DX) &$FF;		\1..31
Mo:= Swap(CpuReg(DX)) &$FF;	\1..12
Year:= CpuReg(CX);		\1980..2099

MoStr:= "JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC ";
Mo:= (Mo-1) *3;
for I:= 0, 2 do ChOut(Dev, MoStr(Mo+I));
ChOut(Dev, ^-);
NumOut(Day);
ChOut(Dev, ^-);
Year:= Rem(Year/100);
NumOut(Year);
end;	\DateOut



proc	DateOut(Dev, Date);	\Output specified date, e.g: 03-Feb-2011
int	Dev, Date;	\date in DOS packed format
int	D, M, Y,	\day, month, year
	I, J;
char	Str;
begin
D:= Date & $001F;		\1..31
M:= Date>>5 & $000F;		\1..12
Y:= (Date>>9 & $007F) + 1980;	\1980..2107

if D < 10 then ChOut(Dev, ^0);
IntOut(Dev, D);
ChOut(Dev, ^-);
Str:= "JanFebMarAprMayJunJulAugSepOctNovDec ";
J:= 3*(M-1);
for I:= 0, 3-1 do ChOut(Dev, Str(I+J));
ChOut(Dev, ^-);
IntOut(Dev, Y);
end;	\DateOut

\------------------------------- VIDEO ROUTINES --------------------------------

proc	WaitVB;			\Wait for start of vertical blank
begin				\(actually waits for start of vertical retrace)
while port($3DA) & $08 do [];	\wait for vertical retrace to go away
repeat until port($3DA) & $08;	\wait for vertical retrace
end;	\WaitVB



proc	SetBkgndColor(X, Y, C);	\Set background color for a character
int	X, Y, C;	\text coordinates (X,Y) and background color (C)
int	AC;		\attribute (high byte) and character (low byte)
begin
\Set cursor position
CallInt($10, $0200, $0000, 0, Y<<8!X);

\Get attribute and character at cursor
AC:= CallInt($10, $0800, $0000);

\Change background color and write attribute back at cursor
CallInt($10,   $0900 ! AC&$00FF,   AC>>8 & $0F ! C<<4,   1);
end;	\SetBkgndColor



proc	Backlight(X0, Y0, X1, Y1, C); \Hilight but don't change foreground color
int	X0, Y0, X1, Y1, C;
int	X, Y;
begin
for Y:= Y0, Y1 do
    for X:= X0, X1 do
	SetBkgndColor(X, Y, C);
end;	\Backlight



proc	SetColorReg(N, RGB);	\Set color register to values in RGB
int	N;	\register number (0..255)
char	RGB;	\array of red, green, and blue values (0..63)
begin
port($3C8):= N;			\select color register to write
port($3C9):= RGB(0);
port($3C9):= RGB(1);
port($3C9):= RGB(2);
end;	\SetColorReg



proc	GetColorReg(N, RGB);	\Return color register values in array RGB
int	N;	\register number (0..255)
char	RGB;	\array of red, green, and blue values (0..63)
begin
port($3C7):= N;			\select color register to read
RGB(0):= port($3C9);
RGB(1):= port($3C9);
RGB(2):= port($3C9);
end;	\GetColorReg

\------------------------------- MOUSE ROUTINES --------------------------------

func	OpenMouse;		\Initializes mouse; returns 'false' if it fails
begin				\Pointer is set to center of screen but hidden
CallInt($21, $3533);		\Make sure mouse vector $33 points to something
if ExtWord(CpuReg(1))=0 & ExtWord(CpuReg(11))=0 then return false;
CallInt($33, $21);		\reset mouse for Windows
CallInt($33, $0B);		\read motion counters to reset them for Windows
return CallInt($33, $0000);	\h/w reset mouse and return status
end;	\OpenMouse



func	MouseButton;		\Return 'true' if a mouse button is down
begin
CallInt($33, $0003);
return CpuReg(1) & $03;
end;	\GetMouseButton

\------------------------------ GRAPHICS ROUTINES ------------------------------

proc	EraseImage;		\Erase the Image array
int	Im, I;
begin
Im:= Image;			\erase 4 (or 2) bytes at a time, for speed
for I:= 0, ImW*ImH/IntSize-1 do Im(I):= 0;
end;	\EraseImage



proc	Erase(Color);		\Fill 320x200 graphic screen with given color
int	Color;
int	X, Y;
begin
for Y:= 0, 200-1 do
    for X:= 0, 320-1 do
	Point(X, Y, Color);
end;	\Erase



proc	FillScreen(Col);	\Fill 320x200 graphic screen with color
int	Col;
begin
asm	{.186
	push	di
	cld
	xor	di, di
	mov	cx, 32000
	push	0a000h
	pop	es
	mov	al, Col
	mov	ah, al
	rep stosw		;es:[di++], cx--
	pop	di
	.8086}
end;	\FillScreen



int	PenX, PenY;		\pen coordinates (pixels) for drawing onto Image
char	Image(ImW*ImH);		\buffer for setting up copy of screen image



proc	MovePen(X, Y);		\Position drawing pen
int	X, Y;
[PenX:= X;  PenY:= Y];



proc	DrawLine(X, Y, C); \Draw line on Image from PenX,PenY to X,Y in color C
int	X, Y, C;	\coordinates of end-of-line, color
int	DX, DY,		\delta X, delta Y
	IX, IY,		\plotting increments along X and Y axes
	SY,		\flag: step along Y direction
	D,		\decision variable: move straight or diagonally
	E, NE,		\East, Northeast (straight or diagonal)
	T, I;		\scratch
[DX:= X - PenX;				\delta X
if DX >= 0 then				\if positive then increment
	IX:= 1
else	[DX:= -DX;   IX:= -1];		\else absolute value and decrement

DY:= Y - PenY;				\same for Y direction
if DY >= 0 then
	IY:= 1
else	[DY:= -DY;   IY:= -1];

if DX >= DY then			\step along X direction
	SY:= false
else	[SY:= true;			\step along Y direction
	T:= DX;   DX:= DY;   DY:= T;	\swap DX and DY
	];

D:= 2*DY - DX;				\set up decision variable stuff
E:= 2*DY;
NE:= 2*(DY-DX);

if PenX>=0 & PenX<ImW & PenY>=0 & PenY<ImH then	\clip
	Image(PenX + PenY*ImW):= C;	\plot initial point
for I:= 1, DX do
	[if D < 0 then
		[D:= D + E;
		if SY then PenY:= PenY + IY
		      else PenX:= PenX + IX;
		]
	else	[D:= D + NE;		\move diagonally
		PenX:= PenX + IX;
		PenY:= PenY + IY;
		];
	if PenX>=0 & PenX<ImW & PenY>=0 & PenY<ImH then	\clip
		Image(PenX + PenY*ImW):= C;
	];
];	\DrawLine



proc	ShowLine(X1, Y1, X2, Y2, Type);	\Display line clipped to a window
\ using the Cohen-Sutherland clipping algorithm.
\ Ref: Fundamentals of Interactive Computer Graphics, J.D.Foley
\ & A. Van Dam, p149.
int	X1, Y1, X2, Y2, Type;
int	Code1, Code2;		\4-bit boolean array
int	TR, IX, IY;


	func	OutCode(X, Y);
	int	X, Y;
	int	Code;
	begin
	Code:= if Y>YHi then 1
	else if Y<YLo then 2
	else 0;
	if X > XHi then Code:=Code!4
	else if X < XLo then Code:=Code!8;
	return Code;
	end;	\OutCode


	proc	Xlat(X, Y);	\Translate to hi-res coordinates
	int	X, Y;
	begin
	IX:= (X-XLo) *Scale /XTick;
	IY:= 191 - (Y-YLo)*Scale/YTick;
	end;	\Xlat


begin	\ShowLine
loop	begin
	Code1:= OutCode(X1, Y1);   Code2:= OutCode(X2, Y2);
	if Code1 & Code2 then quit;		\trivially reject
	if (Code1 ! Code2) = 0 then 		\trivially accept
		begin
		Xlat(X1, Y1);
		Move(IX, IY);
		Xlat(X2, Y2);
		Line(IX, IY, Type);
		quit;
		end;
	\Subdivide the line, since at most one end point is inside the window.
	\ First, if point X1,Y1 is inside the window, exchange end points
	\ (and their OutCodes) to guarantee that point X1,Y1 is outside
	\ the window.
	if Code1 = 0 then			\exchange
		begin
		TR:= X1;   X1:= X2;   X2:= TR;
		TR:= Y1;   Y1:= Y2;   Y2:= TR;
		Code1:= Code2;
		end;
	\Now perform a subdivision, move point X1,Y1 to the intersection
	\ point (X,Y); use the formulas:	Y = Y1 + SLOPE *(X - X1)
	\					X = X1 + (1/SLOPE) *(Y - Y1)
	case of
	    Code1 & 1:			\divide line at top of window
		[X1:= X1 + Fix(Float(X2 - X1) *Float(YHi - Y1) /Float(Y2 - Y1));
		Y1:= YHi];
	    Code1 & 2:			\divide line at bottom of window
		[X1:= X1 + Fix(Float(X2 - X1) *Float(YLo - Y1) /Float(Y2 - Y1));
		Y1:= YLo];
	    Code1 & 4:			\divide line at right edge of window
		[Y1:= Y1 + Fix(Float(Y2 - Y1) *Float(XHi - X1) /Float(X2 - X1));
		X1:= XHi];
	    Code1 & 8:			\divide line at left edge of window
		[Y1:= Y1 + Fix(Float(Y2 - Y1) *Float(XLo - X1) /Float(X2 - X1));
		X1:= XLo]
	other	[];
	end;
end;	\ShowLine



proc	DrawTriangle(Tri);	\Draw a filled triangle onto Image array
int	Tri;			\array of 3 points(x,y) and one 8-bit color
int	X0,Y0, X1,Y1, X2,Y2, Color;
int	Y, T;
int	Top, Bot,		\top and bottom Y coordinates
	YImW;			\Y coordinate times Image width
int	Edge1, Edge2;		\X coordinates along edges of triangles
real	REdge1, REdge2, Slope1, Slope2;
char	Img;

	proc	DrawHalf;	\(optimized procedure with no local variables)
	begin
	if Bot > ImH then Bot:= ImH-1;			\clip to bottom of Image
	YImW:= Top*ImW;
	for Y:= Top, Bot-1 do				\from top to bottom...
		begin
		if Y >= 0 then				\clip to top of Image
			begin
			Edge1:= fix(REdge1);  Edge2:= fix(REdge2);
			if Edge1 > Edge2 then		\order left to right
				[T:= Edge1;  Edge1:= Edge2;  Edge2:= T];
			if Edge1 < 0 then Edge1:= 0;	\clip to left side
			if Edge2 > ImW then Edge2:= ImW;\clip to right side
			for Img:= Image+Edge1+YImW, Image+Edge2-1+YImW do
				Img(0):= Color;		\fill scanline
			end;
		REdge1:= REdge1 + Slope1;		\adj X coords according
		REdge2:= REdge2 + Slope2;		\ to slopes
		YImW:= YImW + ImW;
		end;
	end;	\DrawHalf

begin	\DrawTriangle
X0:= Tri(0,0);			\get vertices
X1:= Tri(1,0);
X2:= Tri(2,0);

if X0>=ImW & X1>=ImW & X2>=ImW then return;	\clip entire triangle
if X0<0 & X1<0 & X2<0 then return;

Y0:= Tri(0,1);
Y1:= Tri(1,1);
Y2:= Tri(2,1);

if Y0 > Y1 then			\sort vertices from top to bottom (Y0<=Y1<=Y2)
	begin
	T:= Y1;  Y1:= Y0;  Y0:= T;
	T:= X1;  X1:= X0;  X0:= T;
	end;
if Y0 > Y2 then
	begin
	T:= Y2;  Y2:= Y0;  Y0:= T;
	T:= X2;  X2:= X0;  X0:= T;
	end;
if Y0 >= ImH then return;	\clip
if Y1 > Y2 then
	begin
	T:= Y2;  Y2:= Y1;  Y1:= T;
	T:= X2;  X2:= X1;  X1:= T;
	end;
if Y2 < 0 then return;		\clip

if Y1 # Y0 then Slope1:= float(X1-X0) / float(Y1-Y0);
if Y2 # Y0 then Slope2:= float(X2-X0) / float(Y2-Y0);
REdge1:= float(X0);  REdge2:= REdge1;
Top:= Y0;  Bot:= Y1;
Color:= Tri(3);
DrawHalf;

if Y2 # Y1 then Slope1:= float(X2-X1) / float(Y2-Y1);
REdge1:= float(X1);
Top:= Y1;  Bot:= Y2;
DrawHalf;
end;	\DrawTriangle



proc	DrawQuad(Quad);		\Draw a filled quadrilateral onto Image array
int	Quad;			\array of 4 points(x,y) and one 8-bit color
int	Tri(4,2);
begin
Tri(3):= Quad(4);		\color

Tri(0,0):= Quad(0,0);
Tri(0,1):= Quad(0,1);
Tri(1,0):= Quad(1,0);
Tri(1,1):= Quad(1,1);
Tri(2,0):= Quad(2,0);
Tri(2,1):= Quad(2,1);
DrawTriangle(Tri);

Tri(0,0):= Quad(0,0);
Tri(0,1):= Quad(0,1);
Tri(1,0):= Quad(2,0);
Tri(1,1):= Quad(2,1);
Tri(2,0):= Quad(3,0);
Tri(2,1):= Quad(3,1);
DrawTriangle(Tri);
end;	\DrawQuad



proc	Rectangle(X0, Y0, X1, Y1, C, F);	\Display a rectangle
int	X0, Y0, X1, Y1, C, F;
int	I, J, T;
begin
if not F then				\if not filled, display hollow rectangle
	begin
	Move(X0, Y0);
	Line(X1, Y0, C);
	Line(X1, Y1, C);
	Line(X0, Y1, C);
	Line(X0, Y0, C);
	return
	end;

\Else display a filled rectangle:
if X1 < X0 then [T:=X1;   X1:=X0;   X0:=T];
if Y1 < Y0 then [T:=Y1;   Y1:=Y0;   Y0:=T];

if (Y1-Y0) > 8*(X1-X0) then		\if tall and skinny
	for I:= X0, X1 do		\make vertical lines
		[Move(I, Y0);   Line(I, Y1, C)]
else	for J:= Y0, Y1 do		\make fast horizontal lines
		[Move(X0, J);   Line(X1, J, C)];
end;	\Rectangle



proc	ShowRectangle(X, Y, W, H, C); \Display a rectangle
int	X, Y, W, H,	\upper-left corner coordinates, width, height (pixels)
	C;		\color
begin
Move(X, Y);
Line(X+W-1, Y, C);
Line(X+W-1, Y+H-1, C);
Line(X, Y+H-1, C);
Line(X, Y, C);
end;	\ShowRectangle



proc	ShowFilledRectangle(X0, Y0, W, H, C);	\Display a filled rectangle
int	X0, Y0, W, H,	\upper-left corner coordinates, width, height (pixels)
	C;		\color
int	Y;
begin
for Y:= Y0, Y0+H-1 do
	[Move(X0, Y);   Line(X0+W-1, Y, C)];
end;	\ShowFilledRectangle



proc	ShowFilledCircle(X0, Y0, R, C);	\Display a filled circle
int	X0, Y0,		\coordinates of center (pixels)
	R,		\radius (pixels)
	C;		\color
int	X, Y,		\working coordinates (pixels)
	R2, Y2;		\squared values
begin
R2:= R*R;
for Y:= -R, R do
	begin
	Y2:= Y*Y;
	for X:= -R, R do
		if X*X + Y2 <= R2 then Point(X+X0, Y+Y0, C);
	end;
end;	\ShowFilledCircle



proc	Circle(X0, Y0, Radius, Color, Segs);	\Display a circle
int	X0, Y0,	\coordinates of center
	Radius,	\radius in (pixels)
	Color,	\line color
	Segs;
int	X, Y, E, U, V;

	proc	PlotOctants;
	begin						\Segment
	if Segs & $01 then Point(X0+Y, Y0+X, Color);	\ 0
	if Segs & $02 then Point(X0+X, Y0+Y, Color);	\ 1
	if Segs & $04 then Point(X0-X, Y0+Y, Color);	\ 2
	if Segs & $08 then Point(X0-Y, Y0+X, Color);	\ 3
	if Segs & $10 then Point(X0-Y, Y0-X, Color);	\ 4
	if Segs & $20 then Point(X0-X, Y0-Y, Color);	\ 5
	if Segs & $40 then Point(X0+X, Y0-Y, Color);	\ 6
	if Segs & $80 then Point(X0+Y, Y0-X, Color);	\ 7
	end;	\PlotOctants

begin	\Circle
X:= 0;
Y:= Radius;
U:= 1;
V:= 1 -Radius -Radius;
E:= 1 -Radius;
while X < Y do
	begin
	PlotOctants;
	if E < 0 then
		[U:= U+2;   V:= V+2;   E:= E+U]
	else	[U:= U+2;   V:= V+4;   E:= E+V;   Y:= Y-1];
	X:= X+1;
	end;
if X = Y then PlotOctants;
end;	\Circle



proc	ShowCircle(X0, Y0, Radius, Color, Filled);	\Display a circle
int	X0, Y0,		\coordinates of center
	Radius,		\radius in (pixels)
	Color,		\line color
	Filled;		\filled circle vs. outlined
int	X, Y, E, U, V;


	proc	PlotOctants;
	begin					\Segment
	if Filled then
		begin
		Move(X0-Y, Y0+X);		\ 3
		Line(X0+Y, Y0+X, Color);	\ 0
		Move(X0+Y, Y0-X);		\ 7
		Line(X0-Y, Y0-X, Color);	\ 4
		Move(X0+X, Y0+Y);		\ 1
		Line(X0-X, Y0+Y, Color);	\ 2
		Move(X0+X, Y0-Y);		\ 6
		Line(X0-X, Y0-Y, Color);	\ 5
		end
	else	begin
		Point(X0+Y, Y0+X, Color);	\ 0
		Point(X0+X, Y0+Y, Color);	\ 1
		Point(X0-X, Y0+Y, Color);	\ 2
		Point(X0-Y, Y0+X, Color);	\ 3
		Point(X0-Y, Y0-X, Color);	\ 4
		Point(X0-X, Y0-Y, Color);	\ 5
		Point(X0+X, Y0-Y, Color);	\ 6
		Point(X0+Y, Y0-X, Color);	\ 7
		end;
	end;	\PlotOctants


begin	\ShowCircle
X:= 0;
Y:= Radius;
U:= 1;
V:= 1 -Radius -Radius;
E:= 1 -Radius;
while X < Y do
	begin
	PlotOctants;
	if E < 0 then
		[U:= U+2;   V:= V+2;   E:= E+U]
	else	[U:= U+2;   V:= V+4;   E:= E+V;   Y:= Y-1];
	X:= X+1;
	end;
if X = Y then PlotOctants;
end;	\ShowCircle



proc	ShowEllipse(X0, Y0, A, B, Color, Filled);	\Display an ellipse
int	X0, Y0,		\coordinates of center
	A, B,		\half width and half height (pixels)
	Color,
	Filled;		\filled ellipse vs. outlined
int	X, Y,
	T1, T2, T3, T4, T5, T6, T7, T8, T9,
	D1, D2;

	proc	Set4Pts(X, Y, Xc, Yc, Color);
	int	X, Y, Xc, Yc, Color;
	begin
	if Filled then
		begin
		Move(Xc-X, Yc+Y);
		Line(Xc+X, Yc+Y, Color);
		Move(Xc-X, Yc-Y);
		Line(Xc+X, Yc-Y, Color);
		end
	else	begin
		Point(Xc+X, Yc+Y, Color);
		Point(Xc+X, Yc-Y, Color);
		Point(Xc-X, Yc+Y, Color);
		Point(Xc-X, Yc-Y, Color);
		end;
	end;	\Set4Pts

begin	\ShowEllipse
X:= A;
Y:= 0;
T1:= A*A;
T2:= T1+T1;
T3:= T2+T2;
T4:= B*B;
T5:= T4+T4;
T6:= T5+T5;
T7:= A*T5;
T8:= T7+T7;
T9:= 0;

D1:= T2 - T7 + T4/2;
D2:= T1/2 - T8 + T5;

\Do region 1 of quadrant
while D2 < 0 do
	begin
	Set4Pts(X, Y, X0, Y0, Color);
	Y:= Y+1;
	T9:= T9+T3;
	if D1 < 0 then
		begin
		D1:= D1 + T9 + T2;
		D2:= D2 + T9;
		end
	else	begin
		X:= X-1;
		T8:= T8 - T6;
		D1:= D1 + T9 + T2 - T8;
		D2:= D2 + T9 + T5 - T8;
		end;
	end;

\Do region 2 of quadrant
repeat	Set4Pts(X, Y, X0, Y0, Color);
	X:= X-1;
	T8:= T8-T6;
	if D2 < 0 then
		begin
		Y:= Y+1;
		T9:= T9 + T3;
		D2:= D2 + T5 + T9 - T8;
		end
	else	D2:= D2 + T5 - T8;
until	X < 0;
end;	\ShowEllipse



proc	XFlood(X, Y, C, C0);	\Fill an area of color C0 with color C
\WARNING: even a 150 pixel area will overflow the stack in 16-bit XPL
int	X, Y,	\seed coordinate (where to start)
	C, C0;	\color to fill with and color to replace
begin
if ReadPix(X, Y) = C0 then
	begin
	Point(X, Y, C);
	XFlood(X+1, Y, C, C0);
	XFlood(X, Y+1, C, C0);
	XFlood(X-1, Y, C, C0);
	XFlood(X, Y-1, C, C0);
	end;
end;	\XFlood



proc	Flood(X, Y, C, C0);	\Fill an area of color C0 with color C
int	X, Y,	\seed coordinate (where to start)
	C, C0;	\color to fill with and color to replace
def	S=8000;	\size of queue (must be an even number)
int	Q(S),	\queue (FIFO)
	F, E;	\fill and empty indexes

	proc	EnQ(X, Y);	\Enqueue coordinate
	int	X, Y;
	begin
	Q(F):= X;
	F:= F+1;
	Q(F):= Y;
	F:= F+1;
	if F >= S then F:= 0;
	end;	\EnQ

	proc	DeQ;		\Dequeue coordinate
	begin
	X:= Q(E);
	E:= E+1;
	Y:= Q(E);
	E:= E+1;
	if E >= S then E:= 0;
	end;	\DeQ

begin	\Flood
F:= 0;   E:= 0;
EnQ(X, Y);
while E # F do
	begin
	DeQ;
	if ReadPix(X, Y) = C0 then
		begin
		Point(X, Y, C);
		EnQ(X+1, Y);	\enqueue adjacent pixels
		EnQ(X-1, Y);
		EnQ(X, Y+1);
		EnQ(X, Y-1);
		end;
	end;
end;	\Flood



proc	GrabSprite(X0, Y0, W, H, Spr);	\Load sprite data array from Screen
int	X0, Y0,		\coordinates in Screen to get sprite from
	W, H;		\width and height of sprite (pixels)
char	Spr;		\address of array in which to load sprite data
int	X, Y;
begin
Spr(0):= W;
Spr(1):= H;
Spr:= Spr+2;
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	Spr(0):= ReadPix(X, Y);
	Spr:= Spr+1;
	end;
end;	\GrabSprite



proc	ShowSprite(X0, Y0, Spr); \Display a sprite
int	X0, Y0;		\coordinates of upper-left corner (pixels)
char	Spr;		\address of sprite data
int	X, Y, I, C, W, H;
begin
W:= Spr(0);		\get width and height (in pixels)
H:= Spr(1);
I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	C:= Spr(I);		\get pixel's color
	if C then		\background (0) is transparent
		Point(X, Y, C);
	I:= I+1;
	end;
end;	\ShowSprite

\------------------------------ VECTOR ROUTINES --------------------------------

proc	VectAdd(V1, V2, V3);	\Add two 3D vectors
real	V1, V2, V3;		\V3:= V1 + V2
begin
V3(0):= V1(0) + V2(0);
V3(1):= V1(1) + V2(1);
V3(2):= V1(2) + V2(2);
end;	\VectAdd



proc	VectSub(V1, V2, V3);	\Subtract two 3D vectors
real	V1, V2, V3;		\V3:= V1 - V2
begin
V3(0):= V1(0) - V2(0);
V3(1):= V1(1) - V2(1);
V3(2):= V1(2) - V2(2);
end;	\VectSub



proc	VectMul(V1, S, V2);	\Multiply 3D vector by a scaler
real	V1, S, V2;		\V2:= V1 * S
begin
V2(0):= V1(0) * S;
V2(1):= V1(1) * S;
V2(2):= V1(2) * S;
end;	\VectMul



func real DotProd(V1, V2);	\Return the dot product of two 3D vectors
real	V1, V2;			\V1  V2
return V1(0)*V2(0) + V1(1)*V2(1) + V1(2)*V2(2);



proc	CrossProd(V1, V2, V3);	\Calculate the cross product of two 3D vectors
real	V1, V2, V3;		\V3:= V1 x V2
begin
V3(0):= V1(1)*V2(2) - V1(2)*V2(1);
V3(1):= V1(2)*V2(0) - V1(0)*V2(2);
V3(2):= V1(0)*V2(1) - V1(1)*V2(0);
end;	\CrossProd



func real VectMag(V);		\Return the magnitude (length) of a 3D vector
real	V;
return sqrt(V(0)*V(0) + V(1)*V(1) + V(2)*V(2));



proc	Unitize(V);		\Convert a 3D vector to its unit vector
real	V;			\vector to convert
real	M;			\magnitude of vector
begin
M:= VectMag(V);
V(0):= V(0) / M;
V(1):= V(1) / M;
V(2):= V(2) / M;
end;	\Unitize



proc	Rotate(V, W, P, R);	\3D rotate vector V
real	V,		\3D vector is 3-element array with X, Y and Z components
	W, P, R;	\yaW, Pitch and Roll (radians)
real	SW, SP, SR, CW, CP, CR, T;
begin
SW:= Sin(W);  SP:= Sin(P);  SR:= Sin(R);
CW:= Cos(W);  CP:= Cos(P);  CR:= Cos(R);

\Rotate about Y axis (yaW):
T:= V(0)*CW + V(2)*SW;		\		       	 
V(2):= V(2)*CW - V(0)*SW;	\		      CW  0  -SW
V(0):= T;			\ [X Y Z]:= [X Y Z] * 0   1   0 
				\		      SW  0   CW
				\		       	 
\Rotate about X axis (Pitch):
T:= V(1)*CP - V(2)*SP;		\		       	 
V(2):= V(2)*CP + V(1)*SP;	\		      1   0   0 
V(1):= T;			\ [X Y Z]:= [X Y Z] * 0   CP  SP
				\		      0  -SP  CP
				\		       	 
\Rotate about Z axis (Roll):
T:= V(0)*CR - V(1)*SR;		\		       	 
V(1):= V(1)*CR + V(0)*SR;	\		       CR  SR	0
V(0):= T;			\ [X Y Z]:= [X Y Z] * -SR  CR	0
				\		       0   0	1
end;	\Rotate					       	 



proc	Unrotate(V, W, P, R);	\3D unrotate vector V
real	V,		\3D vector is 3-element array with X, Y and Z components
	W, P, R;	\yaW, Pitch and Roll (radians)
real	SW, SP, SR, CW, CP, CR, T;
begin
SW:= Sin(-W);  SP:= Sin(-P);  SR:= Sin(-R);
CW:= Cos(W);  CP:= Cos(P);  CR:= Cos(R);		\Cos(-x) = Cos(x)

\Rotate about Z axis (Roll):
T:= V(0)*CR - V(1)*SR;		\		       	 
V(1):= V(1)*CR + V(0)*SR;	\		       CR  SR	0
V(0):= T;			\ [X Y Z]:= [X Y Z] * -SR  CR	0
				\		       0   0	1
				\		       	 
\Rotate about X axis (Pitch):
T:= V(1)*CP - V(2)*SP;		\		       	 
V(2):= V(2)*CP + V(1)*SP;	\		      1   0   0 
V(1):= T;			\ [X Y Z]:= [X Y Z] * 0   CP  SP
				\		      0  -SP  CP
				\		       	 
\Rotate about Y axis (yaW):
T:= V(0)*CW + V(2)*SW;		\		       	 
V(2):= V(2)*CW - V(0)*SW;	\		      CW  0  -SW
V(0):= T;			\ [X Y Z]:= [X Y Z] * 0   1   0 
				\		      SW  0   CW
end;	\Unrotate				       	 



proc	Mat4x1Mul(M, V);	\Multiply matrix M times column vector V
real	M,	\4x4 matrix	 [M] * [V] -> [V]
	V;	\column vector
real	W(4);	\working copy of column vector
int	R;	\row
begin
for R:= 0, 4-1 do
	W(R):= M(R,0)*V(0) + M(R,1)*V(1) + M(R,2)*V(2) + M(R,3)*V(3);
for R:= 0, 4-1 do V(R):= W(R);
end;	\Mat4x1Mul



proc	Mat4x4Mul(M, N);	\Multiply matrix M times matrix N
real	M, N;	\4x4 matrices	 [M] * [N] -> [N]
real	W(4,4);	\working copy of matrix N
int	C;	\column
begin
for C:= 0, 4-1 do
	begin
	W(0,C):= M(0,0)*N(0,C) + M(0,1)*N(1,C) + M(0,2)*N(2,C) + M(0,3)*N(3,C);
	W(1,C):= M(1,0)*N(0,C) + M(1,1)*N(1,C) + M(1,2)*N(2,C) + M(1,3)*N(3,C);
	W(2,C):= M(2,0)*N(0,C) + M(2,1)*N(1,C) + M(2,2)*N(2,C) + M(2,3)*N(3,C);
	W(3,C):= M(3,0)*N(0,C) + M(3,1)*N(1,C) + M(3,2)*N(2,C) + M(3,3)*N(3,C);
	end;
for C:= 0, 4-1 do
	begin
	N(0,C):= W(0,C);
	N(1,C):= W(1,C);
	N(2,C):= W(2,C);
	N(3,C):= W(3,C);
	end;
end;	\Mat4x4Mul

\------------------------------- SOUND ROUTINES --------------------------------

proc	Beep;			\A not-too-obnoxious beep
begin
Sound(false, 1, 1000);		\synchronize with system timer to make tone a
Sound(true, 1, 3000);		\ consistent duration and a consistent sound.
end;	\Beep



proc	Snd(Freq);		\Like Sound intrinsic but it doesn't wait
int	Freq;
begin
POut(PIn($61, 0) ! $03, $61, 0);
POut($B6, $43, 0);		\value, port, 0=byte size
POut(Freq, $42, 0);
POut(Freq>>8, $42, 0);
end;	\Snd



proc	Noise(Vol, Cy, Per);	\Emit sound on beeper speaker (like on Apple)
int	Vol, Cy, Per;		\number of cycles, period in 10 us intervals
int	Cnt, I;
int	T0;
begin
\Each count is 838 ns. Multiplying by 12 gives the period in 10 us intervals.
Cnt:= 12*Per;
if Vol then
	begin
	port($61):= port($61) ! $03;		\enable speaker and timer
	port($43):= $B6;			\set channel 2 for mode 3

	port($42):= Cnt;
	port($42):= Cnt>>8;
	end;

\Delay approximately the same as the original Apple II Sound intrinsic
for I:= 1, Cy do
	begin
	T0:= GetTime;
	repeat until ((GetTime-T0) & $FFFF) >= Cnt;
	end;

port($61):= port($61) & ~$03;			\turn off speaker
end;	\Noise

\--------------------------- MISCELLANEOUS ROUTINES ----------------------------

proc	Sort(A, N);		\Sort array in ascending order
int	A,			\address of array
	N;			\number of elements in array (size)
int	I, J, Gap, JG, T;
begin				\Shell sort
Gap:= N>>1;
while Gap > 0 do
	begin
	for I:= Gap, N-1 do
		begin
		J:= I - Gap;
		loop	begin
			JG:= J + Gap;
			if A(J) <= A(JG) then quit;
			T:= A(J);   A(J):= A(JG);   A(JG):= T;	\swap elements
			J:= J - Gap;
			if J < 0 then quit;
			end;
		end;
	Gap:= Gap>>1;
	end;
end;	\Sort



proc	QSort(Array, Num, Size); \Quicksort Array into ascending order
int	Array,		\address of array to sort
	Num,		\number of elements in the array
	Size;		\size (in bytes) of each element
int	I, J, Mid, Temp;
begin
I:= 0;
J:= Num-1;
Mid:= Array(J>>1);
while I <= J do
	begin
	while Array(I) < Mid do I:= I+1;
	while Array(J) > Mid do J:= J-1;
	if I <= J then
		begin
		Temp:= Array(I);  Array(I):= Array(J);  Array(J):= Temp;
		I:= I+1;
		J:= J-1;
		end;
	end;
if I < Num-1 then QSort(Array+I*Size, Num-I, Size);
if J > 0 then QSort(Array, J+1, Size);
end;	\QSort

\-------------------------------------------------------------------------------

func	Lookup(Class);		\Lookup current identifier and return its index
\Other inputs: Ident, Hash;
\Outputs: Index of identifier. Returns -1 if not found.
int	Class;		\class of symbol (Lab, Mn, Res)
int	I, K, Inx;
begin
Inx:= HashTbl(Hash);
loop	begin
	if Inx= -1 then quit;				\not found
	I:= 0;   K:= Inx;
	while Ident(I)=SymTbl(K) & I<SigChar do
		[I:= I+1;   K:= K+SymSize];
	if I=SigChar & SymClass(Inx)=Class then quit;	\found
	Inx:= SymNext(Inx);	\follow the linkage pointers
	end;
return Inx;
end;	\Lookup



proc	Insert(Class, Val, Type);	\Insert identifier into symbol table
\**** ???? reverse insertion order for maximum efficiency ????
\Other inputs: Ident, Hash;
int	Class, Val, Type;
int	I, Inx, K, SPass;
begin
Inx:= Lookup(Class);
if Inx # -1 then
	begin
	SPass:= Pass;   Pass:= 3;	\kludge to report error on pass 1
	Error("LABEL ALREADY USED");
	Pass:= SPass;
	end;
if SymNum>=SymSize then
	begin
	SPass:= Pass;   Pass:= 3;	\kludge to report error on pass 1
	Error("SYMBOL TABLE OVERFLOW");
	Pass:= SPass;
	SymNum:= SymSize-1;
	end;
K:= SymNum;
for I:= 0, SigChar-1 do
	[SymTbl(K):= Ident(I);   K:= K+SymSize];

SymClass(SymNum):= Class;
SymVal(SymNum):= Val;
SymType(SymNum):= Type;

SymNext(SymNum):= HashTbl(Hash);	\link back
HashTbl(Hash):= SymNum;

SymNum:= SymNum+1;
end;	\Insert



char	FileName($7F+4); 	\for file name on command line (if any)
def	CR = $0D;


\MSB-terminated string version:
proc	GetCmdLine(Str, Ext);	\Read file name from command line
char	Str;	\address of string in which to return the file name
char	Ext;	\default extension (if not typed on command line); e.g: ".TXT"
int	I, J, Ch, HaveExt;
begin
Blit(CpuReg(11), $81, CpuReg(12), Str, $7F);	\copy command line into Str
I:= 0;
J:= 0;
HaveExt:= false;
loop	begin
	Ch:= Str(I);
	if Ch = CR then quit;
	if Ch = ^. then HaveExt:= true;
	I:= I+1;
	if Ch # ^  then			\strip out space characters
		[Str(J):= Ch;  J:= J+1];
	end;
if HaveExt then
	Str(J-1):= Str(J-1) ! $80	\terminate string
else	for I:= 0, 3 do Str(I+J):= Ext(I); \add extension if not typed in
end;	\GetCmdLine


\null-terminated string version:
proc	GetCmdLine(Str, Ext);	\Read file name from command line
char	Str;	\address of string in which to return the file name
char	Ext;	\default extension (if not typed on command line); e.g: ".txt"
int	I, J, Ch, HaveExt;
begin
Blit(CpuReg(11), $81, CpuReg(12), Str, $7F);	\copy command line into Str
I:= 0;
J:= 0;
HaveExt:= false;
loop	begin
	Ch:= Str(I);
	if Ch = CR then quit;
	if Ch = ^. then HaveExt:= true;
	I:= I+1;
	if Ch # ^  then			\strip out space characters
		[Str(J):= Ch;  J:= J+1];
	end;
if HaveExt then
	Str(J):= 0			\terminate string
else	for I:= 0, 4 do Str(I+J):= Ext(I); \add extension if not typed in
end;	\GetCmdLine



proc	ShowEnvironment;	\Display Environment Block
int	CpuReg, I, Ch;
seg int	PspSeg(1);
seg char EnvSeg(1);
begin
\Get segment address where PSP starts
CpuReg:= GetReg;
PspSeg(0):= CpuReg(9);

\Get segment pointer to environment block
EnvSeg(0):= PspSeg(0, $2C>>1);	\divide by 2 to address words, not bytes

\Display ASCIIZ strings until "00"
I:= 0;
loop	begin
	loop	begin
		Ch:= EnvSeg(0,I);
		I:= I+1;
		if Ch = 0 then quit;
		ChOut(0, Ch);
		end;
	CrLf(0);
	Ch:= EnvSeg(0,I);
	I:= I+1;
	if Ch = 0 then quit;
	ChOut(0, Ch);
	end;
end;	\ShowEnvironment

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

proc	GetSBEnv;		\Get Sound Blaster's environment parameters
int	SBPort,
	SBDMA,
	SBIRQ;
seg int	PspSeg(1);
seg char EnvSeg(1);



func	GetEnv(Str);	\Find string (Str) in environment block (EnvSeg) and
char	Str;		\ return the index to the rest of the line in EnvSeg
int	Ch, I, J;
begin
Ch:= EnvSeg(0,0);			\get first character from environment
J:= 1;					\set index past first character
loop	begin				\for all the strings in the environment
	I:= 0;				\set Str index
	loop	begin			\for each char in a line
		if ToUpper(Ch) = ToUpper(Str(I)&$7F) then
			begin
			if Str(I) >= $80 then	\if match then return index
				return J;
			I:= I+1;
			Ch:= EnvSeg(0,J);
			J:= J+1;
			end
		else	begin		\eat rest of line
			while EnvSeg(0,J) # 0 do J:= J+1;
			J:= J+1;	\point past 0
			quit;
			end;
		end;
	Ch:= EnvSeg(0,J);
	J:= J+1;
	if Ch = 0 then quit;
	end;
return 0;		\Str not found
end;	\GetEnv



begin	\GetSBEnv
\Get segment address where PSP starts
CpuReg:= GetReg;
PspSeg(0):= CpuReg(9);

\Get segment pointer to environment block
EnvSeg(0):= PspSeg(0, $2C>>1);	\divide by 2 to address words, not bytes

SBPort:= $220;	\defaults
SBDMA:= 1;
SBIRQ:= 5;
I:= GetEnv("BLASTER");
if I # 0 then
  repeat begin				\BLASTER=A220 I7 D1 T2
	Ch:= EnvSeg(0,I);
	I:= I+1;
	case ToUpper(Ch) of
	  ^A:	begin
		SBPort:= 0;
		Ch:= EnvSeg(0,I);
		I:= I+1;
		while Ch>=^0 & Ch<=^9 do	\only 220..260 allowed
			begin
			SBPort:= SBPort<<4 ! Ch-^0;
			Ch:= EnvSeg(0,I);
			I:= I+1;
			end;
		end;
	  ^I:	begin
		Ch:= EnvSeg(0,I);
		I:= I+1;
		SBIRQ:= Ch-^0;
		end;
	  ^D:	begin
		Ch:= EnvSeg(0,I);
		I:= I+1;
		SBDMA:= Ch-^0;
		end
	other	[];
	end;
until Ch = 0;
HexOut(0, SBPort); CrLf(0);
HexOut(0, SBIRQ); CrLf(0);
HexOut(0, SBDMA); CrLf(0);
end;	\GetSBEnv

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

proc	EditWindow;		\Type text into a window and edit it
\ Edit using arrow keys etc. The window dimensions can be smaller than the area
\ of text (EdBox) that is edited, thus the text scrolls inside the window.

def	WinWidth=20, WinHeight=4;	\window dimensions (in characters)
def	WX0=30, WY0=8,			\location on screen of upper-left corner
	WX1=WX0+WinWidth-1, WY1=WY0+WinHeight-1;	\lower-right corner
def	BoxWidth=40, BoxHeight=6;	\edit box dimensions (characters)

char	EdBox(BoxHeight, BoxWidth);	\edit text buffer



proc	PrintCheck;		\Display all the information in EdBox
int	X, Y, Ch;
begin
Cursor(0, 17);
for Y:= 0, BoxHeight-1 do
	begin
	for X:= 0, BoxWidth-1 do
		begin
		Ch:= EdBox(Y, X);
		ChOut(0, Ch);
		end;
	CrLf(0);
	end;
end;	\PrintCheck



proc	EditText;	\Edit text in EdBox
int	Ch,
	Cx, Cy,		\cursor position in text box (EdBox)
	Wx, Wy,		\location in text box that is displayed in the
			\ upper-left corner of the window (LCD)
	InsMode,	\flag: insert mode (versus overwrite mode)
	X, Y;		\scratch coordinates
begin
Cx:= 0;   Cy:= 0;	\initialize cursor to upper-left corner
Wx:= 0;   Wy:= 0;
InsMode:= true;

loop	begin
	\Display contents of window
	for Y:= Wy, Wy+WinHeight-1 do
		begin
		for X:= Wx, Wx+WinWidth-1 do
			begin
			Cursor(X-Wx+WX0, Y-Wy+WY0);
			Ch:= EdBox(Y, X);
			ChOut(6, Ch);
			end;
		end;

	CallInt($10, $0100, 0, if InsMode then $0607 else $0007); \ _ or 
	Cursor(Cx-Wx+WX0, Cy-Wy+WY0);	\show flashing cursor in correct place
	Ch:= GetKey;
	CallInt($10, $0100, 0, $2000);	\remove flashing cursor

	case Ch of
	 -UpArrow:	Cy:= Cy-1;
	 -DnArrow:	Cy:= Cy+1;
	 -LtArrow:	Cx:= Cx-1;
	 -RtArrow:	Cx:= Cx+1;
	 -PageUp:	Cy:= Cy - (WinHeight-1);
	 -PageDn:	Cy:= Cy + (WinHeight-1);
	 -Home:		Cx:= 0;
	 -End:		begin
			Cx:= 0;
			for X:= -(BoxWidth-1), 0 do
			    if EdBox(Cy, -X) # Sp then
			    	[Cx:= -X+1;   X:= 0];
			end;
	 -Insert:	InsMode:= not InsMode;
	 -Delete:	begin
			for X:= Cx, BoxWidth-2 do	\shift line left <-
				EdBox(Cy, X):= EdBox(Cy, X+1);
			EdBox(Cy, BoxWidth-1):= Sp;
			end;
	  BS:		begin
			if Cx > 0 then
				begin
				Cx:= Cx-1;
				if InsMode then
					begin
					for X:= Cx, BoxWidth-2 do  \shift <-
						EdBox(Cy, X):= EdBox(Cy, X+1);
					EdBox(Cy, BoxWidth-1):= Sp;
					end
				else	EdBox(Cy, Cx):= Sp;
				end;
			end;
	  CR:		[Cx:= 0;   Cy:= Cy+1];
	  Esc:		quit
	other		begin
			if InsMode then
			    for X:= -38, -Cx do		\shift line right ->
			    for X:= -(BoxWidth-2), -Cx do  \shift line right ->
				EdBox(Cy, -X+1):= EdBox(Cy, -X);
			EdBox(Cy, Cx):= Ch;
			Cx:= Cx+1;
			end;

	\Restrict cursor to edit box (EdBox) dimensions			
	if Cx >= BoxWidth then Cx:= BoxWidth-1;
	if Cx < 0 then Cx:= 0;
	if Cy >= BoxHeight then Cy:= BoxHeight-1;
	if Cy < 0 then Cy:= 0;

	\Drag window around with cursor
	if Cx < Wx then Wx:= Cx;
	if Cx >= Wx+WinWidth then Wx:= Cx-(WinWidth-1);
	if Cy < Wy then Wy:= Cy;
	if Cy >= Wy+WinHeight then Wy:= Cy-(WinHeight-1);
	end;	\loop
CallInt($10, $0100, 0, $0607);		\restore normal underline cursor
end;	\EditText



proc	Init;			\Initialize EdBox to all space characters
int	X, Y;
begin
ChOut(0, FF);
for Y:= 0, BoxHeight-1 do
    for X:= 0, BoxWidth-1 do
	EdBox(Y, X):= Sp;
end;	\Init



begin	\EditWindow
Init;
DrawBox(WX0-1, WY0-1, WX1+1, WY1+1, 0);
EditText;
PrintCheck;
end;	\EditWindow

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

real	Slope,		\slope of straight line from FitLine procedure
	Intercept;	\intercept of straight line from FitLine procedure

proc	FitLine(N, X, Y);	\Fit a straight line to N number of X,Y coords
\ and return its slope and intercept
\This uses Gauss's method of least squares
int	N;	\number of coordinates in arrays
real	X, Y;	\arrays of coordinate pairs
int	I;
real	Sum, XMean, YMean, Temp;
begin
Sum:= 0.0;
for I:= 0, N-1 do
	Sum:= Sum + X(I);
XMean:= Sum / Float(N);

Sum:= 0.0;
for I:= 0, N-1 do
	Sum:= Sum + Y(I);
YMean:= Sum / Float(N);

Sum:= 0.0;
for I:= 0, N-1 do
	Sum:= Sum + (X(I) - XMean)*(Y(I) - YMean);
Temp:= Sum;

Sum:= 0.0;
for I:= 0, N-1 do
	Sum:= Sum + (X(I) - XMean)*(X(I) - XMean);

Slope:= Temp / Sum;

Intercept:= YMean - Slope*XMean;
end;	\FitLine



begin	\Main
\Example:
FitLine(3, [2.0, 4.0, 6.0], [2.5, 4.5, 6.5]);
RlOut(0, Slope);
RlOut(0, Intercept);
CrLf(0);
end;	\Main

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

proc	ColumnOut(Dev, N);	\Output 6 columns of numbers
int	Dev;	\output device number
int	N;	\largest number to output
int	H, B, B0, I;
begin
H:= (N+5)/6;			\height of a column

for B0:= 0, H-1 do		\for each row
	begin
	B:= B0;			\value at start of row
	for I:= 1, 6 do		\for 6 columns
		begin
		if B < N then	\if it exists
			begin
			IntOut(Dev, B);
			ChOut(Dev, 9\Tab\);
			end;
		B:= B + H;	\next column
		end;
	CrLf(Dev);
	end;
end;	\ColumnOut



proc	TextIn(Dev, Str, N);	\Input a string ending with a line feed
int	Dev;	\device number to input from
char	Str;	\address of string to write to (Reserve N bytes)
int	N;	\maximum number of bytes allowed in string
int	I, Ch;
begin
I:= 0;
loop	begin
	if I >= N then quit;
	Ch:= ChIn(Dev);
	Str(I):= Ch;
	I:= I + 1;
	if Ch=LF ! Ch=EOF then quit;
	end;
if I > 0 then Str(I-1):= Str(I-1) ! $80
else Str(0):= ^  ! $80;
end;	\TextIn

\---------------------------------- ANSI C -------------------------------------

proc	MemCpy(Dst, Src, Size);	\Copy block of Size bytes from Src to Dst
char	Dst, Src;
int	Size;
for I:= 0, Size-1 do
	Dst(I):= Src(I);
\(also see StrNCopy and Blit)



proc	MemSet(Array, Value, Size); \Set Size many bytes in Array to Value
char	Array;
int	Value, Size;
int	I;
for I:= 0, Size-1 do Array(I):= Value;
\(also see StrNFill)



func	Sqr(X);			\Square
real	X;
return X*X;



int	MemTop,			\top of memory space available to AllocMem
	MemBase;		\pointer to base of un-allocated memory space

func	AllocMem(Bytes);
\"Reserves" memory, but, unlike the Reserve intrinsic, this doesn't release
\ the space to the heap memory pool when the calling procedure returns.
int	Bytes;			\number of bytes to allocate
int	Temp;
begin
Temp:= MemBase;
MemBase:= MemBase + Bytes;	\reserve bytes
if MemBase >= MemTop then Exit("Error: Out of memory.");
return Temp;
end;	\AllocMem

MemBase:= Reserve(4000000);
MemTop:= Reserve(0) - 100;
Text(2,"MemBase="); HexOut(2,MemBase); Crlf(2);	\debug
Text(2,"MemTop="); HexOut(2,MemTop); Crlf(2);	\debug



func	OneBits(N);		\Return the number of 1 bits in N
int	N;
int	C;
begin
C:= 0;
loop	begin
	if N = 0 then return C;
	N:= N & N-1;
	C:= C+1;
	end;
end;	\OneBits



func	RevBits(N);		\Reverse the order of the bits
int	N;
begin
N:=    (N>>1 & $5555) ! (N & $5555)<<1;
N:=    (N>>2 & $3333) ! (N & $3333)<<2;
N:=    (N>>4 & $0F0F) ! (N & $0F0F)<<4;
return (N>>8 & $00FF) ! (N & $00FF)<<8;
end;	\RevBits



func	RevBytes(N);		\Reverse the order of the bytes in a 32-bit word
int	N;			\ (Swap endians)
return N<<24 ! N>>24 ! N<<8 & $00FF0000 ! N>>8 & $0000FF00;



proc	SetFont;		\Set interrupt vector $43 to point to font table
\The font is included here, near the beginning of the code, because BIOS needs 
\ it in the first 64k of our data segment.
int	CpuReg,		\address of CPU register array (from GetReg)
	DataSeg;	\segment address of our data (for BIOS calls)
begin
include Font;		\(must reside within first 64K of data segment)

CpuReg:= GetReg;
DataSeg:= CpuReg(12);

Poke(0, $43*4, Font);
Poke(0, $43*4+1, Font>>8);
Poke(0, $43*4+2, DataSeg);
Poke(0, $43*4+3, DataSeg>>8);
Poke($40, $85, FontHeight);
Poke($40, $85+1, FontHeight>>8);
end;	\SetFont
