/**************************************************************************
*            Unix-like crypt(3) Algorithm for Password Encryption
*
*   File    : crypt3.c
*   Purpose : Provides crypt(3) functionality to ANSI C compilers
*             without a need for the crypt library.
*   Author  : Michael Dipperstein
*   Date    : November 3, 1998
*
***************************************************************************
*   I am releasing the source that I have provided into public domain
*   without any restrictions, warranties, or copyright claims of my own.
*
*   The code below has been cleaned and compiles correctly under, gcc,
*   lcc, and Borland's bcc C compilers.  A bug involving the left and
*   right halves of the encrypted data block in the widely published
*   crypt3.c source has been fixed by this version.  All implicit register
*   declarations have been removed, because they generated suboptimal code.
*   All constant data has been explicitly declared as const and all
*   declarations have been given a minimal scope, because I'm paranoid.
*
*   Caution: crypt() returns a pointer to static data.  I left it this way
*            to maintain backward compatibility.  The downside is that
*            successive calls will cause previous results to be lost.
*            This can easily be changed with only minor modifications to
*            the function crypt().
**************************************************************************/
#include <unistd.h>

/* Initial permutation */
const char __IP[] =
{
    58, 50, 42, 34, 26, 18, 10, 2,
    60, 52, 44, 36, 28, 20, 12, 4,
    62, 54, 46, 38, 30, 22, 14, 6,
    64, 56, 48, 40, 32, 24, 16, 8,
    57, 49, 41, 33, 25, 17,  9, 1,
    59, 51, 43, 35, 27, 19, 11, 3,
    61, 53, 45, 37, 29, 21, 13, 5,
    63, 55, 47, 39, 31, 23, 15, 7,
};

/* Final permutation, FP = IP^(-1) */
const char __FP[] = {
    40, 8, 48, 16, 56, 24, 64, 32,
    39, 7, 47, 15, 55, 23, 63, 31,
    38, 6, 46, 14, 54, 22, 62, 30,
    37, 5, 45, 13, 53, 21, 61, 29,
    36, 4, 44, 12, 52, 20, 60, 28,
    35, 3, 43, 11, 51, 19, 59, 27,
    34, 2, 42, 10, 50, 18, 58, 26,
    33, 1, 41,  9, 49, 17, 57, 25,
};

/**************************************************************************
* Permuted-choice 1 from the key bits to yield C and D.
* Note that bits 8,16... are left out:
* They are intended for a parity check.
**************************************************************************/
static const char PC1_C[] =
{
    57, 49, 41, 33, 25, 17,  9,
     1, 58, 50, 42, 34, 26, 18,
    10,  2, 59, 51, 43, 35, 27,
    19, 11,  3, 60, 52, 44, 36,
};

static const char PC1_D[] =
{
    63, 55, 47, 39, 31, 23, 15,
     7, 62, 54, 46, 38, 30, 22,
    14,  6, 61, 53, 45, 37, 29,
    21, 13,  5, 28, 20, 12,  4,
};

/* Sequence of shifts used for the key schedule. */
static const char shifts[] =
    {1, 1, 2, 2, 2, 2, 2, 2, 1, 2, 2, 2, 2, 2, 2, 1};

/**************************************************************************
* Permuted-choice 2, to pick out the bits from the CD array that generate
* the key schedule.
**************************************************************************/
static const char PC2_C[] =
{
    14, 17, 11, 24,  1,  5,
     3, 28, 15,  6, 21, 10,
    23, 19, 12,  4, 26,  8,
    16,  7, 27, 20, 13,  2,
};

static const char PC2_D[] =
{
    41, 52, 31, 37, 47, 55,
    30, 40, 51, 45, 33, 48,
    44, 49, 39, 56, 34, 53,
    46, 42, 50, 36, 29, 32,
};

/* The C and D arrays used to calculate the key schedule. */
static char C[ 28 ];
static char D[ 28 ];

/* The key schedule.  Generated from the key. */
char __KS[ 16 ][ 48 ];

/* The E bit-selection table. */
char __E[ 48 ];
static const char e2[] =
{
    32,  1,  2,  3,  4,  5,
     4,  5,  6,  7,  8,  9,
     8,  9, 10, 11, 12, 13,
    12, 13, 14, 15, 16, 17,
    16, 17, 18, 19, 20, 21,
    20, 21, 22, 23, 24, 25,
    24, 25, 26, 27, 28, 29,
    28, 29, 30, 31, 32,  1,
};

/**************************************************************************
* Function:    setkey
*
* Description: Set up the key schedule from the encryption key.
*
* Inputs:      char *key
*              pointer to 64 character array.  Each character represents a
*              bit in the key.
*
* Returns:     none
**************************************************************************/
void setkey( char *key )
{
    int i, j, k, temp;

    /**********************************************************************
    * First, generate C and D by permuting the key.  The low order bit of
    * each 8-bit char is not used, so C and D are only 28 bits apiece.
    **********************************************************************/
    for( i = 0; i < 28; i++ )
    {
        C[ i ] = key[ PC1_C[ i ] - 1 ];
        D[ i ] = key[ PC1_D[ i ] - 1 ];
    }

    /**********************************************************************
    * To generate Ki, rotate C and D according to schedule and pick up a
    * permutation using PC2.
    **********************************************************************/
    for( i = 0; i < 16; i++ )
    {
        /* rotate */
        for( k = 0; k < shifts[ i ]; k++ )
        {
            temp = C[ 0 ];

            for( j = 0; j < 28 - 1; j++ )
                C[ j ] = C[ j + 1 ];

            C[ 27 ] = temp;
            temp = D[ 0 ];
            for( j = 0; j < 28 - 1; j++ )
                D[ j ] = D[ j + 1 ];

            D[ 27 ] = temp;
        }

        /* get Ki. Note C and D are concatenated */
        for( j = 0; j < 24; j++ )
        {
            __KS[ i ][ j ] = C[ PC2_C[ j ] - 1 ];
            __KS[ i ][ j + 24 ] = D[ PC2_D[ j ] - 28 -1 ];
        }
    }

    /* load E with the initial E bit selections */
    for( i = 0; i < 48; i++ )
        __E[ i ] = e2[ i ];
}

