/*****************************************************************************
Stripped-down printf()
Chris Giese <geezer@execpc.com>	http://www.execpc.com/~geezer
Release date: Dec 12, 2003
Blair Campbell <Blairdude@gmail.com>
Release date: Oct 06, 2006
This code is public domain (no copyright).
You can do whatever you want with it.

Revised Oct 06, 2006
- added +, #, [space], F, *, L, '(posix),  and precision support to printf
  Note that ' just uses commas for simplicity, not the locale-specific separator
- integrated into clib
- Need to reduce amount of code

Revised Dec 12, 2003
- fixed vsprintf() and sprintf() in test code

Revised Jan 28, 2002
- changes to make characters 0x80-0xFF display properly

Revised June 10, 2001
- changes to make vsprintf() terminate string with '\0'

Revised May 12, 2000
- math in DO_NUM is now unsigned, as it should be
- %0 flag (pad left with zeroes) now works
- actually did some TESTING, maybe fixed some other bugs

%[flag][width][.prec][mod][conv]
flag:	-	left justify, pad right w/ blanks	DONE
	0	pad left w/ 0 for numerics		        DONE
	+	always print sign, + or -	            DONE
	' '	(blank)					                DONE
	#	0X, 0x, or 0 (X, x, or o)               DONE

width:		(field width)			            DONE

prec:		(precision)				            DONE

conv:	d,i	decimal int				            DONE
	u	decimal unsigned			            DONE
	o	octal					                DONE
	x,X	hex					                    DONE
	f,e,g,E,G float				                no
	c	char					                DONE
	s	string					                DONE
	p	ptr					                    DONE

mod:N	near ptr    				            DONE
	F	far ptr					                DONE
	h	short (16-bit) int			            DONE
	l	long (32-bit) int			            DONE
	L	long long (64-bit) int		            DONE
*****************************************************************************/
#include <string.h>				/* strlen() */
#include <stdio.h>				/* stdout, putchar(), fputs() */
#include <stdlib.h>
#include <stdarg.h>
#include <dos.h>
#include <io.h>
#include <_printf.h>

/* flags used in processing format string */
#define	PR_LJ	0x0001			/* left justify */
#define	PR_CA	0x0002			/* use A-F instead of a-f for hex */
#define	PR_SG	0x0004			/* signed numeric conversion (%d vs. %u) */
#define	PR_32	0x0008			/* long (32-bit) numeric conversion */
#define	PR_16	0x0010			/* short (16-bit) numeric conversion */
#define	PR_WS	0x0020			/* PR_SG set and num was < 0 */
#define	PR_LZ	0x0040			/* pad left with '0' instead of ' ' */
#define	PR_FP	0x0080			/* pointers are far */
#define PR_64   0x0100			/* long long (64-bit) numeric conversion */
#define PR_PD   0x0200          /* Pound (#) symbol specified */
#define PR_AL   0x0400          /* Always print sign */
#define PR_BL   0x0800          /* Print blank if positive */
#define PR_TR   0x1000          /* Truncate string */
#define PR_CM   0x2000          /* Separate numbers with commas */
#define PR_08   0x4000          /* Char (as opposed to integer) numeric arg */
/* largest number handled is 2^32-1, lowest radix handled is 8.
2^32-1 in base 8 has 11 digits (add 5 for trailing NUL and for slop) */
#define	PR_BUFLEN	39
#ifdef __COMPILE_LONG_LONG__
#define __HAVE_LONG_LONG__
#define __vfnprintf __llvfnprintf
#define _fnprintf   _llfnprintf
#endif

/*****************************************************************************
name:	__vfnprintf
action:	minimal subfunction for ?printf, calls function
	'fn' with arg 'ptr' for each character to be output
returns:total number of characters output
*****************************************************************************/
int __vfnprintf( int( *fn )( int c ), const char *fmt, va_list args )
{
	unsigned        flags, actual_wd, count, given_wd, given_pc, len, group;
	char *          buf[ PR_BUFLEN ];
    char far *      where, far * fp;
	unsigned char   state, radix;
	long            num;
#ifdef __HAVE_LONG_LONG__
	long long       longnum;
#endif

	state = flags = count = given_wd = given_pc = len = group = 0;
/* begin scanning format specifier list */
	for( ; *fmt; fmt++ ) {
		switch( state ) {
/* STATE 0: AWAITING % */
			case 0:
				if( *fmt != '%' ) {	/* not %... */
					if( fn( *fmt ) == EOF ) return( -1 );
                    /* ...just echo it */
					count++;
					break;
				}
/* found %, get next char and advance state to check if next char is a flag */
				state++;
				fmt++;
				/* FALL THROUGH */
/* STATE 1: AWAITING FLAGS (%-0) */
			case 1:
				if( *fmt == '%' ) {	/* %% */
					if( fn( *fmt ) == EOF ) return( -1 );
					count++;
					state = flags = given_wd = 0;
					break;
				}
CHECKFLAGS:
				if( *fmt == '-' ) {
					if( flags & PR_LJ )	/* %-- is illegal */
						state = flags = given_wd = 0;
					else
						flags |= PR_LJ;
                    fmt++;
				}
/* check now for '%0...' */
				if( *fmt == '0' ) {
					flags |= PR_LZ;
					fmt++;
                    goto CHECKFLAGS;
				}
/* check for pound symbol (print 0X or 0x for hex numbers) */
                if( *fmt == '#' ) {
                    flags |= PR_PD;
                    fmt++;
                    goto CHECKFLAGS;
                }
/* check for always-print-sign symbol */
                if( *fmt == '+' ) {
                    flags |= PR_AL;
                    fmt++;
                    goto CHECKFLAGS;
                }
/* check for print-blank-if-positive */
                if( *fmt == ' ' ) {
                    flags |= PR_BL;
                    fmt++;
                    goto CHECKFLAGS;
                }
/* check for print-thousands-grouping chars */
                if( *fmt == '\'' ) {
                    flags |= PR_CM;
                    fmt++;
                    goto CHECKFLAGS;
                }
                state++;
				/* FALL THROUGH */
/* STATE 2: AWAITING (NUMERIC) FIELD WIDTH */
			case 2:
				if( *fmt >= '0' && *fmt <= '9' ) {
					given_wd = 10 * given_wd + ( *fmt - '0' );
					break;
				} else if(  *fmt == '*' ) {
                    given_wd = va_arg( args, int );
                    /* and fall through */
                }
/* not field width: advance state to check if it's a modifier */
				state++;
				/* FALL THROUGH */
/* STATE 3: AWAITING PRECISION SPECIFICATION (.num) */
            case 3:
                if( *fmt == '.' ) {
                    fmt++;
                    if( *fmt == '0' ) given_pc = 0;
                    else if( *fmt > '0' && *fmt <= '9' ) {
                        given_pc = 0;
                        /* and loop */
                        while( *fmt >= '0' && *fmt <= '9' ) {
                            given_pc = 10 * given_pc + ( *fmt - '0' );
                            fmt++;
                        }
                        flags |= PR_TR;
                    } else if( *fmt == '*' ) {
                        given_pc = va_arg( args, int );
                        if( given_pc ) flags |= PR_TR;
                        /* and fall through */
                    }
                }
                state++;
/* STATE 4: AWAITING MODIFIER CHARS (FNlh) */
			case 4:
				if( *fmt == 'F' ) {
					flags |= PR_FP;
					break;
				}
				if( *fmt == 'N' )
					break;
				if( *fmt == 'l' ) {
					flags |= PR_32;
                    flags |= PR_FP; /* 
                                     * Under posix, 'l' can also specify 'long'
                                     * pointers
                                     */
					break;
				}
#ifdef __HAVE_LONG_LONG__
                /*
                 * Under posix, "ll" means long long, also, intmax_t evaluates
                 * to long long
                 */
				if( *fmt == 'L' ||
                    strncmp( &*fmt, "ll", 2 ) == 0 ||
                    *fmt == 'j' ) {
                    if( *fmt == 'l' ) fmt++;
					flags |= PR_64;
                    flags |= PR_FP;
					break;
				}
#endif
                /* 'z' means size_t, 't' means ptrdiff_t, both unsigned ints */
                if( *fmt == 'z' || *fmt == 't' || *fmt == 'h' ) {
                    fmt++;
                    if( *fmt != 'h' ) {
                        fmt--;
                        flags |= PR_16;
                    } else flags |= PR_08;
                    /* under posix, "hh" pertains to char arguments */
                    break;
                }
/* not modifier: advance state to check if it's a conversion char */
				state++;
				/* FALL THROUGH */
/* STATE 5: AWAITING CONVERSION CHARS (Xxpndiuocs) */
			case 5:
				where = ( char far * )buf + PR_BUFLEN - 1;
				*where = '\0';
				switch( *fmt ) {
					case 'X':
						flags |= PR_CA;
						/* FALL THROUGH */
					case 'x':
						radix = 16;
						goto DO_NUM;
					case 'd':
					case 'i':
						flags |= PR_SG;
						/* FALL THROUGH */
					case 'u':
						radix = 10;
						goto DO_NUM;
					case 'o':
						radix = 8;
/* load the value to be printed. l=long=32 bits: */
DO_NUM:
                        if( flags & PR_TR ) {
                            flags &= ~PR_TR;    /* Don't truncate numbers */
                            flags |= PR_LZ;
                            if( given_pc ) given_wd = given_pc;
                        }
                        if( flags & PR_32 )
							num = va_arg( args, unsigned long );
/* h=short=16 bits (signed or unsigned) */
#ifdef __HAVE_LONG_LONG__
						else if( flags & PR_64 ) {
                            if( flags & PR_SG ) {
                                longnum = va_arg( args, long long );
                            } else {
                                longnum = va_arg( args, unsigned long long );
                            }
						}
#endif
                        else if( flags & PR_08 ) {
                            if( flags & PR_SG )
                                num = va_arg( args, signed char );
                            else num = va_arg( args, unsigned char );
                        }
						else /*if( flags & PR_16 )*/ {
							if( flags & PR_SG )
								num = va_arg( args, short );
							else num = va_arg( args, unsigned short );
						}
/* no h nor l: sizeof(int) bits (signed or unsigned) *//*
						else {
							if( flags & PR_SG )
								num = va_arg( args, int );
							else
								num = va_arg( args, unsigned int );
						}*/
                        /* After all, in 16-bit compilers, shorts ARE ints */
/* take care of sign */
						if( flags & PR_SG ) {
							if( num < 0 ) {
								flags |= PR_WS;
								num = -num;
							}
						}

                        if( radix == 8 && num == 0 ) flags &= ~PR_PD;
                        /* only print 0num if argument is non-zero */
/* convert binary to octal/decimal/hex ASCII
OK, I found my mistake. The math here is _always_ unsigned */
						do {
#ifdef __HAVE_LONG_LONG__
							if( flags & PR_64 ) {
								unsigned long long temp;

								temp = ( unsigned long long )longnum % radix;
								where--;
                                if( flags & PR_CM &&
                                    group == 3 &&
                                    temp ) {
                                    *where-- = ',';
                                    group = 0;
                                }
								if( temp < 10LL )
									*where = temp + '0';
								else if( flags & PR_CA )
									*where = temp - 10LL + 'A';
								else
									*where = temp - 10LL + 'a';
								longnum = ( unsigned long long )longnum / radix;
                                group++;
							}
							else {
#endif
								unsigned long temp;

								temp = ( unsigned long )num % radix;
								where--;
                                if( flags & PR_CM &&
                                    group == 3 &&
                                    temp ) {
                                    *where-- = ',';
                                    group = 0;
                                }
								if( temp < 10 )
									*where = temp + '0';
								else if( flags & PR_CA )
									*where = temp - 10 + 'A';
								else
									*where = temp - 10 + 'a';
								num = ( unsigned long )num / radix;
                                group++;
#ifdef __HAVE_LONG_LONG__
							}
#endif
                        } while( num != 0 );

						goto EMIT;
                    case 'p':
                        if( flags & PR_FP ) {
                            where = va_arg( args, char far * );
                            count += _fnprintf( fn, "%04x:%04x",
                                                FP_SEG( where ),
                                                FP_OFF( where ) );
                        } else {
                            where = va_arg( args, char * );
                            count += _fnprintf( fn, "%04x", FP_OFF( where ) );
                        }
                        break;
                    case 'n':
                        if( flags & PR_FP )
                            *va_arg( args, int far * ) = count;
                        else *va_arg( args, int * ) = count;
                        break;
                    case 'C':
                        flags |= PR_FP;
					case 'c':
/* disallow pad-left-with-zeroes for %c */
						flags &= ~PR_LZ;
						where--;
                        if( flags & PR_FP ) 
                            *where = va_arg( args, unsigned int );
                        else *where = va_arg( args, unsigned char );
						actual_wd = 1;
						goto EMIT2;
                    case 'S': /* same as Fs or ls */
                        flags |= PR_FP;
					case 's':
/* disallow pad-left-with-zeroes for %s */
						flags &= ~PR_LZ;
                        if( flags & PR_FP ) where = va_arg( args, char far * );
						else where = va_arg( args, char * );
EMIT:
                        actual_wd = 0;
                        fp = where;
                        while( *fp++ ) actual_wd++;

						if( flags & PR_WS )
							actual_wd++;
/* if we pad left with ZEROES, do the sign now */
						if( ( flags & ( PR_WS | PR_LZ ) ) ==
                            ( PR_WS | PR_LZ ) ) {
							if( fn( '-' ) == EOF ) return( -1 );
							count++;
						} else if( ( flags & PR_SG ) && ( flags & PR_AL ) ) {
                            if( fn( '+' ) == EOF ) return( -1 );
                            count++;
                        } else if( ( ( flags & PR_BL ) &&
                                   ( flags & PR_SG ) ) &&
                                   !( flags & PR_WS ) ) {
                            if( fn( ' ' ) == EOF ) return( -1 );
                            count++;
                        }
/* pad on left with spaces or zeroes (for right justify) */
EMIT2:
/* if we're printing # characters, we should do it now */
                        if( flags & PR_PD && radix == 16 )
                            if( fn( '0' ) == EOF ||
                                fn( flags & PR_CA ? 'X' : 'x' ) == EOF )
                                return( -1 );
                            else count += 2;
                        else if( flags & PR_PD &&
                                 radix == 8 )
                            if( fn( '0' ) == EOF ) return( -1 );
                            else count++;

                        if( ( flags & PR_LJ ) == 0 ) {
							while( given_wd > actual_wd ) {
								if( fn( flags & PR_LZ ? '0' : ' ' ) == EOF )
                                    return( -1 );
								count++;
								given_wd--;
							}
						}
/* if we pad left with SPACES, do the sign now */
						if( ( flags & ( PR_WS | PR_LZ ) ) == PR_WS ) {
							if( fn( '-' ) == EOF ) return( -1 );
							count++;
						}
/* emit string/char/converted number */
                        len = 0;
						    while( *where != '\0' ) {
                                len++;
                                if( len > given_pc && ( flags & PR_TR ) ) break;
							    if( fn( *where++ ) == EOF ) return( -1 );
    							count++;
	    					}
/* pad on right with spaces (for left justify) */
						if( ( given_wd < actual_wd ) || flags & PR_TR )
                            given_wd = 0;
						else given_wd -= actual_wd;
						for( ; given_wd; given_wd-- ) {
							if( fn( ' ' ) == EOF ) return( -1 );
							count++;
						}
						break;
					default:
						break;
				}
			default:
				state = flags = given_wd = radix = group = 0;
				break;
		}
	}
	return( count );
}

int _fnprintf( int( *fn )( int c ), const char *fmt, ... )
{
    va_list args;

    va_start( args, fmt );

    return( __vfnprintf( fn, fmt, args ) );
}

