#include <sys/stat.h>
#include <dirent.h>
#include <ftw.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

int ftw( char *directory,
         int( *funcptr )( const char *name, const struct stat *ptr, int flag ),
         int depth )
{
    register DIR *  dirp;
    struct dirent * entp;
    struct stat     stats;
    register char * p;
    register int    i;
    long            seekpoint;
    char *          fullpath;

    /* If can't stat, tell the user so. */
    if( stat( directory, &stats ) < 0 ) 
        return( ( *funcptr )( directory, &stats, FTW_NS ) );

    /* If it's not a directory, call the user's function. */
    if( ( stats.st_mode & S_IFMT ) != S_IFDIR )
       /* Saying "FTW_F" here is lying; what if this is a symlink? */
       return( ( *funcptr )( directory, &stats, FTW_F ) );

    /* Open directory; if we can't, tell the user so. */
    dirp = opendir( directory );
    if( dirp == NULL ) return( ( *funcptr )( directory, &stats, FTW_DNR ) );

    /* See if user wants to go further. */
    i = ( *funcptr )( directory, &stats, FTW_D );
    if( i ) {
       closedir( dirp );
       return( i );
    }

    /* Get ready to hold the full paths. */
    i = strlen( directory );
    fullpath = malloc( i + 1 + PATH_MAX + 1 );
    if( fullpath == NULL ) {
       closedir( dirp );
       return( -1 );
    }
    
    strcpy( fullpath, directory );
    p = &fullpath[ i ];
    if( i && p[ -1 ] != '/' ) *p++ = '/';

    /* Read all entries in the directory.. */
    while( entp = readdir( dirp ) )
       if( strcmp( entp->d_name, "." ) != 0 &&
           strcmp( entp->d_name, ".." ) != 0 ) {
           if( depth <= 1 ) {
               /* Going too deep; checkpoint and close this directory. */
               seekpoint = telldir( dirp );
               closedir( dirp );
               dirp = NULL;
           }

           /* Process the file. */
           strcpy( p, entp->d_name );
           i = ftw( fullpath, funcptr, depth - 1 );
           if( i ) {
               /* User's finished; clean up. */
               free( fullpath );
               if( dirp ) closedir( dirp );
               return( i );
           }

           /* Reopen the directory if necessary. */
           if( dirp == NULL ) {
               dirp = opendir( directory );
               if( dirp == NULL ) {
                   free( fullpath );
                   return( -1 );
               }
               seekdir( dirp, seekpoint );
           }
       }

    /* Clean up. */
    free( fullpath );
    closedir( dirp );
    return( 0 );
} 
