/* The gencat program will convert a UNIX message text source file
   (.msg file) into a cats message catalog. */

/* Copyright (C) 2000 Jim Hall <jhall1@isd.net> */

/*
  This program is provided to assist developers in using the "cats"
  library.  As such, the gencat program is also distributed under the
  GNU Lesser GPL:

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/*
  The gencat command creates a message catalog file (usually *.cat)
  from message text source files (usually *.msg). The gencat command
  merges the message text source files, specified by the SourceFile
  parameter, into a formatted message catalog, specified by the CatalogFile
  parameter. After entering messages into a source file, use the gencat
  command to process the source file to create a message catalog. The
  gencat command creates a catalog file if one does not already exist.
  If the catalog file does exist, the gencat command includes the new
  messages in the catalog file.

  You can specify any number of message text source files. The gencat
  command processes multiple source files, one after another, in the
  sequence specified. Each successive source file modifies the catalog.
  If the set and message numbers collide, the new message text defined
  in the SourceFile parameter replaces the old message text currently
  contained in the CatalogFile parameter. Message numbers must be in
  the range of 1 through NL_MSGMAX. The set number must be in the range
  of 1 through NL_SETMAX.

  The gencat command does not accept symbolic message identifiers. You
  must run the mkcatdefs command if you want to use symbolic message
  identifiers.
*/

/*
  An example of a (simple) .msg file is:

$
$       template.msg
$
$       message catalog for template example
$

$ Module: template.c
$

$set 1
$ General application messages
1 Template
2 Template Open
3 Template Save As
4 Template Help

$set 2
$ These messages are used in the main window of the template example.
1 File
2 Open...
3 Save As...
4 Print
5 Clear
6 Exit
7 Help
8 Overview...
*/

/*
  My message catalogs look like this: (Cats) 

1.1:Hello world
7.4:Failure writing to drive A:
*/


#include <stdio.h>
#include <stdlib.h>			/* free, atoi */
#include <string.h>			/* strtok */

/* functions */

char *get_line (FILE *pfile, int continue_ch);
void gencat (FILE *pfile);

/* main program */

int
main (int argc, char **argv)
{
  int i;
  FILE *pfile;

  /* read the command line, and process all files */

  /*
    The general usage of GNU gencat is:

    Usage: gencat [OPTION...] -o OUTPUT-FILE [INPUT-FILE]...
    or:  gencat [OPTION...] [OUTPUT-FILE [INPUT-FILE]...]

    -H, --header=NAME          Create C header file NAME containing symbol
                               definitions

        --new                  Do not use existing catalog, force new
                               output file
                            
    -o, --output=NAME          Write output to file NAME
    -?, --help                 Give this help list
        --usage                Give a short usage message
    -V, --version              Print program version
  */

  /*
    But for this version of "Cats" gencat, we'll use:

      gencat [[INPUT-FILE] ...]

    and I'll migrate to the UNIX command line interface later on.
  */

  for (i = 1; i < argc; i++)
    {
      pfile = fopen (argv[i], "r");
      if (!pfile)
	{
	  fprintf (stderr, "gencat: %s: cannot open file\n", argv[i]);
	}

      else
	{
	  gencat (pfile);
	  fclose (pfile);
	}
    } /* for */

  if (i == 1)
    {
      gencat (stdin);
    }

  exit (0);
} /* main */

void
gencat (FILE *pfile)
{
  char *str;
  int set;
  int num;

  set = 0;

  while ((str = get_line (pfile, '\\')) != NULL)
    {
      if ((str[0] == '$') || (str[0] == '\0'))
	{
	  /* might be a comment, or a "$set" command */

	  if (strncmp (str, "$set", 4) == 0)
	    {
	      /* change the set */

	      strtok (str, " ");
	      set = atoi (strtok (NULL, " "));
	    }

	  /* else, a comment, and we just ignore it */
	}

      else
	{
	  /* format is "1.3:Save As..." */

	  num = atoi (strtok (str, " "));
	  printf ("%d.%d:%s\n", set, num, strtok (NULL, ""));
	}

      /* get_line will malloc memory, so we need to free it */

      free (str);
    } /* while */

  return;
} /* gencat */
