/* db.c */

/* Functions that support an in-memory key-value database */

/*
  Copyright (C) 1999-2000 Jim Hall <jhall1@isd.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>			/* malloc, qsort */
#include <string.h>			/* strncpy */
#include "db.h"

db_t *db_ary;				/* pointer to array */
size_t db_nmemb;			/* number of elements in db_ary */
size_t db_maxmemb;			/* max. no. of elements in db_ary */

int
db_init (size_t init_size)
{
  /* malloc the memory at the initial size */

  /* returns: true (1) if successful, false (0) if failure */

  db_ary = (db_t *) malloc (sizeof (db_t) * init_size);
  if (db_ary)
    {
      db_nmemb = 0;
      db_maxmemb = init_size;
      return (1);
    }

  /* else */

  return (0);
}

int
db_free (void)
{
  /* free the memory associated with the db_ary */

  /* always returns true (1) */

  free (db_ary);
  return (1);
}

int
db_insert (char *key, char *value)
{
  /* insert a new value into the array */

  /* returns: true (1) if successful, false (0) if failure */

  db_t *db_tmp;
  db_t *exist;

  /* does the key already exist? */

  exist = db_fetch (key);
  if (exist)
    {
      /* key already exists.  replace it */

#ifndef NDEBUG
      printf ("*db_insert* replacing key %s with value %s\n", key, value);
#endif /* NDEBUG */

      strncpy (exist->value, value, DB_VALLEN);
    }

  else
    {
      /* key does not exist.  add new one */

      /* check if we have enough room */

      if (db_nmemb >= db_maxmemb)
	{
	  /* need to allocate more memory first */

	  /* simple assumption: double the memory */

	  db_tmp = (db_t *) realloc (db_ary, sizeof (db_t) * db_maxmemb * 2);
	  if (!db_tmp)
	    {
	      /* failed */

	      return (0);
	    }

	  /* else, reassign the memory */

	  db_ary = db_tmp;
	} /* if not enough room */

#ifndef NDEBUG
      printf ("*db_insert* inserting key %s with value %s\n", key, value);
#endif /* NDEBUG */

      strncpy (db_ary[db_nmemb].key, key, DB_KEYLEN);
      strncpy (db_ary[db_nmemb].value, value, DB_VALLEN);
      db_nmemb++;

      /* re-sort the array */

#ifndef NDEBUG
      printf ("*db_insert* re-sort the array..\n");
#endif /* NDEBUG */

      qsort (db_ary, db_nmemb, sizeof (db_t),
             (int(*)(const void *, const void *))db_compare);

    } /* else */

  /* done - success! */

  return (1);
}

int
db_compare (db_t *p1, db_t *p2)
{
  /* compares two database elements (key) suitable for qsort */

  /* returns: <0 if p1<p2, ==0 if p1==p2, >0 if p1>p2 */

  int ret;

  ret = strcmp (p1->key, p2->key);
  return (ret);
}

db_t *
db_fetch (char *key)
{
  /* fetches a key-value pair from the database, based on a key */

  /* returns: ptr to the key-value pair if successful, NULL if failed */

  db_t *ptr;

  ptr = bsearch (key, db_ary, db_nmemb, sizeof (db_t),
                 (int(*)(const void *, const void *))db_compare);
  return (ptr);
}
