/****************************************************************************
*
*                            Open Watcom Project
*
*    Portions Copyright (c) 1983-2002 Sybase, Inc. All Rights Reserved.
*
*  ========================================================================
*
*    This file contains Original Code and/or Modifications of Original
*    Code as defined in and that are subject to the Sybase Open Watcom
*    Public License version 1.0 (the 'License'). You may not use this file
*    except in compliance with the License. BY USING THIS FILE YOU AGREE TO
*    ALL TERMS AND CONDITIONS OF THE LICENSE. A copy of the License is
*    provided with the Original Code and Modifications, and is also
*    available at www.sybase.com/developer/opensource.
*
*    The Original Code and all software distributed under the License are
*    distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
*    EXPRESS OR IMPLIED, AND SYBASE AND ALL CONTRIBUTORS HEREBY DISCLAIM
*    ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF
*    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR
*    NON-INFRINGEMENT. Please see the License for the specific language
*    governing rights and limitations under the License.
*
*  ========================================================================
*
* Description:  Processing of segment and group related directives:
*               - SEGMENT, ENDS
*               - GROUP
*               - .CODE, .DATA, .DATA?, .CONST, .STACK, .FARDATA, .FARDATA?
****************************************************************************/


#include <ctype.h>

#include "globals.h"
#include "memalloc.h"
#include "symbols.h"
#include "parser.h"
#include "directiv.h"
#include "input.h"
#include "tokenize.h"
#include "queues.h"
#include "expreval.h"
#include "omf.h"
#include "fastpass.h"
#include "macro.h"
#include "fixup.h"

#include "myassert.h"

#define is_valid_id_char( ch ) \
    ( isalpha(ch) || isdigit(ch) || ch=='_' || ch=='@' || ch=='$' || ch=='?' )

/* prototypes */
extern ret_code         checkword( char **token );
extern int              SetAssumeCSCurrSeg( void );

extern symbol_queue     Tables[];       // tables of definitions

#define DEFAULT_STACK_SIZE      1024
#define DEFAULT_CODE_CLASS    "CODE"

enum {
#undef fix
#define fix( tok, str, val, init )              tok

#include "dirtoken.h"
};

typedef enum {
    SIM_CODE = 0,
    SIM_STACK,
    SIM_DATA,
    SIM_DATA_UN,            // .DATA?
    SIM_FARDATA,
    SIM_FARDATA_UN,         // .FARDATA?
    SIM_CONST,
    SIM_NONE,
    SIM_LAST = SIM_NONE
} sim_seg;

typedef struct {
    sim_seg     seg;                    // segment id
    char        close[MAX_LINE_LEN];    // closing line for this segment
//    int         stack_size;             // size of stack segment
} last_seg_info;        // information about last opened simplified segment

extern struct asm_sym   *SegOverride;
extern asm_sym          *sym_CurSeg;

uint                    segdefidx;      // Number of Segment definition
static uint             grpdefidx;      // Number of Group definition
uint                    LnamesIdx;      // Number of LNAMES definition


seg_item                *CurrSeg;       // points to stack of opened segments
bool                    Use32;          // if current segment is 32-bit

char first_issued;

// strings used by simplified segment directives

static char *SimCodeBegin[ SIM_LAST ] = {
        "%s SEGMENT %s %s PUBLIC '%s'",
        "%s SEGMENT %s %s STACK 'STACK'",
        "%s SEGMENT %s %s PUBLIC 'DATA'",
        "%s SEGMENT %s %s PUBLIC 'BSS'",
        "%s SEGMENT %s %s PRIVATE 'FAR_DATA'",
        "%s SEGMENT %s %s PRIVATE 'FAR_BSS'",
//        "%s SEGMENT %s %s PUBLIC 'CONST' READONLY"
        "%s SEGMENT %s %s PUBLIC 'CONST'"
};

static char *SegmNames[ SIM_LAST ] = {
    "_TEXT", "STACK", "_DATA", "_BSS", "FAR_DATA", "FAR_BSS", "CONST"
};

/* Code generated by simplified segment definitions */

static last_seg_info    lastseg;        // last opened simplified segment

#if FASTPASS
// saved state
static last_seg_info    saved_lastseg; 
static int              saved_NumSegs;
static dir_node         **saved_SegStack;
static char             *saved_CurSeg_name;
static char             saved_first_issued;
static bool             saved_Use32;
#endif

dir_node         *flat_grp; /* magic FLAT group */

/* generic byte buffer, used for OMF LEDATA records only */
static uint_8           codebuf[ 1024 ];

#define ROUND_UP( i, r ) (((i)+((r)-1)) & ~((r)-1))

static int push_seg( dir_node *seg )
/**********************************/
/* Push a segment into the current segment stack */
{
    seg_item    *curr;

    for( curr = CurrSeg; curr; curr = curr->next ) {
        if( curr->seg == seg ) {
            DebugMsg(("push_seg: segment is already pushed\n"));
            AsmErr( BLOCK_NESTING_ERROR, curr->seg->sym.name );
            return( ERROR );
        }
    }
    push( &CurrSeg, seg );
    SetAssumeCSCurrSeg();
    return( NOT_ERROR );
}

static dir_node *pop_seg( void )
/******************************/
/* Pop a segment out of the current segment stack */
{
    dir_node    *seg;

    /**/myassert( CurrSeg != NULL );
    seg = pop( &CurrSeg );
    SetAssumeCSCurrSeg();
    return( seg );
}

direct_idx GetLnameIdx( char *name )
/**********************************/
{
    struct asm_sym      *sym;
    dir_node            *dir;

    sym = SymSearch( name );
    if( sym == NULL )
        return( LNAME_NULL );

    dir = (dir_node *)sym;
    if( sym->state == SYM_UNDEFINED ) {
        return( LNAME_NULL );
    } else if( sym->state == SYM_GRP ) {
        return( dir->e.grpinfo->lname_idx );
    } else if( sym->state == SYM_SEG ) {
        return( dir->e.seginfo->lname_idx );
    } else {    /* it is an lname record */
        return( dir->sym.idx );
    }
}

// add a class name

static direct_idx InsertClassLname( char *name )
/**********************************************/
{
    asm_sym *sym;

    if( strlen( name ) > MAX_LNAME ) {
        AsmError( LNAME_TOO_LONG );
        return( LNAME_NULL );
    }

    /* the classes aren't inserted into the symbol table
     but they are in a queue */

    sym = SymCreate( name, FALSE );
    sym->state = SYM_CLASS_LNAME;
    sym->idx = ++LnamesIdx;

    /* put it into the lname table */

    AddLnameData( (dir_node *)sym );

    return( LnamesIdx );
}

uint_32 GetCurrOffset( void )
/*************************/
{
    if( CurrSeg )
        return( CurrSeg->seg->e.seginfo->current_loc );
    return( 0 );
}

dir_node *GetCurrSeg( void )
/**************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    return( CurrSeg->seg );
}

int GetCurrClass( void )
/**************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    return( CurrSeg->seg->e.seginfo->segrec->d.segdef.class_name_idx );
}

uint_32 GetCurrSegAlign( void )
/*****************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    switch( CurrSeg->seg->e.seginfo->segrec->d.segdef.align ) {
    case ALIGN_ABS: // same as byte aligned ?
    case ALIGN_BYTE:
        return( 1 );
    case ALIGN_WORD:
        return( 2 );
    case ALIGN_DWORD:
        return( 4 );
    case ALIGN_PARA:
        return( 16 );
    case ALIGN_PAGE:
        return( 256 );
    case ALIGN_4KPAGE:
        return( 4096 );
    default:
        return( 0 );
    }
}

uint_32 GetCurrSegStart( void )
/*****************************/
{
    if( CurrSeg == NULL )
        return( 0 );
#if 0
    /**/myassert(( CurrSeg->seg->e.seginfo->current_loc - BufSize )
                  == CurrSeg->seg->e.seginfo->start_loc );
#endif
    return( CurrSeg->seg->e.seginfo->start_loc );
}

static dir_node *CreateGroup( char *name )
/****************************************/
{
    dir_node    *grp;

    grp = (dir_node *)SymSearch( name );

    if( grp == NULL || grp->sym.state == SYM_UNDEFINED) {
        if (grp == NULL)
            grp = dir_insert( name, TAB_GRP );
        else
            dir_change( grp, TAB_GRP );
        grp->e.grpinfo->idx = ++grpdefidx;
        grp->e.grpinfo->lname_idx = ++LnamesIdx;
        AddLnameData( grp );
    } else if( grp->sym.state != SYM_GRP ) {
        AsmErr( SYMBOL_PREVIOUSLY_DEFINED, name );
        grp = NULL;
    }
    return( grp );
}

// handle GROUP directive (OMF only)

ret_code GrpDef( int i )
/*****************/
{
    char        *name;
    dir_node    *grp;
    seg_item    *new;
    seg_item    *curr;
    dir_node    *seg;

    if( i < 0 ) {
        AsmError( GRP_NAME_MISSING );
        return( ERROR );
    }

    if (Options.output_format == OFORMAT_COFF ||
        Options.output_format == OFORMAT_ELF) {
        AsmError(GROUP_DIRECTIVE_INVALID_FOR_COFF);
        return( ERROR );
    }

    name = AsmBuffer[i]->string_ptr;
    grp = CreateGroup( name );
    if( grp == NULL )
        return( ERROR );

    for( i+=2;              // skip over the GROUP directive
         i < Token_Count;   // stop at the end of the line
         i+=2 ) {           // skip over commas
        name = AsmBuffer[i]->string_ptr;
        /* Add the segment name */

        /* the tokenizer has problems with "dotnames" */
        if (*name == '.' && *(name+1) == NULLC && AsmBuffer[i+1]->token == T_ID) {
            char * p = AsmBuffer[i+1]->string_ptr;
            AddTokens(i, -1);
            AsmBuffer[i]->string_ptr = StringBufferEnd;
            *(AsmBuffer[i]->string_ptr) = '.';
            strcpy(AsmBuffer[i]->string_ptr+1, p);
            StringBufferEnd = StringBufferEnd + strlen(AsmBuffer[i]->string_ptr) + 1;
            name = AsmBuffer[i]->string_ptr;
        }

        if( checkword( &name ) == ERROR ) {
            AsmError( EXPECTING_COMMA );
            return( ERROR );
        }

        seg = (dir_node *)SymSearch( name );
        if (Parse_Pass == PASS_1) {
            if( seg == NULL ) {
                seg = dir_insert( name, TAB_SEG );
            } else if( seg->sym.state == SYM_UNDEFINED ) {
                dir_change( seg, TAB_SEG );
            } else if( seg->sym.state != SYM_SEG ) {
                AsmErr( SYMBOL_PREVIOUSLY_DEFINED, name );
                return( ERROR );
            } else if( seg->e.seginfo->group != NULL) {
                // segment is in this group already?
                if (seg->e.seginfo->group == &grp->sym )
                    continue;
                // segment is in another group
                DebugMsg(("GrpDef: segment >%s< is in group >%s< already\n", name, seg->e.seginfo->group->name));
                AsmErr( SEGMENT_IN_GROUP, name );
                return( ERROR );
            }
        } else {
            if( seg == NULL || seg->sym.state != SYM_SEG || seg->sym.defined == FALSE) {
                AsmErr( SEG_NOT_DEFINED, name );
                return( ERROR );
            }
            continue;
        }
        /* set the grp index of the segment */
        seg->e.seginfo->group = &grp->sym;

        new = AsmAlloc( sizeof(seg_item) );
        new->seg = seg;
        new->next = NULL;
        grp->e.grpinfo->numseg++;

        /* insert the segment at the end of linked list */
        if( grp->e.grpinfo->seglist == NULL ) {
            grp->e.grpinfo->seglist = new;
        } else {
            curr = grp->e.grpinfo->seglist;
            while( curr->next != NULL ) {
                curr = curr->next;
            }
            curr->next = new;
        }
    }
    return( NOT_ERROR );
}

static int SetUse32( void )
/*************************/
{
    if( CurrSeg == NULL ) {
        Use32 = ModuleInfo.defUse32;
    } else {
        GlobalVars.code_seg = SEGISCODE( CurrSeg );
        Use32 = CurrSeg->seg->e.seginfo->Use32;
        if( Use32 && ( ( ModuleInfo.curr_cpu & P_CPU_MASK ) < P_386 ) ) {
            AsmError( WRONG_CPU_FOR_32BIT_SEGMENT );
            return( ERROR );
        }
    }
    WordSize.offset = Use32 ? 4 : 2;
    return( NOT_ERROR );
}

// 32bit is set as default for FLAT memory model

int SetUse32Def( bool flag )
/**************************/
{
    if( ( CurrSeg == NULL )               // outside any segments
        && ( ModuleInfo.model == MOD_NONE             // model not defined
            || ModuleInfo.cmdline ) ) {   // model defined on cmdline by -m?
        ModuleInfo.defUse32 = flag;
    }
    return( SetUse32() );
}

// close segment

static ret_code CloseSeg(char * name)
{
    struct asm_sym      *sym;

    DebugMsg(("CloseSeg(%s) enter\n", name));
    if( CurrSeg == NULL ) {
        AsmError( SEGMENT_NOT_OPENED );
        return( ERROR );
    }

    if( SymCmpFunc(CurrSeg->seg->sym.name, name ) != 0 ) {
        DebugMsg(("CloseSeg(%s): ENDS segment name not on top of stack\n"));
        AsmErr( BLOCK_NESTING_ERROR, name );
        return( ERROR );
    }

    DebugMsg(("CloseSeg(%s): current ofs=%X\n", name, CurrSeg->seg->e.seginfo->current_loc));

    if (Parse_Pass > PASS_1 && Options.output_format == OFORMAT_OMF)
        omf_FlushCurrSeg();

    pop_seg();

    return( NOT_ERROR );
}

// SEGMENT/ENDS if pass is > 1

ret_code SetCurrSeg( int i )
/**********************/
{
    char        *name;
    uint        ofs;
    struct asm_sym      *sym;

    name = AsmBuffer[i]->string_ptr;

    switch( AsmBuffer[i+1]->value ) {
    case T_SEGMENT:
        if (Options.output_format == OFORMAT_OMF)
            omf_FlushCurrSeg();
        sym = SymSearch( name );
        DebugMsg(("SetCurrSeg(%s) sym=%X\n", name, sym));
        if ( sym == NULL || sym->state != SYM_SEG) {
            AsmErr(SEG_NOT_DEFINED, name);
            return( ERROR );
        }
        push_seg( (dir_node *)sym );
        break;
    case T_ENDS:
        CloseSeg(name);
        break;
    default:
        break;
    }

    return( SetUse32() );
}


static seg_type ClassNameType( dir_node * dir, char *name )
/*****************************************/
{
    int     slen;
    char    uname[257];

    if (dir->e.seginfo->segrec->d.segdef.align == SEGDEF_ALIGN_ABS)
        return( SEGTYPE_ABS );

    if( name == NULL )
        return( SEGTYPE_UNDEF );

    if( ModuleInfo.model != MOD_NONE ) {
        if( stricmp( name, Options.code_class ) == 0 ) {
            return( SEGTYPE_CODE );
        }
    }
    slen = strlen( name );
    strcpy( uname, name );
    strupr( uname );
    switch( slen ) {
    default:
    case 5:
        // 'CONST'
        if( memcmp( uname, "CONST", 6 ) == 0 )
            return( SEGTYPE_DATA );
        // 'STACK'
        if( memcmp( uname, "STACK", 6 ) == 0 )
            return( SEGTYPE_DATA );
        // 'DBTYP'
        if( memcmp( uname, "DBTYP", 6 ) == 0 )
            return( SEGTYPE_DATA );
        // 'DBSYM'
        if( memcmp( uname, "DBSYM", 6 ) == 0 )
            return( SEGTYPE_DATA );
    case 4:
        // 'CODE'
        if( memcmp( uname , "CODE", 5 ) == 0 )
            return( SEGTYPE_CODE );
        // '...DATA'
        if( memcmp( uname + slen - 4, "DATA", 4 ) == 0 )
            return( SEGTYPE_DATA );
    case 3:
        // '...BSS'
        if( memcmp( uname + slen - 3, "BSS", 3 ) == 0 )
            return( SEGTYPE_BSS );
    case 2:
    case 1:
    case 0:
        return( SEGTYPE_UNDEF );
    }
}

/* get the type of the segment by checking it's name */
/* this is called only if the class gives no hint */

static seg_type SegmentNameType( char *name )
/*******************************************/
{
    int     slen;
    char    uname[257];

    slen = strlen( name );
    strcpy( uname, name );
    strupr( uname );
    switch( slen ) {
    default:
    case 5:
        // '..._TEXT'
        if( memcmp( uname + slen - 5, SegmNames[SIM_CODE], 5 ) == 0 )
            return( SEGTYPE_CODE );
        // '..._DATA'
        if( memcmp( uname + slen - 5, SegmNames[SIM_DATA], 5 ) == 0 )
            return( SEGTYPE_DATA );
        // 'CONST'
        if( memcmp( uname + slen - 5, SegmNames[SIM_CONST], 5 ) == 0 )
            return( SEGTYPE_DATA );
    case 4:
        // '..._BSS'
        if( memcmp( uname + slen - 4, "_BSS", 4 ) == 0 )
            return( SEGTYPE_BSS );
    case 3:
    case 2:
    case 1:
    case 0:
        return( SEGTYPE_UNDEF );
    }
}

// SEGMENT and ENDS directives (Pass ONE only!)

ret_code SegDef( int i )
/*****************/
{
    char                defined;
    char                *token;
    obj_rec             *seg;
//    obj_rec             *oldobj;
    direct_idx          classidx;
    uint                type;           // type of option
    uint                initstate = 0;  // to show if a field is initialized
    char                oldreadonly;    // readonly value of a defined segment
    char                oldsegtype;     // iscode value of a defined segment
    char                olduse32;
    char                oldalign;
    char                oldcombine;
    uint                oldclassidx;
    dir_node            *dir;
    char                *name;
    struct asm_sym      *sym;
    expr_list           opndx;

    if( i < 0 ) {
        AsmError( SEG_NAME_MISSING );
        return( ERROR );
    }

    name = AsmBuffer[i]->string_ptr;

    DebugMsg(("SegDef enter, segment=%s, cmd=%s\n", name, AsmBuffer[i+1]->string_ptr));

    switch( AsmBuffer[i+1]->value ) {
    case T_SEGMENT:
        /* Check to see if the segment is already defined */
        sym = SymSearch( name );
        if( sym == NULL || sym->state == SYM_UNDEFINED) {
            // segment is not defined (yet)
            if (sym == NULL)
                sym = (asm_sym *)dir_insert( name, TAB_SEG );
            else
                dir_change((dir_node *)sym, TAB_SEG);
            dir = (dir_node *)sym;
            seg = dir->e.seginfo->segrec;
            seg->d.segdef.idx = ++segdefidx;
            defined = FALSE;
            if (Options.output_format == OFORMAT_COFF ||
                Options.output_format == OFORMAT_ELF) {
                char * p;
                if (p = strchr(sym->name, '$')) {
                    char buffer[MAX_LINE_LEN];
                    dir_node *dir2;
                    memcpy(buffer, sym->name, p - sym->name);
                    buffer[p - sym->name] = NULLC;
                    if ((dir2 = (dir_node *)SymSearch(buffer)) && dir2->sym.state == SYM_SEG) {
                        dir->e.seginfo->readonly = dir2->e.seginfo->readonly;
                        dir->e.seginfo->segtype  = dir2->e.seginfo->segtype;
                        dir->e.seginfo->Use32    = dir2->e.seginfo->Use32;
                        dir->e.seginfo->segrec->d.segdef.align          = dir2->e.seginfo->segrec->d.segdef.align;
                        dir->e.seginfo->segrec->d.segdef.combine        = dir2->e.seginfo->segrec->d.segdef.combine;
                        dir->e.seginfo->segrec->d.segdef.class_name_idx = dir2->e.seginfo->segrec->d.segdef.class_name_idx;
                    }
                }
            }
        } else if ( sym->state == SYM_SEG ) {
            // segment already defined
            dir = (dir_node *)sym;
            seg = dir->e.seginfo->segrec;
            defined = TRUE;
            oldreadonly = dir->e.seginfo->readonly;
            oldsegtype  = dir->e.seginfo->segtype;
            olduse32    = dir->e.seginfo->Use32;
            oldalign    = dir->e.seginfo->segrec->d.segdef.align;
            oldcombine  = dir->e.seginfo->segrec->d.segdef.combine;
            oldclassidx = dir->e.seginfo->segrec->d.segdef.class_name_idx;
            if( dir->e.seginfo->lname_idx == 0 ) {
                // segment was mentioned in a group statement, but not really set up
                defined = FALSE;
                seg->d.segdef.idx = ++segdefidx;
            }
        } else {
            // symbol is different kind, error
            DebugMsg(("SegDef: symbol redefinition\n"));
            AsmErr( SYMBOL_REDEFINITION, name );
            return( ERROR );
        }
#if 0
        if( lastseg.stack_size > 0 ) {
            seg->d.segdef.seg_length = lastseg.stack_size;
            lastseg.stack_size = 0;
        }
#endif
        i += 2; /* go past segment name and "SEGMENT " */

        for( ; i < Token_Count; i ++ ) {
            DebugMsg(("SegDef: i=%u, string=%s token=%X\n", i, AsmBuffer[i]->string_ptr, AsmBuffer[i]->token));
            if( AsmBuffer[i]->token == T_STRING ) {

                /* the class name - the only token which is of type STRING */
                token = AsmBuffer[i]->string_ptr;
                /* string must be delimited by [double]quotes */
                if (AsmBuffer[i]->string_delim == '<' ||
                    (*token != '"' && *token != '\'')) {
                    AsmErr( SYNTAX_ERROR_EX, token );
                    continue;
                }
                /* remove the quote delimiters */
                token++;
                *(token+AsmBuffer[i]->value) = NULLC;

                classidx = FindLnameIdx( token );
                if( classidx == LNAME_NULL ) {
                    classidx = InsertClassLname( token );
                    if( classidx == LNAME_NULL ) {
                        return( ERROR );
                    }
                }
                seg->d.segdef.class_name_idx = classidx;
                continue;
            }

            /* go through all tokens EXCEPT the class name */
            token = AsmBuffer[i]->string_ptr;

            /* look up the type of token.
             check readonly, align, combine and word size types */
            type = token_cmp( token, TOK_READONLY, TOK_USE32 );
            if( type == ERROR ) {
                type = token_cmp(token, TOK_FLAT, TOK_FLAT );
                if( type == ERROR ) {
                    AsmErr( UNKNOWN_SEGMENT_ATTRIBUTE, token );
                    continue;
                }
            }

            /* initstate is used to check if any field is already
            initialized */

            if( initstate & TypeInfo[type].init ) {
                AsmErr( SEGMENT_ATTRIBUTE_DEFINED_ALREADY, token );
                continue;
            } else {
                initstate |= TypeInfo[type].init; // mark it initialized
            }
            switch( type ) {
            case TOK_READONLY:
                dir->e.seginfo->readonly = TRUE;
                break;
            case TOK_BYTE:
            case TOK_WORD:
            case TOK_DWORD:
            case TOK_PARA:
            case TOK_PAGE:
                seg->d.segdef.align = TypeInfo[type].value;
                break;
            case TOK_PRIVATE:
            case TOK_PUBLIC:
            case TOK_STACK:
            case TOK_COMMON:
            case TOK_MEMORY:
                seg->d.segdef.combine = TypeInfo[type].value;
                break;
            case TOK_AT:
                seg->d.segdef.combine = TypeInfo[type].value;
                seg->d.segdef.align = SEGDEF_ALIGN_ABS;
                i++;
                if (EvalOperand( &i, Token_Count, &opndx, TRUE ) == ERROR)
                    break;
                if (opndx.type == EXPR_CONST && opndx.string == NULL) {
                    seg->d.segdef.abs.frame = opndx.value;
                    seg->d.segdef.abs.offset = 0;
                } else {
                    AsmError( CONSTANT_EXPECTED );
                }
                break;
            case TOK_USE16:
            case TOK_USE32:
                dir->e.seginfo->Use32 = TypeInfo[type].value;
                break;
            case TOK_FLAT:
                DefineFlatGroup();
                dir->e.seginfo->Use32 = 1;
                dir->e.seginfo->group = &flat_grp->sym;
                break;
            default:
                AsmErr( UNKNOWN_SEGMENT_ATTRIBUTE, token );
                break;
            }
        } /* end for */

        token = GetLname( seg->d.segdef.class_name_idx );
        if( dir->e.seginfo->segtype != SEGTYPE_CODE ) {
            seg_type res;

            res = ClassNameType( dir, token );
            if( res != SEGTYPE_UNDEF ) {
                dir->e.seginfo->segtype = res;
            } else {
                res = SegmentNameType( name );
                dir->e.seginfo->segtype = res;
            }
        }

        if( defined ) {
            char * p = NULL;
            /* Check if new definition is different from previous one */

            // oldobj = dir->e.seginfo->segrec;
            if(  oldreadonly      != dir->e.seginfo->readonly )
                p = "readonly";
            else if ( oldalign    != seg->d.segdef.align )
                p = "alignment";
            else if ( oldcombine  != seg->d.segdef.combine )
                p = "combine";
            else if ( olduse32    != dir->e.seginfo->Use32 )
                p = "segment word size";
            else if ( oldclassidx != seg->d.segdef.class_name_idx )
                p = "class";

            if (p) {
                DebugMsg(("seg attr changed: %s, %s\n", dir->sym.name, p));
                AsmErr( SEGDEF_CHANGED, dir->sym.name, p);
                return( ERROR );
            }
            // definition is the same as before
            if( push_seg( dir ) == ERROR ) {
                return( ERROR );
            }
        } else {
            /* A new definition */
            if( push_seg( dir ) == ERROR ) {
                return( ERROR );
            }
            sym = &dir->sym;
            sym->defined = TRUE;
            SetSymSegOfs( sym );
            sym->offset = 0;
            if( dir->e.seginfo->lname_idx == 0 ) {
                dir->e.seginfo->lname_idx = ++LnamesIdx;
            }
            AddLnameData( dir );
        }

        if( CurrSeg != NULL ) {
            if( !ModuleInfo.mseg
                && ( CurrSeg->seg->e.seginfo->Use32 != ModuleInfo.use32 ) ) {
                ModuleInfo.mseg = TRUE;
            }
            if( CurrSeg->seg->e.seginfo->Use32 ) {
                ModuleInfo.use32 = TRUE;
            }
        }
        break;
    case T_ENDS:
        if (CloseSeg(name) == ERROR)
            return( ERROR );
        break;
    default:
        return( ERROR );
    }
    return( SetUse32() );
}

static void input_group( int type, char *name )
/*********************************************/
/* for simplified segment directives, emit DGROUP GROUP instruction */
{
    char        buffer[MAX_LINE_LEN];

    /* no DGROUP for FLAT or COFF/ELF */
    if( ModuleInfo.model == MOD_FLAT ||
        Options.output_format == OFORMAT_COFF ||
        Options.output_format == OFORMAT_ELF)
        return;

    strcpy( buffer, "DGROUP GROUP " );
    if( name != NULL ) {
        strcat( buffer, name );
    } else {
        switch( type ) {
        case T_DOT_DATA:
            strcat( buffer, SegmNames[SIM_DATA] );
            break;
        case T_DOT_DATA_UN:
            strcat( buffer, SegmNames[SIM_DATA_UN] );
            break;
        case T_DOT_CONST:
            strcat( buffer, SegmNames[SIM_CONST] );
            break;
        case T_DOT_STACK:
            if (ModuleInfo.distance == STACK_FAR)
                return;
            strcat( buffer, SegmNames[SIM_STACK] );
            break;
        }
    }
    InputQueueLine( buffer );
}

static void close_lastseg( void )
/*******************************/
/* close the last opened simplified segment? */
/* sounds like a bad idea! the only segment which can be closed
 is the current one.
*/
{
#if 0
    if( lastseg.seg != SIM_NONE ) {
        lastseg.seg = SIM_NONE;
        if( ( CurrSeg == NULL ) && ( *lastseg.close != '\0' ) ) {
            AsmError( DONT_MIX_SIM_WITH_REAL_SEGDEFS );
            return;
        }
        InputQueueLine( lastseg.close );
    }
#else
    lastseg.seg = SIM_NONE;
    DebugMsg(("close_lastseg: current seg=%s\n", sym_CurSeg->string_ptr));
    if ( sym_CurSeg->string_ptr && (*sym_CurSeg->string_ptr != '\0') ) {
        char buffer[MAX_LINE_LEN];
        strcpy( buffer, sym_CurSeg->string_ptr );
        strcat( buffer, " ENDS" );
        InputQueueLine( buffer );
    }
#endif
}


static char * SetSimSeg(int segm, char * name, char * buffer)
{
    char *pAlign = "WORD";
    char *pAlignSt = "PARA";
    char *pUse = "";
    char *pCC = "CODE";

    if ( ModuleInfo.defUse32 ) {
        if ( ModuleInfo.model == MOD_FLAT)
            pUse = "FLAT";
        else
            pUse = "USE32";
        if (( ModuleInfo.curr_cpu & P_CPU_MASK ) <= P_386)
            pAlign = "DWORD";
        else
            pAlign = "PARA";
        pAlignSt = pAlign;
    }

    if (segm == SIM_STACK || segm == SIM_FARDATA || segm == SIM_FARDATA_UN)
        pAlign = pAlignSt;

    if (Options.code_class)
        pCC = Options.code_class;

    if (name == NULL) {
        if (first_issued & (1 << segm))
            sprintf(buffer, "%s SEGMENT", SegmNames[segm]);
        else {
            first_issued |= (1 << segm);
            sprintf(buffer, SimCodeBegin[segm], SegmNames[segm], pAlign, pUse, pCC);
        }
    } else {
        asm_sym * sym = SymSearch(name);
        if (sym && sym->state == SYM_SEG)
            sprintf(buffer, "%s SEGMENT", name);
        else
            sprintf(buffer, SimCodeBegin[segm], name, pAlign, pUse, pCC);
    }
    return(buffer);
}

static char * EndSimSeg(int segm, char * buffer)
{
    sprintf(buffer, "%s ENDS", SegmNames[segm]);
    return(buffer);
}

static char *get_sim_code_end( char *buffer, char *name )
/*******************************************************/
{
    strcpy( buffer, name );
    strcat( buffer, " ENDS");
    return( buffer );
}


int SimplifiedSegDir( int i )
/*****************/
/* Handles simplified segment, based on optasm pg. 142-146 */
{
    char        buffer[ MAX_LINE_LEN ];
    unsigned    size;
    char        *string;
    int         type;
    int         seg;

    DebugMsg(("SimplifiedSegDir(%u) enter\n", i ));

    if( ModuleInfo.model == MOD_NONE ) {
        AsmError( MODEL_IS_NOT_DECLARED );
        return( ERROR );
    }
    ModuleInfo.cmdline = FALSE;

    PushLineQueue();

    type = AsmBuffer[i]->value;

    if( type != T_DOT_STACK ) {
        close_lastseg();  /* emit a "xxx ENDS" line to close current seg */
    }

    buffer[0] = NULLC;
    i++; /* get past the directive token */
    if( i < Token_Count && type != T_DOT_STACK ) {
        /* the segment name might begin with a dot.
         Then use ->pos instead of ->string_ptr!
         */
        if ( AsmBuffer[i]->token == T_DOT &&
             is_valid_id_char( *(AsmBuffer[i]->pos+1) ) ) {
            string = AsmBuffer[i]->pos;
            i++;
        } else
            string = AsmBuffer[i]->string_ptr;
        i++;
    } else {
        string = NULL;
    }

    switch( type ) {
    case T_DOT_CODE:
        if( string == NULL )
            string = Options.text_seg;
        InputQueueLine( SetSimSeg(SIM_CODE, string, buffer) );
        get_sim_code_end( lastseg.close, string );
        lastseg.seg = SIM_CODE;

        if( ModuleInfo.model == MOD_TINY ) {
            InputQueueLine( "ASSUME CS:DGROUP" );
        } else if( ModuleInfo.model == MOD_FLAT ) {
            //if ( SegAssumeTable[ASSUME_CS].flat == FALSE || SegAssumeTable[ASSUME_CS].error == TRUE )
                InputQueueLine( "ASSUME CS:FLAT" );
        } else {
            strcpy( buffer, "ASSUME CS:" );
            strcat( buffer, string );
            InputQueueLine( buffer );
        }
        break;
    case T_DOT_STACK:
        size = DEFAULT_STACK_SIZE;
        if( i < Token_Count ) {
            expr_list opndx;
            if (EvalOperand( &i, Token_Count, &opndx, TRUE ) == ERROR)
                return(ERROR);
            if( opndx.type != EXPR_CONST || opndx.string != NULL ) {
                AsmError( CONSTANT_EXPECTED );
                return( ERROR );
            }
            size = opndx.value;
        }
        InputQueueLine( SetSimSeg( SIM_STACK, NULL, buffer ));
        /* add stack to dgroup for segmented models */
        input_group( type, NULL );
        InputQueueLine( "ORG 0" );
        sprintf( buffer, "db %u dup (?)", size );
        InputQueueLine( buffer );
        InputQueueLine( EndSimSeg( SIM_STACK, buffer ));
        break;
    case T_DOT_DATA:
    case T_DOT_DATA_UN:             // .data?
    case T_DOT_CONST:
        if( type == T_DOT_DATA ) {
            if( string == NULL )
                string = Options.data_seg;
            seg = SIM_DATA;
        } else if( type == T_DOT_DATA_UN ) {
            seg = SIM_DATA_UN;
            string = NULL;
        } else {
            seg = SIM_CONST;
            string = NULL;
        }

        InputQueueLine( SetSimSeg(seg, string, buffer));
        InputQueueLine( "ASSUME CS:ERROR" );
        input_group( type, string );
        lastseg.seg = seg;
        if( string != NULL ) {
            strcpy( lastseg.close, string );
            strcat( lastseg.close, " ENDS" );
        } else {
            strcpy( lastseg.close, EndSimSeg(seg, buffer) );
        }
        break;
    case T_DOT_FARDATA:
    case T_DOT_FARDATA_UN:  // .fardata?
        seg = ( type == T_DOT_FARDATA ) ? SIM_FARDATA : SIM_FARDATA_UN;

        InputQueueLine( SetSimSeg(seg, string, buffer));
        InputQueueLine( "ASSUME CS:ERROR" );
        if( string != NULL ) {
            strcpy( lastseg.close, string );
            strcat( lastseg.close, " ENDS" );
        } else {
            strcpy( lastseg.close, EndSimSeg(seg, buffer));
        }
        lastseg.seg = seg;
        break;
    default:
        /**/myassert( 0 );
        break;
    }
    if (AsmBuffer[i]->token != T_FINAL)
        AsmError( SYNTAX_ERROR );
    DebugMsg(("SimSeg exit\n"));
    return( NOT_ERROR );
}

void set_def_seg_name( void )
/***********************************/
{
    int len;
    char *p;

    /* set Options.code_class */
    if( Options.code_class == NULL ) {
        Options.code_class = AsmAlloc( sizeof( DEFAULT_CODE_CLASS ) + 1 );
        strcpy( Options.code_class, DEFAULT_CODE_CLASS );
    }
    /* set Options.text_seg based on module name */
    if( Options.text_seg == NULL ) {
        switch( ModuleInfo.model ) {
        case MOD_MEDIUM:
        case MOD_LARGE:
        case MOD_HUGE:
            len = strlen( ModuleInfo.name ) + strlen( SegmNames[SIM_CODE] ) + 1;
            Options.text_seg = AsmAlloc( len );
            strcpy( Options.text_seg, ModuleInfo.name );
            strcat( Options.text_seg, SegmNames[SIM_CODE] );
            break;
        default:
            p = SegmNames[SIM_CODE];

            Options.text_seg = AsmAlloc( strlen(p) + 1 );
            strcpy( Options.text_seg, p );
            break;
        }
    }
    /* set Options.data_seg */
    if( Options.data_seg == NULL ) {
        p = SegmNames[SIM_DATA];

        Options.data_seg = AsmAlloc( strlen( p ) + 1 );
        strcpy( Options.data_seg, p );
    }
    return;
}

void DefineFlatGroup( void )
/***********************/
{

    if( ModuleInfo.flatgrp_idx == 0 ) {
        flat_grp = CreateGroup( "FLAT" );
        if( flat_grp != NULL ) {
            ModuleInfo.flatgrp_idx = GetGrpIdx( &flat_grp->sym );
        }
    }
}

uint GetSegIdx( struct asm_sym *sym )
/*************************************/
/* get idx to sym's segment */
{
    if( sym == NULL )
        return( 0 );
    if( ((dir_node *)sym)->e.seginfo->segrec != NULL )
        return( ((dir_node *)sym)->e.seginfo->segrec->d.segdef.idx );
    return( 0 );
}

uint GetGrpIdx( struct asm_sym *sym )
/***********************************/
/* get index of sym's group */
{
    if( sym == NULL )
        return( 0 );
    return( ((dir_node *)sym)->e.grpinfo->idx );
}

extern struct asm_sym *GetGrp( struct asm_sym *sym )
/**************************************************/
/* get ptr to sym's group sym */
{
    dir_node            *curr;

    curr = GetSeg( sym );
    if( curr != NULL )
        return( curr->e.seginfo->group );
    return( NULL );
}

int SymIs32( struct asm_sym *sym )
/********************************/
/* get sym's offset size (32=1, 16=0) */
{
    dir_node            *curr;

    curr = GetSeg( sym );
    if( curr == NULL ) {
        if( sym->state == SYM_EXTERNAL && sym->mem_type != MT_ABS) {
            if( ModuleInfo.mseg ) {
                return( sym->use32);
            } else {
                return( ModuleInfo.use32 );
            }
        } else if( sym->state == SYM_PROC)
            return(sym->use32);
    } else if( curr->e.seginfo->segrec != NULL ) {
        return( curr->e.seginfo->Use32 );
    }
    return( 0 );
}

int FixOverride( int index )
/**************************/
/* Fix segment or group override */
{
    struct asm_sym      *sym;

    sym = SymLookup( AsmBuffer[index]->string_ptr );
    /**/myassert( sym != NULL );
    if( sym->state == SYM_GRP ) {
        SegOverride = sym;
        return( NOT_ERROR );
    } else if( sym->state == SYM_SEG ) {
        SegOverride = sym;
        return( NOT_ERROR );
    }
    AsmError( SYNTAX_ERROR );
    return( ERROR );
}

void SetSymSegOfs( struct asm_sym *sym )
/*******************************************/
{
    sym->segment = &GetCurrSeg()->sym;
    sym->offset = GetCurrOffset();
}

// called for .MODEL directive
// allowes to use simplified segment directives
// PushLineQueue() has already been called

int SegmentModulePrologue(int type)
{
    char        buffer[ MAX_LINE_LEN ];

    lastseg.seg = SIM_NONE;
    //lastseg.stack_size = 0;

    if (Parse_Pass == PASS_1) {
        /* Generates codes for code segment */
        InputQueueLine( SetSimSeg( SIM_CODE, Options.text_seg, buffer ) );
        InputQueueLine( get_sim_code_end( buffer, Options.text_seg ) );

        /* Generates codes for data segment */
        InputQueueLine( SetSimSeg( SIM_DATA, NULL, buffer )) ;
        InputQueueLine( EndSimSeg( SIM_DATA, buffer));

        /* create DGROUP for BIN/OMF if model isn't FLAT */
        if( type != MOD_FLAT &&
            (Options.output_format == OFORMAT_OMF ||
             Options.output_format == OFORMAT_BIN)) {
            /* Generates codes for grouping */
            strcpy( buffer, "DGROUP GROUP " );
            if( type == MOD_TINY ) {
                strcat( buffer, Options.text_seg );
                strcat( buffer, "," );
            }
            strcat( buffer, Options.data_seg );
            InputQueueLine( buffer );
        }
    }
    return(NOT_ERROR);
}

// END directive has been found, close all segments
//

int SegmentModuleEnd(void)
{
    if( lastseg.seg != SIM_NONE ||
        (CurrSeg != NULL && ModuleInfo.model != MOD_NONE)) {
        DebugMsg(("SegmentModuleEnd: segment index %u must be closed\n", lastseg.seg));
    } else if (CurrSeg) {
        AsmErr(BLOCK_NESTING_ERROR, CurrSeg->seg->sym.name);
    }

    /* clear segment stack */

    while(CurrSeg && (CloseSeg(CurrSeg->seg->sym.name) == NOT_ERROR));

    return(NOT_ERROR);
}

// init. called for each pass

void SegmentInit( int pass )
{
    dir_node    *curr;
    uint_32     i;
    struct asmfixup *fix;

    first_issued = 0;

    if ( pass == PASS_1 ) {
        segdefidx   = 0;
        grpdefidx   = 0;
        LnamesIdx   = 1; /* the first Lname is a null-string */
    }

    /* Reset length of all segments to zero */
    for( curr = Tables[TAB_SEG].head; curr; curr = curr->next ) {
        if( ( curr->sym.state != SYM_SEG ) || ( curr->sym.segment == NULL ) )
            continue;
        if (curr->e.seginfo->CodeBuffer == NULL) {
            switch (Options.output_format) {
            case OFORMAT_OMF:
                curr->e.seginfo->CodeBuffer = codebuf;
                break;
            default: /* COFF, ELF, BIN */
                /* the segment can grow in step 2-n due to jump
                  modifications. worst case is no_of_short_jumps * 3 for 32bit.
                  for a quick solution just add 25% to the size if segment
                  contains labels */
                i = curr->e.seginfo->segrec->d.segdef.seg_length;
                if (curr->e.seginfo->labels)
                    i = i + (i >> 2);
                curr->e.seginfo->CodeBuffer = AsmAlloc(i);
                break;
            }
        }
        if( curr->e.seginfo->segrec->d.segdef.combine != COMB_STACK ) {
            curr->e.seginfo->segrec->d.segdef.seg_length = 0;
        }
        curr->e.seginfo->start_loc = 0;
        curr->e.seginfo->current_loc = 0;
#if BIN_SUPPORT
        curr->e.seginfo->initial = FALSE;
#endif

        if (Options.output_format == OFORMAT_COFF) {
            curr->e.seginfo->FixupListHeadCoff = NULL;
            curr->e.seginfo->FixupListTailCoff = NULL;
        } else if(Options.output_format == OFORMAT_ELF) {
            curr->e.seginfo->FixupListHeadElf = NULL;
            curr->e.seginfo->FixupListTailElf = NULL;
        }
    }

    Use32 = FALSE;

#if FASTPASS
    if ( pass != PASS_1 && UseSavedState == TRUE ) {
        i = saved_NumSegs;
        while(i) {
            i--;
            push_seg(*(saved_SegStack+i));
        }

        memcpy(&lastseg, &saved_lastseg, sizeof(last_seg_info));
        sym_CurSeg->string_ptr = saved_CurSeg_name;
        first_issued           = saved_first_issued;
        Use32                  = saved_Use32;
    }
#endif
}
#if FASTPASS
void SegmentSaveState( void )
{
    int i;
    seg_item * curr;

    for (i = 0, curr = CurrSeg;curr;curr = curr->next, i++);

    saved_NumSegs = i;
    if (i) {
        saved_SegStack = AsmAlloc(i*sizeof(dir_node *));
        for (i = 0, curr = CurrSeg;curr;curr = curr->next, i++)
            *(saved_SegStack+i) = curr->seg;
    }

    memcpy(&saved_lastseg, &lastseg, sizeof(last_seg_info));
    saved_CurSeg_name = sym_CurSeg->string_ptr;
    saved_first_issued = first_issued;
    saved_Use32        = Use32;
}
#endif
