/****************************************************************************
*
*                            Open Watcom Project
*
*    Portions Copyright (c) 1983-2002 Sybase, Inc. All Rights Reserved.
*
*  ========================================================================
*
*    This file contains Original Code and/or Modifications of Original
*    Code as defined in and that are subject to the Sybase Open Watcom
*    Public License version 1.0 (the 'License'). You may not use this file
*    except in compliance with the License. BY USING THIS FILE YOU AGREE TO
*    ALL TERMS AND CONDITIONS OF THE LICENSE. A copy of the License is
*    provided with the Original Code and Modifications, and is also
*    available at www.sybase.com/developer/opensource.
*
*    The Original Code and all software distributed under the License are
*    distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
*    EXPRESS OR IMPLIED, AND SYBASE AND ALL CONTRIBUTORS HEREBY DISCLAIM
*    ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF
*    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR
*    NON-INFRINGEMENT. Please see the License for the specific language
*    governing rights and limitations under the License.
*
*  ========================================================================
*
* Description:  Processing of segment and group related directives.
*
****************************************************************************/


#include "asmglob.h"
#include <ctype.h>

#include "memalloc.h"
#include "parser.h"
#include "symbols.h"
#include "directiv.h"
#include "queues.h"
#include "equate.h"
#include "asmdefs.h"
#include "asmfixup.h"
#include "mangle.h"
#include "labels.h"
#include "asminput.h"
#include "expreval.h"
#include "types.h"
#include "condasm.h"
#include "hll.h"
#include "macro.h"

#include "myassert.h"

/* prototypes */
extern int              OrgDirective( int );
extern uint             checkword( char **token );

#define DEFAULT_STACK_SIZE      1024

enum {
#undef fix
#define fix( tok, str, val, init )              tok

#include "directd.h"
};

typedef enum {
    SIM_CODE = 0,
    SIM_STACK,
    SIM_DATA,
    SIM_DATA_UN,            // .DATA?
    SIM_FARDATA,
    SIM_FARDATA_UN,         // .FARDATA?
    SIM_CONST,
    SIM_NONE,
    SIM_LAST = SIM_NONE
} sim_seg;

typedef struct {
    sim_seg     seg;                    // segment id
    char        close[MAX_LINE];        // closing line for this segment
    int_16      stack_size;             // size of stack segment
} last_seg_info;        // information about last opened simplified segment

// strings used by simplified segment directives

static char *SimCodeBegin[ SIM_LAST ] = {
        "%s SEGMENT %s %s PUBLIC '%s' IGNORE",
        "%s SEGMENT %s %s STACK 'STACK' IGNORE",
        "%s SEGMENT %s %s PUBLIC 'DATA' IGNORE",
        "%s SEGMENT %s %s PUBLIC 'BSS' IGNORE",
        "%s SEGMENT %s %s PRIVATE 'FAR_DATA' IGNORE",
        "%s SEGMENT %s %s PRIVATE 'FAR_BSS' IGNORE",
        "%s SEGMENT %s %s PUBLIC 'CONST' READONLY IGNORE"
};

static char *SegmNames[ SIM_LAST ] = {
    "_TEXT", "STACK", "_DATA", "_BSS", "FAR_DATA", "FAR_BSS", "CONST"
};


extern  char            write_to_file;  // write if there is no error

extern  uint_32         BufSize;        // buffer for emitted bytes curr seg
extern struct asm_sym   *SegOverride;

seg_list                *CurrSeg;       // points to stack of opened segments
uint                    LnamesIdx;      // Number of LNAMES definition

extern uint             segdefidx;      // Number of Segment definition

#define DEFAULT_CODE_CLASS      "CODE"
#define DEFAULT_CODE_NAME       "_TEXT"
#define DEFAULT_DATA_NAME       "_DATA"
#define SIM_DATA_OFFSET         5        // strlen("_DATA")
#define SIM_FARDATA_OFFSET      8        // strlen("FAR_DATA")

static char *SimCodeEnd[ SIM_LAST ] = {
        "_TEXT ENDS",
        "STACK ENDS",
        "_DATA ENDS",
        "_BSS  ENDS",
        "FAR_DATA ENDS",
        "FAR_BSS  ENDS",
        "CONST ENDS"
};

/* Code generated by simplified segment definitions */

static last_seg_info    lastseg;        // last opened simplified segment

extern dir_node *const_CurSeg;

#define ROUND_UP( i, r ) (((i)+((r)-1)) & ~((r)-1))

static int SetAssumeCSCurrSeg( void )
/*************************************/
{
    assume_info     *info;

    info = &(SegAssumeTable[ ASSUME_CS ]);
    if( CurrSeg == NULL ) {
        info->symbol = NULL;
        info->flat = FALSE;
        info->error = TRUE;
        const_CurSeg->sym.string_ptr = "";
    } else {
        if( CurrSeg->seg->e.seginfo->group != NULL )
            info->symbol = GetGrp( &CurrSeg->seg->sym );
        else
            info->symbol = &CurrSeg->seg->sym;
        info->flat = FALSE;
        info->error = FALSE;
        const_CurSeg->sym.string_ptr = CurrSeg->seg->sym.name;
        DebugMsg(("SetAssumeCSCurrSeg: current segment=%s\n", CurrSeg->seg->sym.name));
    }
    return( NOT_ERROR );
}

static int push_seg( dir_node *seg )
/**********************************/
/* Push a segment into the current segment stack */
{
    seg_list    *curr;

    for( curr = CurrSeg; curr; curr = curr->next ) {
        if( curr->seg == seg ) {
            AsmError( BLOCK_NESTING_ERROR );
            return( ERROR );
        }
    }
    push( &CurrSeg, seg );
    SetAssumeCSCurrSeg();
    return( NOT_ERROR );
}

static dir_node *pop_seg( void )
/******************************/
/* Pop a segment out of the current segment stack */
{
    dir_node    *seg;

    /**/myassert( CurrSeg != NULL );
    seg = pop( &CurrSeg );
    SetAssumeCSCurrSeg();
    return( seg );
}

direct_idx GetLnameIdx( char *name )
/**********************************/
{
    struct asm_sym      *sym;
    dir_node            *dir;

    sym = SymSearch( name );
    if( sym == NULL )
        return( LNAME_NULL );

    dir = (dir_node *)sym;
    if( sym->state == SYM_UNDEFINED ) {
        return( LNAME_NULL );
    } else if( sym->state == SYM_GRP ) {
        return( dir->e.grpinfo->lname_idx );
    } else if( sym->state == SYM_SEG ) {
        return( dir->e.seginfo->lname_idx );
    } else {    /* it is an lname record */
        return( dir->e.lnameinfo->idx );
    }
}

static direct_idx InsertClassLname( char *name )
/**********************************************/
{
    dir_node            *dir;

    if( strlen( name ) > MAX_LNAME ) {
        AsmError( LNAME_TOO_LONG );
        return( LNAME_NULL );
    }

    dir = dir_insert( name, TAB_CLASS_LNAME );

    /* put it into the lname table */

    AddLnameData( dir );

    return( LnamesIdx );
}

direct_idx LnameInsert( char *name )
/**********************************/
{
    struct asm_sym      *sym;
    dir_node            *dir;

    sym = SymSearch( name );

    if( sym != NULL ) {
        dir = (dir_node *)sym;
        /* The name is already in the table*/
        if( sym->state == SYM_GRP ) {
            if( dir->e.grpinfo->lname_idx == 0 ) {
                dir->e.grpinfo->lname_idx = ++LnamesIdx;
            }
        } else if( sym->state == SYM_SEG ) {
            if( dir->e.seginfo->lname_idx == 0 ) {
                dir->e.seginfo->lname_idx = ++LnamesIdx;
            }
        } else {
            return( ERROR );
        }
    } else {
        if( strlen( name ) > MAX_LNAME ) {
            AsmError( LNAME_TOO_LONG );
            return( ERROR );
        }

        dir = dir_insert( name, TAB_LNAME );
    }

    /* put it into the lname table */

    AddLnameData( dir );

    return( LnamesIdx );
}

uint_32 GetCurrAddr( void )
/*************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    return( CurrSeg->seg->e.seginfo->current_loc );
}

dir_node *GetCurrSeg( void )
/**************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    return( CurrSeg->seg );
}

uint_32 GetCurrSegAlign( void )
/*****************************/
{
    if( CurrSeg == NULL )
        return( 0 );
    switch( CurrSeg->seg->e.seginfo->segrec->d.segdef.align ) {
    case ALIGN_ABS: // same as byte aligned ?
    case ALIGN_BYTE:
        return( 1 );
    case ALIGN_WORD:
        return( 2 );
    case ALIGN_DWORD:
        return( 4 );
    case ALIGN_PARA:
        return( 16 );
    case ALIGN_PAGE:
        return( 256 );
    case ALIGN_4KPAGE:
        return( 4096 );
    default:
        return( 0 );
    }
}

uint_32 GetCurrSegStart( void )
/*****************************/
{
    if( CurrSeg == NULL )
        return( 0 );
#if 0
    /**/myassert( !write_to_file
                || ( CurrSeg->seg->e.seginfo->current_loc - BufSize )
                  == CurrSeg->seg->e.seginfo->start_loc );
#endif
    return( CurrSeg->seg->e.seginfo->start_loc );
}





static dir_node *CreateGroup( char *name )
/****************************************/
{
    dir_node    *grp;

    grp = (dir_node *)SymSearch( name );

    if( grp == NULL ) {
        grp = dir_insert( name, TAB_GRP );
        LnameInsert( name );
    } else if( grp->sym.state == SYM_UNDEFINED ) {
        dir_change( grp, TAB_GRP );
        LnameInsert( name );
    } else if( grp->sym.state != SYM_GRP ) {
        AsmErr( SYMBOL_PREVIOUSLY_DEFINED, name );
        grp = NULL;
    }
    return( grp );
}

int GrpDef( int i )
/*****************/
{
    char        *name;
    dir_node    *grp;
    seg_list    *new;
    seg_list    *curr;
    dir_node    *seg;

    if( i < 0 ) {
        AsmError( GRP_NAME_MISSING );
        return( ERROR );
    }

    name = AsmBuffer[i]->string_ptr;
    grp = CreateGroup( name );
    if( grp == NULL )
        return( ERROR );

    for( i+=2;              // skip over the GROUP directive
         i < Token_Count;   // stop at the end of the line
         i+=2 ) {           // skip over commas
        name = AsmBuffer[i]->string_ptr;
        /* Add the segment name */

        if( checkword( &name ) == ERROR ) {
            AsmError( EXPECTING_COMMA );
            return( ERROR );
        }

        seg = (dir_node *)SymSearch( name );
        if( seg == NULL ) {
            seg = dir_insert( name, TAB_SEG );
        } else if( seg->sym.state == SYM_UNDEFINED ) {
            dir_change( seg, TAB_SEG );
        } else if( seg->sym.state != SYM_SEG ) {
            AsmErr( SYMBOL_PREVIOUSLY_DEFINED, name );
            return( ERROR );
        }
        if( seg->e.seginfo->group == &grp->sym )    // segment is in group already
            continue;
        if( seg->e.seginfo->group != NULL ) {      // segment is in another group
            AsmError( SEGMENT_IN_GROUP );
            return( ERROR );
        }
        /* set the grp index of the segment */
        seg->e.seginfo->group = &grp->sym;

        new = AsmAlloc( sizeof(seg_list) );
        new->seg = seg;
        new->next = NULL;
        grp->e.grpinfo->numseg++;

        /* insert the segment at the end of linked list */
        if( grp->e.grpinfo->seglist == NULL ) {
            grp->e.grpinfo->seglist = new;
        } else {
            curr = grp->e.grpinfo->seglist;
            while( curr->next != NULL ) {
                curr = curr->next;
            }
            curr->next = new;
        }
    }
    return( NOT_ERROR );
}

static int SetUse32( void )
/*************************/
{
    if( CurrSeg == NULL ) {
        Use32 = ModuleInfo.defUse32;
    } else {
        Globals.code_seg = SEGISCODE( CurrSeg );
        Use32 = CurrSeg->seg->e.seginfo->segrec->d.segdef.use_32;
        if( Use32 && ( ( CodeInfo->info.cpu & P_CPU_MASK ) < P_386 ) ) {
            AsmError( WRONG_CPU_FOR_32BIT_SEGMENT );
            return( ERROR );
        }
    }
    return( NOT_ERROR );
}

// 32bit is set as default for FLAT memory model

int SetUse32Def( bool flag )
/**************************/
{
    if( ( CurrSeg == NULL )               // outside any segments
        && ( ModuleInfo.model == MOD_NONE             // model not defined
            || ModuleInfo.cmdline ) ) {   // model defined on cmdline by -m?
        ModuleInfo.defUse32 = flag;
    }
    return( SetUse32() );
}

// set current segment if pass is > 1

int  SetCurrSeg( int i )
/**********************/
{
    char        *name;
    struct asm_sym      *sym;

    name = AsmBuffer[i]->string_ptr;

    switch( AsmBuffer[i+1]->value ) {
    case T_SEGMENT:
        FlushCurrSeg();
        sym = SymSearch( name );
        DebugMsg(("SetCurrSeg(%s) sym=%X\n", name, sym));
        if ( sym == NULL ) {
            AsmErr(SEG_NOT_DEFINED, name);
            return( ERROR );
        }
        push_seg( (dir_node *)sym );
        break;
    case T_ENDS:
        FlushCurrSeg();
        pop_seg();
        break;
    default:
        break;
    }
    return( SetUse32() );
}


static seg_type ClassNameType( char *name )
/*****************************************/
{
    int     slen;
    char    uname[257];

    if( name == NULL )
        return( SEGTYPE_UNDEF );
    if( ModuleInfo.model != MOD_NONE ) {
        if( stricmp( name, Options.code_class ) == 0 ) {
            return( SEGTYPE_ISCODE );
        }
    }
    slen = strlen( name );
    strcpy( uname, name );
    strupr( uname );
    switch( slen ) {
    default:
    case 5:
        // 'CONST'
        if( memcmp( uname, "CONST", 6 ) == 0 )
            return( SEGTYPE_ISDATA );
        // 'STACK'
        if( memcmp( uname, "STACK", 6 ) == 0 )
            return( SEGTYPE_ISDATA );
        // 'DBTYP'
        if( memcmp( uname, "DBTYP", 6 ) == 0 )
            return( SEGTYPE_ISDATA );
        // 'DBSYM'
        if( memcmp( uname, "DBSYM", 6 ) == 0 )
            return( SEGTYPE_ISDATA );
    case 4:
        // 'CODE'
        if( memcmp( uname , "CODE", 5 ) == 0 )
            return( SEGTYPE_ISCODE );
        // '...DATA'
        if( memcmp( uname + slen - 4, "DATA", 4 ) == 0 )
            return( SEGTYPE_ISDATA );
    case 3:
        // '...BSS'
        if( memcmp( uname + slen - 3, "BSS", 3 ) == 0 )
            return( SEGTYPE_ISDATA );
    case 2:
    case 1:
    case 0:
        return( SEGTYPE_UNDEF );
    }
}

static seg_type SegmentNameType( char *name )
/*******************************************/
{
    int     slen;
    char    uname[257];

    slen = strlen( name );
    strcpy( uname, name );
    strupr( uname );
    switch( slen ) {
    default:
    case 5:
        // '..._TEXT'
        if( memcmp( uname + slen - 5, DEFAULT_CODE_NAME, 5 ) == 0 )
            return( SEGTYPE_ISCODE );
        // '..._DATA'
        if( memcmp( uname + slen - 5, DEFAULT_DATA_NAME, 5 ) == 0 )
            return( SEGTYPE_ISDATA );
        // 'CONST'
        if( memcmp( uname, "CONST", 5 ) == 0 )
            return( SEGTYPE_ISDATA );
    case 4:
        // '..._BSS'
        if( memcmp( uname + slen - 4, "_BSS", 4 ) == 0 )
            return( SEGTYPE_ISDATA );
    case 3:
    case 2:
    case 1:
    case 0:
        return( SEGTYPE_UNDEF );
    }
}

// SEGMENT and ENDS directives

int SegDef( int i )
/*****************/
{
    char                defined;
    char                *token;
    obj_rec             *seg;
    obj_rec             *oldobj;
    direct_idx          classidx;
    uint                type;           // type of option
    uint                initstate = 0;  // to show if a field is initialized
    char                oldreadonly;    // readonly value of a defined segment
    char                oldiscode;      // iscode value of a defined segment
    bool                ignore;
    dir_node            *dirnode;
    char                *name;
    struct asm_sym      *sym;
    expr_list           opndx;

    if( i < 0 ) {
        AsmError( SEG_NAME_MISSING );
        return( ERROR );
    }

    name = AsmBuffer[i]->string_ptr;

    DebugMsg(("SegDef enter, segment=%s, cmd=%s\n", name, AsmBuffer[i+1]->string_ptr));

    switch( AsmBuffer[i+1]->value ) {
    case T_SEGMENT:
        seg = ObjNewRec( CMD_SEGDEF );

        /* Check to see if the segment is already defined */
        sym = SymSearch( name );
        if( sym == NULL || sym->state == SYM_UNDEFINED) {
            // segment is not defined (yet)
            if (sym ==NULL)
                dirnode = dir_insert( name, TAB_SEG );
            else
                dir_change(dirnode, TAB_SEG);
            seg->d.segdef.idx = ++segdefidx;
            defined = FALSE;
            ignore = FALSE;
        } else if ( sym->state == SYM_SEG ) {
            // segment already defined
            dirnode = (dir_node *)sym;
            defined = TRUE;
            oldreadonly = dirnode->e.seginfo->readonly;
            oldiscode = dirnode->e.seginfo->iscode;
            ignore = dirnode->e.seginfo->ignore;
            if( dirnode->e.seginfo->lname_idx == 0 ) {
                // segment was mentioned in a group statement, but not really set up
                defined = FALSE;
                seg->d.segdef.idx = ++segdefidx;
            }
        } else {
            // symbol is different kind, error
            AsmErr( SYMBOL_REDEFINITION, name );
            return( ERROR );
        }

        /* Setting up default values */
        if( !defined ) {
            seg->d.segdef.align = ALIGN_PARA;
            seg->d.segdef.combine = COMB_INVALID;
            seg->d.segdef.use_32 = ModuleInfo.defUse32;
            seg->d.segdef.class_name_idx = 1;
            /* null class name, in case none is mentioned */
            dirnode->e.seginfo->readonly = FALSE;
            dirnode->e.seginfo->iscode = SEGTYPE_UNDEF;
        } else {
            oldobj = dirnode->e.seginfo->segrec;
            dirnode->e.seginfo->readonly = oldreadonly;
            dirnode->e.seginfo->iscode = oldiscode;
            seg->d.segdef.align = oldobj->d.segdef.align;
            seg->d.segdef.combine = oldobj->d.segdef.combine;
            if( !ignore ) {
                seg->d.segdef.use_32 = oldobj->d.segdef.use_32;
            } else {
                seg->d.segdef.use_32 = ModuleInfo.defUse32;
            }
            seg->d.segdef.class_name_idx = oldobj->d.segdef.class_name_idx;
        }
        dirnode->e.seginfo->ignore = FALSE; // always false unless set explicitly
        seg->d.segdef.access_valid = FALSE;
        seg->d.segdef.seg_length = 0;

        if( lastseg.stack_size > 0 ) {
            seg->d.segdef.seg_length = lastseg.stack_size;
            lastseg.stack_size = 0;
        }

        i+=2; /* go past segment name and "SEGMENT " */

        for( ; i < Token_Count; i ++ ) {
            DebugMsg(("SegDef: i=%u, string=%s token=%X\n", i, AsmBuffer[i]->string_ptr, AsmBuffer[i]->token));
            if( AsmBuffer[i]->token == T_STRING ) {

                /* the class name - the only token which is of type STRING */
                token = AsmBuffer[i]->string_ptr;
                /* remove the quote delimiters */
                if ((*token == '"') || (*token == '\'')) {
                    token++;
                    *(token+AsmBuffer[i]->value) = 0;
                }


                classidx = FindLnameIdx( token );
                if( classidx == LNAME_NULL ) {
                    classidx = InsertClassLname( token );
                    if( classidx == LNAME_NULL ) {
                        return( ERROR );
                    }
                }
                seg->d.segdef.class_name_idx = classidx;
                continue;
            }

            /* go through all tokens EXCEPT the class name */
            token = AsmBuffer[i]->string_ptr;

            // look up the type of token
            type = token_cmp( token, TOK_READONLY, TOK_AT );
            if( type == ERROR ) {
                type = token_cmp(token, TOK_FLAT, TOK_FLAT );
                if( type == ERROR ) {
                    AsmError( UNDEFINED_SEGMENT_OPTION );
                    return( ERROR );
                }
            }

            /* initstate is used to check if any field is already
            initialized */

            if( initstate & TypeInfo[type].init ) {
                AsmError( SEGMENT_PARA_DEFINED ); // initialized already
                return( ERROR );
            } else {
                initstate |= TypeInfo[type].init; // mark it initialized
            }
            switch( type ) {
            case TOK_READONLY:
                dirnode->e.seginfo->readonly = TRUE;
                break;
            case TOK_BYTE:
            case TOK_WORD:
            case TOK_DWORD:
            case TOK_PARA:
            case TOK_PAGE:
                seg->d.segdef.align = TypeInfo[type].value;
                break;
            case TOK_PRIVATE:
            case TOK_PUBLIC:
            case TOK_STACK:
            case TOK_COMMON:
            case TOK_MEMORY:
                seg->d.segdef.combine = TypeInfo[type].value;
                break;
            case TOK_USE16:
            case TOK_USE32:
                seg->d.segdef.use_32 = TypeInfo[type].value;
                break;
            case TOK_IGNORE:
                dirnode->e.seginfo->ignore = TRUE;
                break;
            case TOK_AT:
                seg->d.segdef.align = SEGDEF_ALIGN_ABS;
                i++;
                if ((EvalOperand( &i, Token_Count, &opndx, TRUE ) != ERROR) &&
                    opndx.type == EXPR_CONST && opndx.string == NULL) {
                    seg->d.segdef.abs.frame = AsmBuffer[i]->value;
                    seg->d.segdef.abs.offset = 0;
                } else {
                    AsmError( UNDEFINED_SEGMENT_OPTION );
                    return( ERROR );
                }
                break;
            case TOK_FLAT:
                seg->d.segdef.use_32 = 1;
                break;
            default:
                AsmError( UNDEFINED_SEGMENT_OPTION );
                return( ERROR );
            }
        }
        token = GetLname( seg->d.segdef.class_name_idx );
        if( dirnode->e.seginfo->iscode != SEGTYPE_ISCODE ) {
            seg_type res;

            res = ClassNameType( token );
            if( res != SEGTYPE_UNDEF ) {
                dirnode->e.seginfo->iscode = res;
            } else {
                res = SegmentNameType( name );
                dirnode->e.seginfo->iscode = res;
            }
        }

        if( defined && !ignore && !dirnode->e.seginfo->ignore ) {
            /* Check if new definition is different from previous one */

            oldobj = dirnode->e.seginfo->segrec;
            if( ( oldreadonly != dirnode->e.seginfo->readonly )
                || ( oldobj->d.segdef.align != seg->d.segdef.align )
                || ( oldobj->d.segdef.combine != seg->d.segdef.combine )
                || ( oldobj->d.segdef.use_32 != seg->d.segdef.use_32 )
                || ( oldobj->d.segdef.class_name_idx != seg->d.segdef.class_name_idx ) ) {
                ObjKillRec( seg );
                AsmError( SEGDEF_CHANGED );
                return( ERROR );
            } else {
                // definition is the same as before
                ObjKillRec( seg );
                if( push_seg( dirnode ) == ERROR ) {
                    return( ERROR );
                }
            }
        } else if( defined && ignore ) {
            /* reset to the new values */
            oldobj = dirnode->e.seginfo->segrec;
            oldobj->d.segdef.align = seg->d.segdef.align;
            oldobj->d.segdef.combine = seg->d.segdef.combine;
            oldobj->d.segdef.use_32 = seg->d.segdef.use_32;
            oldobj->d.segdef.class_name_idx = seg->d.segdef.class_name_idx;

            ObjKillRec( seg );
            if( push_seg( dirnode ) == ERROR ) {
                return( ERROR );
            }
        } else if( defined && dirnode->e.seginfo->ignore ) {
            /* keep the old values */
            dirnode->e.seginfo->readonly = oldreadonly;
            dirnode->e.seginfo->iscode = oldiscode;
            dirnode->e.seginfo->ignore = ignore;
            ObjKillRec( seg );
            if( push_seg( dirnode ) == ERROR ) {
                return( ERROR );
            }
        } else {
            /* A new definition */
            dirnode->e.seginfo->segrec = seg;
            dirnode->e.seginfo->start_loc = 0;
            dirnode->e.seginfo->current_loc = 0;
            if( push_seg( dirnode ) == ERROR ) {
                return( ERROR );
            }
            sym = &dirnode->sym;
            SetSymSegOfs( sym );
            sym->offset = 0;
            LnameInsert( name );
        }
        if( CurrSeg != NULL ) {
            if( !ModuleInfo.mseg
                && ( CurrSeg->seg->e.seginfo->segrec->d.segdef.use_32 != ModuleInfo.use32 ) ) {
                ModuleInfo.mseg = TRUE;
            }
            if( CurrSeg->seg->e.seginfo->segrec->d.segdef.use_32 ) {
                ModuleInfo.use32 = TRUE;
            }
        }
        break;
    case T_ENDS:
        if( CurrSeg == NULL ) {
            AsmError( SEGMENT_NOT_OPENED );
            return( ERROR );
        }

        sym = SymSearch( name );
        if( sym == NULL ) {
            AsmErr( SEG_NOT_DEFINED, name );
            return( ERROR );
        }
        if( (struct dir_node *)sym != CurrSeg->seg ) {
            AsmError( BLOCK_NESTING_ERROR );
            return( ERROR );
        }
        DebugMsg(("SegDef(%s) ENDS, current ofs=%X\n", name, CurrSeg->seg->e.seginfo->current_loc));
        pop_seg();
        break;
    default:
        return( ERROR );
    }
    return( SetUse32() );
}

static void input_group( int type, char *name )
/*********************************************/
/* emit any GROUP instruction */
{
    char        buffer[MAX_LINE_LEN];

    if( ModuleInfo.model == MOD_FLAT )
        return;
    strcpy( buffer, "DGROUP GROUP " );
    if( name != NULL ) {
        strcat( buffer, name );
    } else {
        switch( type ) {
        case T_DOT_DATA:
            strcat( buffer, DEFAULT_DATA_NAME );
            break;
        case T_DOT_DATA_UN:
            strcat( buffer, "_BSS" );
            break;
        case T_DOT_CONST:
            strcat( buffer, "CONST" );
            break;
        case T_DOT_STACK:
            strcat( buffer, "STACK" );
            break;
        }
    }
    InputQueueLine( buffer );
}

static void close_lastseg( void )
/*******************************/
/* close the last opened simplified segment? */
/* sounds like a bad idea! the only segment which can be closed
 is the current one.
*/
{
#if 0
    if( lastseg.seg != SIM_NONE ) {
        lastseg.seg = SIM_NONE;
        if( ( CurrSeg == NULL ) && ( *lastseg.close != '\0' ) ) {
            AsmError( DONT_MIX_SIM_WITH_REAL_SEGDEFS );
            return;
        }
        InputQueueLine( lastseg.close );
    }
#else
    lastseg.seg = SIM_NONE;
    if (const_CurSeg->sym.string_ptr && (*const_CurSeg->sym.string_ptr != '\0')) {
        char buffer[MAX_LINE_LEN];
        strcpy(buffer, const_CurSeg->sym.string_ptr);
        strcat(buffer, " ENDS");
        InputQueueLine( buffer );
    }
#endif
}


static char *get_sim_code_end( char *buffer, char *name )
/*******************************************************/
{
    strcpy( buffer, name );
    strcat( buffer, " ENDS");
    return( buffer );
}

char * SetSimSeg(int segm, char * name, char * buffer)
{
    char *pAlign = "WORD";
    char *pAlignSt = "PARA";
    char *pUse = "";
    char *pCC = "CODE";

    if ( ModuleInfo.defUse32 ) {
        pUse = "USE32";
        if ((CodeInfo->info.cpu & P_CPU_MASK ) <= P_386)
            pAlign = "DWORD";
        else
            pAlign = "PARA";
        pAlignSt = pAlign;
    }

    if (name == NULL)
        name = SegmNames[segm];

    if (segm == SIM_STACK || segm == SIM_FARDATA || segm == SIM_FARDATA_UN)
        pAlign = pAlignSt;

    if (Options.code_class)
        pCC = Options.code_class;

    sprintf(buffer, SimCodeBegin[segm], name, pAlign, pUse, pCC);

    return(buffer);
}

int SimSeg( int i )
/*****************/
/* Handles simplified segment, based on optasm pg. 142-146 */
{
    char        buffer[ MAX_LINE_LEN ];
    int         bit;
    char        *string;
    int         type;
    int         seg;

    if( ModuleInfo.model == MOD_NONE ) {
        AsmError( MODEL_IS_NOT_DECLARED );
        return( ERROR );
    }
    ModuleInfo.cmdline = FALSE;

    PushLineQueue();

    if( AsmBuffer[i]->value != T_DOT_STACK ) {
        close_lastseg();
    }
    buffer[0] = '\0';
    type = AsmBuffer[i]->value;
    i++; /* get past the directive token */
    if( i < Token_Count ) {
        string = AsmBuffer[i]->string_ptr;
    } else {
        string = NULL;
    }

    switch( type ) {
    case T_DOT_CODE:
        if( string == NULL )
            string = Options.text_seg;
        InputQueueLine( SetSimSeg(SIM_CODE, string, buffer) );
        get_sim_code_end( lastseg.close, string );
        lastseg.seg = SIM_CODE;

        if( ModuleInfo.model == MOD_TINY ) {
            InputQueueLine( "ASSUME CS:DGROUP" );
        } else {
            strcpy( buffer, "ASSUME CS:" );
            strcat( buffer, string );
            InputQueueLine( buffer );
        }
        break;
    case T_DOT_STACK:
        InputQueueLine( SetSimSeg(SIM_STACK, NULL, buffer));
        input_group( type, NULL );
        InputQueueLine( SimCodeEnd[SIM_STACK] );
        if( i < Token_Count ) {
            if( AsmBuffer[i]->token != T_NUM ) {
                AsmError( CONSTANT_EXPECTED );
                return( ERROR );
            } else {
                lastseg.stack_size = AsmBuffer[i]->value;
            }
        } else {
            lastseg.stack_size = DEFAULT_STACK_SIZE;
        }
        break;
    case T_DOT_DATA:
    case T_DOT_DATA_UN:             // .data?
    case T_DOT_CONST:
        if( type == T_DOT_DATA ) {
            if( string == NULL )
                string = Options.data_seg;
            seg = SIM_DATA;
        } else if( type == T_DOT_DATA_UN ) {
            seg = SIM_DATA_UN;
            string = NULL;
        } else {
            seg = SIM_CONST;
            string = NULL;
        }

        InputQueueLine( SetSimSeg(seg, string, buffer));
        InputQueueLine( "ASSUME CS:ERROR" );
        input_group( type, string );
        lastseg.seg = seg;
        if( string != NULL ) {
            strcpy( lastseg.close, string );
            strcat( lastseg.close, SimCodeEnd[seg] + SIM_DATA_OFFSET );
        } else {
            strcpy( lastseg.close, SimCodeEnd[seg] );
        }
        break;
    case T_DOT_FARDATA:
    case T_DOT_FARDATA_UN:  // .fardata?
        seg = ( type == T_DOT_FARDATA ) ? SIM_FARDATA : SIM_FARDATA_UN;

        InputQueueLine( SetSimSeg(seg, string, buffer));
        InputQueueLine( "ASSUME CS:ERROR" );
        if( string != NULL ) {
            strcpy( lastseg.close, string );
            strcat( lastseg.close, SimCodeEnd[seg] + SIM_FARDATA_OFFSET );
        } else {
            strcpy( lastseg.close, SimCodeEnd[seg] );
        }
        lastseg.seg = seg;
        break;
    default:
        /**/myassert( 0 );
        break;
    }
    return( NOT_ERROR );
}

void set_def_seg_name( void )
/***********************************/
{
    int len;

    /* set Options.code_class */
    if( Options.code_class == NULL ) {
        Options.code_class = AsmAlloc( sizeof( DEFAULT_CODE_CLASS ) + 1 );
        strcpy( Options.code_class, DEFAULT_CODE_CLASS );
    }
    /* set Options.text_seg based on module name */
    if( Options.text_seg == NULL ) {
        switch( ModuleInfo.model ) {
        case MOD_MEDIUM:
        case MOD_LARGE:
        case MOD_HUGE:
            len = strlen( ModuleInfo.name ) + sizeof( DEFAULT_CODE_NAME ) + 1;
            Options.text_seg = AsmAlloc( len );
            strcpy( Options.text_seg, ModuleInfo.name );
            strcat( Options.text_seg, DEFAULT_CODE_NAME );
            break;
        default:
            Options.text_seg = AsmAlloc( sizeof( DEFAULT_CODE_NAME ) + 1 );
            strcpy( Options.text_seg, DEFAULT_CODE_NAME );
            break;
        }
    }
    /* set Options.data_seg */
    if( Options.data_seg == NULL ) {
        Options.data_seg = AsmAlloc( sizeof( DEFAULT_DATA_NAME ) + 1 );
        strcpy( Options.data_seg, DEFAULT_DATA_NAME );
    }
    return;
}

void DefFlatGroup( void )
/***********************/
{
    dir_node    *grp;

    if( MAGIC_FLAT_GROUP == 0 ) {
        grp = CreateGroup( "FLAT" );
        if( grp != NULL ) {
            MAGIC_FLAT_GROUP = GetGrpIdx( &grp->sym );
        }
    }
}

uint GetSegIdx( struct asm_sym *sym )
/*************************************/
/* get idx to sym's segment */
{
    if( sym == NULL )
        return( 0 );
    if( ((dir_node *)sym)->e.seginfo->segrec != NULL )
        return( ((dir_node *)sym)->e.seginfo->segrec->d.segdef.idx );
    return( 0 );
}

uint GetGrpIdx( struct asm_sym *sym )
/***********************************/
/* get index of sym's group */
{
    if( sym == NULL )
        return( 0 );
    return( ((dir_node *)sym)->e.grpinfo->idx );
}

extern struct asm_sym *GetGrp( struct asm_sym *sym )
/**************************************************/
/* get ptr to sym's group sym */
{
    dir_node            *curr;

    curr = GetSeg( sym );
    if( curr != NULL )
        return( curr->e.seginfo->group );
    return( NULL );
}

int SymIs32( struct asm_sym *sym )
/********************************/
/* get sym's segment size */
{
    dir_node            *curr;

    curr = GetSeg( sym );
    if( curr == NULL ) {
        if( sym->state == SYM_EXTERNAL ) {
            if( ModuleInfo.mseg ) {
                curr = (dir_node *)sym;
                return( curr->e.extinfo->use32);
            } else {
                return( ModuleInfo.use32 );
            }
        } else if( sym->state == SYM_PROC)
            return(((dir_node *)sym)->e.procinfo->use32);
    } else if( curr->e.seginfo->segrec != NULL ) {
        return( curr->e.seginfo->segrec->d.segdef.use_32 );
    }
    return( 0 );
}

int FixOverride( int index )
/**************************/
/* Fix segment or group override */
{
    struct asm_sym      *sym;

    sym = SymLookup( AsmBuffer[index]->string_ptr );
    /**/myassert( sym != NULL );
    if( sym->state == SYM_GRP ) {
        SegOverride = sym;
        return( NOT_ERROR );
    } else if( sym->state == SYM_SEG ) {
        SegOverride = sym;
        return( NOT_ERROR );
    }
    AsmError( SYNTAX_ERROR );
    return( ERROR );
}

static enum assume_segreg search_assume( struct asm_sym *sym,
                         enum assume_segreg def, int override )
/**********************************************************/
{
    if( sym == NULL )
        return( ASSUME_NOTHING );
    if( def != ASSUME_NOTHING ) {
        if( SegAssumeTable[def].symbol != NULL ) {
            if( SegAssumeTable[def].symbol == sym )
                return( def );
            if( !override && ( SegAssumeTable[def].symbol == GetGrp( sym ) ) ) {
                return( def );
            }
        }
    }
    for( def = 0; def < NUM_SEGREGS; def++ ) {
        if( SegAssumeTable[def].symbol == sym ) {
            return( def );
        }
    }
    if( override )
        return( ASSUME_NOTHING );

    for( def = 0; def < NUM_SEGREGS; def++ ) {
        if( SegAssumeTable[def].symbol == NULL )
            continue;
        if( SegAssumeTable[def].symbol == GetGrp( sym ) ) {
            return( def );
        }
    }

    return( ASSUME_NOTHING );
}

int Use32Assume( enum assume_segreg prefix )
/***************************************/
{
    dir_node        *dir;
    seg_list        *seg_l;
    struct asm_sym  *sym_assume;

    if( SegAssumeTable[prefix].flat )
        return( 1 );
    sym_assume = SegAssumeTable[prefix].symbol;
    if( sym_assume == NULL )
        return( EMPTY );
    if( sym_assume->state == SYM_SEG ) {
        dir = (dir_node *)sym_assume;
        if( dir->e.seginfo->segrec == NULL ) {
            return( EMPTY );
        } else {
            return( dir->e.seginfo->segrec->d.segdef.use_32 );
        }
    } else if( sym_assume->state == SYM_GRP ) {
        dir = (dir_node *)sym_assume;
        seg_l = dir->e.grpinfo->seglist;
        dir = seg_l->seg;
        if( dir->e.seginfo->segrec == NULL ) {
            return( EMPTY );
        } else {
            return( dir->e.seginfo->segrec->d.segdef.use_32 );
        }
    }
    return( EMPTY );
}

enum assume_segreg GetPrefixAssume( struct asm_sym *sym, enum assume_segreg prefix )
/****************************************************************************/
{
    struct asm_sym  *sym_assume;

    if( Parse_Pass == PASS_1 )
        return( prefix );

    if( SegAssumeTable[prefix].flat ) {
        Frame = FRAME_GRP;
        Frame_Datum = MAGIC_FLAT_GROUP;
        return( prefix );
    }
    sym_assume = SegAssumeTable[prefix].symbol;
    if( sym_assume == NULL ) {
        if( sym->state == SYM_EXTERNAL ) {
#if 0 //NYI: Don't know what's going on here
            type = GetCurrGrp();
            if( type != 0 ) {
                Frame = FRAME_GRP;
            } else {
                type = GetSegIdx( GetCurrSeg() );
                /**/myassert( type != 0 );
                Frame = FRAME_SEG;
            }
            Frame_Datum = type;
#endif
            return( prefix );
        } else {
            return( ASSUME_NOTHING );
        }
    }

    if( sym_assume->state == SYM_SEG ) {
        Frame = FRAME_SEG;
        Frame_Datum = GetSegIdx( sym_assume->segment );
    } else if( sym_assume->state == SYM_GRP ) {
        Frame = FRAME_GRP;
        Frame_Datum = GetGrpIdx( sym_assume );
    }
    if( ( sym->segment == sym_assume )
        || ( GetGrp( sym ) == sym_assume )
        || ( sym->state == SYM_EXTERNAL ) ) {
        return( prefix );
    } else {
        return( ASSUME_NOTHING );
    }
}

enum assume_segreg GetAssume( struct asm_sym *sym, enum assume_segreg def )
/*******************************************************************/
{
    enum assume_segreg  reg;

    if( ( def != ASSUME_NOTHING ) && SegAssumeTable[def].flat ) {
        Frame = FRAME_GRP;
        Frame_Datum = MAGIC_FLAT_GROUP;
        return( def );
    }
    if( SegOverride != NULL ) {
        reg = search_assume( SegOverride, def, 1 );
    } else {
        reg = search_assume( sym->segment, def, 0 );
    }
    if( reg == ASSUME_NOTHING ) {
        if( sym->state == SYM_EXTERNAL && sym->segment == NULL ) {
            reg = def;
        }
    }
    if( reg != ASSUME_NOTHING ) {
        if( SegAssumeTable[reg].symbol == NULL ) {
        } else if( SegAssumeTable[reg].symbol->state == SYM_SEG ) {
            Frame = FRAME_SEG;
            Frame_Datum = GetSegIdx( SegAssumeTable[reg].symbol );
        } else {
            Frame = FRAME_GRP;
            Frame_Datum = GetGrpIdx( SegAssumeTable[reg].symbol );
        }
        return( reg );
    }
    return( ASSUME_NOTHING );
}

void SetSymSegOfs( struct asm_sym *sym )
/*******************************************/
{
    sym->segment = &GetCurrSeg()->sym;
    sym->offset = GetCurrAddr();
}

int SegmentModulePrologue(int type)
{
    char        buffer[ MAX_LINE_LEN ];

    lastseg.seg = SIM_NONE;
    lastseg.stack_size = 0;

    /* Generates codes for code segment */
    InputQueueLine( SetSimSeg( SIM_CODE, Options.text_seg, buffer ) );
    InputQueueLine( get_sim_code_end( buffer, Options.text_seg ) );

    /* Generates codes for data segment */
    InputQueueLine( SetSimSeg( SIM_DATA, NULL, buffer )) ;
    InputQueueLine( SimCodeEnd[SIM_DATA] );

    if( type != MOD_FLAT ) {
        /* Generates codes for grouping */
        strcpy( buffer, "DGROUP GROUP " );
        if( type == MOD_TINY ) {
            strcat( buffer, Options.text_seg );
            strcat( buffer, "," );
        }
        strcat( buffer, Options.data_seg );
        InputQueueLine( buffer );
    }
    return(NOT_ERROR);
}

int SegmentModuleEnd(void)
{
    if( lastseg.seg != SIM_NONE ) {
        DebugMsg(("ModuleEnd: segment index %u must be closed\n", lastseg.seg));
        close_lastseg();
        InputQueueLine( CurrString );
        return( EMPTY );
    }

    /* close all opened segments */

    while(CurrSeg)
        pop_seg();

    return(NOT_ERROR);
}

