/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* sun-sp85.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/05/13  B. Dorsey          Wrote original version
 * 93/09/12  B. Dorsey		Changes for nautilus (no functional changes)
 * 93/09/16  R. Berry		Optimized encode()
 * 95/06/08  B. Dorsey		Renamed to sp85 from apsd
 */

#include <stdio.h>
#include <math.h>

#include "machine.h"

/* Conversion functions */
#define	Irint(X)	((int)((X) < 0. ? (X) - 0.5 : (X) + 0.5))
#define audio_s2f(X)	(((UINT16)(X)) == 0x8000 ? -1. :	\
			((float)((INT16)(X))) / 32767.)
#define audio_f2s(X)	((X) >= 1. ? 32767 : (X) <= -1. ? -32767 :\
			(INT16)(Irint((X) * 32767.)))

/* Operating Parameters */
#define FRAME_SIZE      120
#define YPMIN           0.001
#define YPMAX           0.150
#define YPSIZE          6
#define SLOPE_MULT      1.0

static struct {
    float           y[2];
    float           ypdelta;
}               handle;

static struct filter_t {
    float           h1;
    float           h2;
}               filters[] = {
    1.53, -0.72,		/* filter coefficients */
    0.95, -0.34,
    0.48, -0.21,
    -0.63, -0.36
};

void
sp85_init()
{
    /* initializations */
    handle.y[0] = 0.0;
    handle.y[1] = 0.0;
    handle.ypdelta = (YPMAX - YPMIN) / ((float) (1 << YPSIZE) - 1);
}

void
sp85_encode(INT16 * x, UINT8 * bits)
{
    int             i, step, state;
    float           yy, y, phi0, phi1, step_size;
    float           *xp, xr[FRAME_SIZE];
    UINT8           mask = 0x80, data = 0;	/* from AddBit() */

    /* Convert input to floating point
     * Compute normalized autocorrelation at lag 0 & 1
     * Compute step size based on RMS value
     */
    xp = xr;
    yy = *xp++ = audio_s2f(x[0]);	/* priming the loop */
    phi0 = yy * yy + 1.0e-20;		/* 1.0e-20 to prevent divide by 0 */
    step_size = phi1 = 0.0;

    for (i = 1; i < FRAME_SIZE; i++, xp++) {
	*xp = audio_s2f(x[i]);	/* convert to floating pt */
	phi0 += *xp * *xp;	/* autocorr at lag 0 */
	phi1 += *xp * yy;	/* autocorr at lag 1 */
	step_size += (*xp - yy) * (*xp - yy);	/* rms calc */
	yy = *xp;
    }
    phi1 /= phi0;		/* normalize autocorr at lag 1 */

    /* select predictor state */
    if (phi1 > 0.7)
	state = 0;
    else if (phi1 > 0.4)
	state = 1;
    else if (phi1 > 0.0)
	state = 2;
    else
	state = 3;

    /* compute step size based on RMS value of input */
    step_size = SLOPE_MULT * sqrt(step_size / (FRAME_SIZE - 1));

    /* check step size for bounds */
    if (step_size < YPMIN)
	step_size = YPMIN;
    else if (step_size > YPMAX)
	step_size = YPMAX;

    /* quantize step size to YPSIZE bits */
    step = Irint((step_size - YPMIN) / handle.ypdelta);

    /* save predictor state and quantized step size in output */
    *bits++ = state + (step << 2);

    /* decode step size from quantized step size */
    step_size = YPMIN + (float) step *handle.ypdelta;

    /* compute output bits */
    for (i = 0; i < FRAME_SIZE; i++) {
	y = filters[state].h1 * handle.y[0] +
	    filters[state].h2 * handle.y[1];

	handle.y[1] = handle.y[0];

	if (xr[i] > y) {
	    y += step_size;
	    data |= mask;
	}
	else
	    y -= step_size;

	if (!(mask >>= 1)) {
	    *bits++ = data;
	    data = mask;
	    mask = 0x80;
	}

	handle.y[0] = y;
    }
}

void
sp85_decode(UINT8 * bits, INT16 * x)
{
    int             i, step, state;
    float           y, step_size;

    /* get predictor state and step size from input */
    state = *bits & 0x3;	/* ls 2 bits */
    step = *bits++ >> 2;	/* ms 6 bits */

    /* decode step size from quantized step size */
    step_size = YPMIN + (float) step * handle.ypdelta;

    /* compute output from input bits */
    for (i = 0; i < FRAME_SIZE; i++) {
	y = filters[state].h1 * handle.y[0] + filters[state].h2 * handle.y[1];
	handle.y[1] = handle.y[0];
	if (bits[i>>3] & 0x80>>(i&7))		/* Mycal's suggestion */
	    y += step_size;
	else
	    y -= step_size;
	handle.y[0] = y;
	x[i] = audio_f2s(y);
    }
}
