//	PGPJNENC.CPP
//	Runtime Loadable Encryptor Extension for Pegasus Mail for Windows.
//	Copyright (c) 1995, David Harris, All Rights Reserved.
//
//	Pretty Good Privacy modifications by John Navas
//	Copyright (c) 1995, John Navas, All Rights Reserved.
//
//	Encryptor module: this module implements the actual Encryptor
//	Interface Routines called by WinPMail at runtime. For more
//	information on the actual form and function of each call, consult
//	the file CRYPTIF.TXT in the FORMS subdirectory of your WINPMAIL
//	installation directory.
//
//	The author grants explicit permission for this source code to be
//	used or modified as required, subject only to the conditions that
//	the copyright notices above are preserved; that this source code
//	not be used in any product distributed in competition with this
//  product; that by using this code you agree that the code is
//	provided without warranty of any kind, either explicit or implied,
//	and you use it at your own risk.
//
//	This module implements an interface to Pretty Good Privacy (tm)
//	and (c) Copyright 1990-1994 Philip Zimmermann. All rights reserved.
//
//	The Massachusetts Institute of Technology is the distributor of PGP
//	version 2.6, for distribution in the USA only.  It is available from
//	"net-dist.mit.edu," a controlled FTP site that has restrictions and
//	limitations, similar to those used by RSA Data Security, Inc., to comply
//	with export control requirements.  The software resides in the directory
//	/pub/PGP.

#include "pgpjn.h"
#include "winexejn.h"
#include "resource.h"

///////////////////////////////////////////////////////////////////////////////
// PGP window state

#ifdef _DEBUG
const int PGPwindow = SW_SHOWNORMAL;
#else
const int PGPwindow = SW_SHOWMINNOACTIVE;
#endif // _DEBUG

///////////////////////////////////////////////////////////////////////////////
// FRAME A TEXT STRING IF IT HAS EMBEDDED BLANKS

// frame temporary object (for function call argument)
#define FRAMEIT(s) FrameIt(MyBuffer(((char*)(s))?lstrlen(s)+1+2:0,(s)))

char*
FrameIt(char* string)
{
	if (string)
		if (strchr(string, ' ')) {
			int len = lstrlen(string);
			
			memmove(string + 1, string, len + 1);
			string[0] = '"';
			lstrcat(string, "\"");
		}
	return string;
}

///////////////////////////////////////////////////////////////////////////////
// REMOVE CONTROL CHARACTERS
// Embedded CR-LF are kept

static
char*
RemoveCtrl(char* string)
{
	int len = lstrlen(string);
	
	for (int n = len; --n >= 0;)		// examine each character decending
		if (__isascii(string[n]) && iscntrl(string[n])) {	// just control characters
			if (n > 0 && '\r' == string[n-1] && '\n' == string[n] && '\0' != string[n+1]) {
				--n;					// skip over embedded CR-LF
				continue;
			}
			memmove(string + n, string + n + 1, (len--) - n);	// wipe out control character
		}
	return string;
}

///////////////////////////////////////////////////////////////////////////////
// SCAN FOR TEXT LINE IN FILE

static
BOOL									// FALSE if found; TRUE if not found
ScanFile(
MyFile2& fil,							// input file to scan
const char* string,						// string to scan for
BOOL after = TRUE						// FALSE leaves pos before; TRUE leaves pos after
)
{
	int len = lstrlen(string);			// length of string
	
	for (;;) {
		char buf[80];					// input buffer
	    		
		fil.Gets(buf, sizeof(buf));	// read a line
		if (fil.Eof() || fil.Error())
			return TRUE;
		if (! strncmp(buf, string, len)) {
			if (! after)
				fil.Seek(- (long) strlen(buf), SEEK_CUR);	// back up over line
			break;
		}
		Yield();
	}
	return FALSE;
}

///////////////////////////////////////////////////////////////////////////////
// DECODE PGP SIGNATURE MESSAGE

typedef enum { NONE = -1, BAD = 0, GOOD = 1, NOKEY = 2 } SigType;

SigType
SigStatus(
const char* name,						// name of file to process
char* text = NULL						// save PGP sig status here
)
{
	SigType ret;
	
	int glen = lstrlen(SigGood);
	int blen = lstrlen(SigBad);
	int klen = lstrlen(SigNoKey);
	
	char buf[80 + 2 + 1];				// input line buffer
	MyFile2 tmp(name, "r", 8 * BUFSIZ);	// auto close

	// find PGP message in stderr output
	for (;;) {
		tmp.Gets(buf, sizeof(buf));
		if (tmp.Error() || tmp.Eof()) {
			buf[0] = '\0';						// null default
			ret = NONE;							// no signature
			break;
		}
		// see if PGP found a good signature
		if (! strncmp(buf, SigGood, glen)) {
			MessageBeep(MB_OK);					// catch users attention (WinPMail doesn't)
			ret = GOOD;							// good signature
			break;
		}
		// see if PGP found a bad signature
		if (! strncmp(buf, SigBad, blen)) {
			MessageBeep(MB_ICONEXCLAMATION);	// catch users attention (WinPMail doesn't)
			ret = BAD;							// bad signature
			break;
		}
		// see if PGP couldn't find the key for a signature
		if (! strncmp(buf, SigNoKey, klen)) {
			MessageBeep(MB_ICONEXCLAMATION);	// catch users attention (WinPMail doesn't)
			ret = NOKEY;						// no key found
			break;
		}
	}                
	// process PGP text for WinPMail
	if (text) {
		buf[80] = '\0';
		RemoveCtrl(buf);
		lstrcpy(text, buf);
	}
	return ret;
}

///////////////////////////////////////////////////////////////////////////////
// CENTER A DIALOG BOX

void
CenterDialog(HWND hwndDlg)
{
	RECT rc;

	// center dialog box on display			
	GetWindowRect(hwndDlg, &rc);
	SetWindowPos(hwndDlg, NULL,
		((GetSystemMetrics(SM_CXSCREEN) - (rc.right - rc.left)) / 2),
		((GetSystemMetrics(SM_CYSCREEN) - (rc.bottom - rc.top)) / 2),
		0, 0, SWP_NOSIZE | SWP_NOACTIVATE);
}

///////////////////////////////////////////////////////////////////////////////
// LOAD STDERR OUTPUT INTO DIALOG CONTROL

static
BOOL									// FALSE for success; TRUE for failure
LoadStderrOut(
HWND hwndItm,							// dialog box control
const char* stderrout					// stderr output capture file
)
{
	MyFile1 fil(stderrout, OF_READ);	// automatically closed
	if (HFILE_ERROR == fil)
		return TRUE;					// return failure
	LONG len = fil.GetLength();			// length of file
	MyBuffer buf((size_t) (len + 1));	// alloc buffer for entire file (auto free)
	if (! (char*) buf)
		return TRUE;					// return failure
	len = _lread(fil, buf, (UINT) len);	// read entire file
	buf[len] = '\0';					// turn file into string
	RemoveCtrl(buf);					// remove control characters
	// setup stderr output in dialog edit control
	SetWindowText(hwndItm, buf);		// load text
	PostMessage(hwndItm, EM_SETSEL, 0, MAKELONG(0, -1));	// scroll to end of text
	return FALSE;						// return success
}

///////////////////////////////////////////////////////////////////////////////
// DISPLAY PGP STDERR OUTPUT

static const char* DialogTag;			// error summary tag
static const char* DialogFileErr;		// file with stderr output

BOOL
CALLBACK _export
DialogProcErr(
HWND hwndDlg,	/* handle of dialog box	*/
UINT msg,		/* message	*/
WPARAM wParam,	/* first message parameter	*/
LPARAM lParam	/* second message parameter	*/
)
{
	switch (msg) {
	case WM_INITDIALOG:
		// center dialog box
		CenterDialog(hwndDlg);
	    // error summary tag (using system font)
		SendDlgItemMessage(hwndDlg, IDC_STDERRTAG, WM_SETFONT, NULL, 0L);
    	SetDlgItemText(hwndDlg, IDC_STDERRTAG, DialogTag);
    	// process stderr output (file -> buffer -> dialog)
    	LoadStderrOut(GetDlgItem(hwndDlg, IDC_STDERROUT), DialogFileErr);
    	// catch users attention
		MessageBeep(MB_ICONEXCLAMATION);
		break;

	case WM_COMMAND :
		switch (wParam) {
		case IDOK:
		case IDCANCEL:
			EndDialog(hwndDlg, 0);
			return TRUE;
		}
		break;
	}
	
	return FALSE;						// did not process
}

static
void
StderrDialog(
const char* tag,						// description of error
const char* stderrout,					// stderr output capture file
HWND hwndOwner = hwndParent				// handle of parent window
)
{
	// save data for dialog box
	DialogTag = tag;
	DialogFileErr = stderrout;
	
	// modal dialog box
	DialogBox(hLibInstance,
		MAKEINTRESOURCE(IDD_STDERRDLG),
		hwndOwner,
		(DLGPROC) DialogProcErr);
}

///////////////////////////////////////////////////////////////////////////////
// RUN DOS PGP EXECUTABLE

int										// return code
static
RunPGP(
UINT fiCmdShow,							// window state
char* options,
char* sourcefile,
char* destfile,
char* recipient,
char* key,
char* stderr							// optional redirection for stderr
)
{
	MyTempFile temppath;
	char tempfile[_MAX_FNAME+_MAX_EXT];
	int ret;

	// prepare temporary file
	{
		char fname[_MAX_FNAME];
		char ext[_MAX_EXT];
		MyFile2 tmp(temppath, "w");
		if (! tmp.Error())
			tmp.Printf("%s\n%s\n%s\n%s\n%s\n%s\n",
				PGPpath, options, sourcefile, destfile, recipient, stderr);
		if (tmp.Error()) {
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				"Unable to create temporary file!",	/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONSTOP | MB_OK);			/* style of message box	*/
			return SOME_ERROR;
		}
		_splitpath(temppath, NULL, NULL, fname, ext);
		_makepath(tempfile, NULL, NULL, fname, ext);
	}

    // run PGP
    {
    	char buf[256];

		// build arguments    	
    	lstrcpy(buf, tempfile);
    	if (*key) {
    		if (strchr(key, '"'))
				MessageBox(
					hwndParent,						/* handle of parent window	*/
					"Pass phrase may not contain a doublequote (\") character!",	/* address of text in message box	*/
					"PGP Interface",				/* address of title of message box	*/
					MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
    		lstrcat(buf, " \"");
    		lstrcat(buf, key);
    		lstrcat(buf, "\"");
    	}
    	if (strlen(buf) >= 127)
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				"Your pass phrase is too long!",	/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
		ret = WinExeJN(PGPJNDOS, buf, MyDir, fiCmdShow);
	}

	if (ret != RAN_OK) {
#ifdef _DEBUG
		MyDebugOutput("PGP did NOT run OK!\n");
#endif // _DEBUG
		return ret;
	}
	if (_access(temppath, 00)) {
#ifdef _DEBUG
		MyDebugOutput("PGP returned 0\n");
#endif // _DEBUG
		return RAN_OK;
	}
	{
		MyFile1 tmp(temppath, OF_READ);

		if (tmp != HFILE_ERROR) {
			char buf[8];
			UINT len = _lread(tmp, buf, sizeof(buf) - 1);

			buf[len] = '\0';
			ret = atoi(buf);
		}
	}

#ifdef _DEBUG
	MyDebugOutput("PGP returned %d\n", ret);
#endif // _DEBUG
	return ret;
}

///////////////////////////////////////////////////////////////////////////////
// ENCRYPT_FILE

//  Returns:  1 if the file was successfully encrypted.
//            2 if the file was successfully encrypted but
//              may contain 8-bit data and should be sent
//              encoded using BASE64.
//            0 on failure.
extern "C"
int
FAR PASCAL _export
ENCRYPT_FILE(
char *sourcefile,
char *destfile,
char *recipient,						// normal putlic key selection
char *key,								// may be used to override recipient
int textfile
)
{
	int ret;

#ifdef _DEBUG
	MyDebugOutput("ENCRYPT_FILE called (r='%s' k='%s')\n", recipient, key);
#endif // _DEBUG

	// make temp file for stderr output
	MyTempFile stderrout;
		
	// run PGP
	{
		char rk[256];			// key buffer

		// choose which to use as key
		lstrcpyn(rk, (key != NULL && *key != '\0') ? key : recipient, 128);
		// ensure termination
		rk[128] = '\0';
		// frame it if necessary
		FrameIt(rk);
		// master key logic
		if (*PGPJNKey) {
			lstrcat(rk, " ");
			lstrcat(rk, PGPJNKey);
		}
		
		// run PGP
		ret = RunPGP(PGPwindow, "+bat -fea", sourcefile, destfile,
			rk, "", stderrout);
	}

	// return OK result
	if (RAN_OK == ret && ! _access(destfile, 00)) {
#ifdef _DEBUG
		MyDebugOutput("ENCRYPT_FILE returns 1\n");
#endif // _DEBUG
		return 1;
	}

	// failure
	StderrDialog("Encryption failed!  (Empty file will be sent.)", stderrout);
	remove(destfile);
#ifdef _DEBUG
	MyDebugOutput("ENCRYPT_FILE returns 0\n");
#endif // _DEBUG
	return 0;
}

///////////////////////////////////////////////////////////////////////////////
// SIGN_FILE

//  Returns:  1 if the message was successfully processed
//            2 if the file was successfully processed but
//              may contain 8-bit data and should be sent
//              encoded using BASE64.
//            0 if an error occurred during processing
//           -1 if the operation is not supported
extern "C"
int
FAR PASCAL _export
SIGN_FILE(
char *sourcefile,
char *destfile,
char *recipient,						// used for optional encryption
char *key,								// reserved for pass phrase
int encrypt_as_well
)
{
	int ret;

#ifdef _DEBUG
	MyDebugOutput("SIGN_FILE called (r='%s' k='%s')\n", recipient, key);
#endif // _DEBUG

	// make temp file for stderr output
	MyTempFile stderrout;
		
	{
		char rk[256];					// key buffer

		// frame recipient if necessary
		lstrcpyn(rk, recipient, 128);
		rk[128] = '\0';					// ensure termination
		FrameIt(rk);					// frame it if necessary

		// add master key if necessary
		if (*PGPJNKey) {
			lstrcat(rk, " ");
			lstrcat(rk, PGPJNKey);
		}
		
		// run PGP
		ret = RunPGP(PGPwindow,
			(! encrypt_as_well) ? "+bat -fsta" : "+bat -feas",
			sourcefile, destfile, rk, key, stderrout);
	}

	// check result
	if (RAN_OK == ret && ! _access(destfile, 00)) {
		// success
		ret = 1;
	}
	else {
		// failure
		StderrDialog("Sign File failed!", stderrout);
		remove(destfile);
		ret = 0;
	}

#ifdef _DEBUG
	MyDebugOutput("SIGN_FILE returns %d\n", ret);
#endif // _DEBUG

	return ret;
}

///////////////////////////////////////////////////////////////////////////////
// DECRYPT_FILE

//  Returns:  1 if the file was successfully decrypted
//            2 if the file was decrypted but is non-textual
//            0 on password or system failure
extern "C"
int
FAR PASCAL _export
DECRYPT_FILE(
char *sourcefile,
char *destfile,
char *key
)
{
	int ret = 0;

#ifdef _DEBUG
	MyDebugOutput("DECRYPT_FILE called (k='%s')\n", key);
#endif // _DEBUG

	do {	// makes it easy to break out
		int rtc;
		char buf[256];					// i/o buffer
		long pos, len, cop, pos1;		// file positions and length
		int startlen = lstrlen(PGPBegin);
		int stoplen = lstrlen(PGPEnd);
		
		{
			// open files for initial copy (binary mode for repositioning)
			MyFile2 fin(sourcefile, "rb", 8 * BUFSIZ);
			MyFile2 fout(destfile, "wb", 8 * BUFSIZ);
			// make sure input and output files got opened
			if (fin.Error() || fout.Error())
				break;
	
			// get length of input file
			len = fin.GetLength();
	
			// copy input to output up until PGP body
			for (;;) {
				fin.Gets(buf, sizeof(buf));
				if (fin.Eof() || fin.Error())
					break;
				if (! strncmp(buf, PGPBegin, startlen))
					break;
				fout.Puts(buf);
				Yield();
			}
			if (fin.Error()) {			// error
				MessageBox(
					hwndParent,						/* handle of parent window	*/
					"I/O error on decryption input file!",	/* address of text in message box	*/
					"PGP Interface",				/* address of title of message box	*/
					MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
				break;
			}
	
			// save next line in PGP body and output marker
			pos = fin.Tell();
			pos1 = fout.Tell();
		}

		// make temp file for stderr output
		MyTempFile stderrout;
		
		// run PGP, which should add decrypted message to output
		rtc = RunPGP(PGPwindow, "+bat -f", sourcefile, destfile, "", key, stderrout);
		// 1: decryption failed
		// 2: cyphertext not found

		{		
			// open files for final copy/append (binary mode for repositioning)
			MyFile2 fin(sourcefile, "rb", 8 * BUFSIZ);
			MyFile2 fout(destfile, "r+b", 8 * BUFSIZ);
			// make sure input and output files got opened
			if (fin.Error() || fout.Error())
				break;
	
			// start in PGP body
			fin.Seek(pos, SEEK_SET);
			// position output
			fout.Seek(0, SEEK_END);
	
			cop = fout.Tell() - pos1;		// amount copied by PGP
			if (rtc != RAN_OK && (0 == cop || len == cop)) {
				// PGP just dups the file if it can't find valid ciphertext
				if (len == cop) {
					MyFile2 ferr(stderrout, "w");
					if (! ferr.Error())
						ferr.Printf("PGP could not find valid ciphertext to decrypt.\n"
							"(The message may have been corrupted.)");
					fout.Seek(pos1, SEEK_SET);	// remove the PGP dup
					fout.Trunc(pos1);	// and truncate the file
				}
				StderrDialog("Decryption failed!", stderrout);
				fout.Puts(buf);			// copy start line to output
			}
			else {
				// tell user about signature
				switch (SigStatus(stderrout, buf)) {
				case BAD:
					StderrDialog("Invalid PGP signature!", stderrout);
				case GOOD:
				case NOKEY:
					fout.Printf("%s\r\n%s\r\n%s\r\n", PGPBeginAdvise, buf, PGPEndAdvise);
					break;
				}

				// scan for end of PGP body
				if (ScanFile(fin, PGPEnd))
					break;
			}
						
			// copy/append remainder of input to output
			for (;;) {
				fin.Gets(buf, sizeof(buf));
				if (fin.Error() || fin.Eof())
					break;
				fout.Puts(buf);
				Yield();
			}
		}
				
		ret = 1;
	} while (0);	// never loop

    // return to caller
#ifdef _DEBUG
	MyDebugOutput("DECRYPT_FILE returns %d\n", ret);
#endif // _DEBUG
	return ret;
}

///////////////////////////////////////////////////////////////////////////////
// DESTROY_FILE

//  Returns:  1 if the file was destroyed
//            0 if the file was not destroyed (in which
//              case WinPMail will delete it as described).
extern "C"
int
FAR PASCAL _export
DESTROY_FILE(char *sourcefile)			// pathname of file to destroy
{
	MyFile2 fil(sourcefile, "r+b", 8 * BUFSIZ);	// open file for binary appending

#ifdef _DEBUG
	MyDebugOutput("DESTROY_FILE called (f='%s')\n", sourcefile);
#endif // _DEBUG

	if (fil.Error())
		return 0;						// open failure
	if (fil.Erase())
		return 0;						// erase failure
	fil.Close();						// close the file
	remove(sourcefile);					// delete it from the disk
#ifdef _DEBUG
	MyDebugOutput("DESTROY_FILE normal return\n");
#endif // _DEBUG
	return 1;							// return success
}

///////////////////////////////////////////////////////////////////////////////
// VERIFY_SIGNATURE

//  Returns:  1 if the digital signature was valid
//            0 if the digital signature was invalid
//           -1 if no digital signature could be found
//           -2 if the operation is not supported
extern "C"
int
FAR PASCAL _export
VERIFY_SIGNATURE(
char *sourcefile,
char *text								// limited to 80 characters
)
{
	int ret;

#ifdef _DEBUG
	MyDebugOutput("VERIFY_SIGNATURE called (f='%s')\n", sourcefile);
#endif // _DEBUG

	// make temp file for stderr output
	MyTempFile stderrout;
	
	// run PGP (all i want is return code)
	ret = RunPGP(PGPwindow, "+bat -f", sourcefile, "nul", "", "", stderrout);

	// check result
	switch (ret) {
	case RAN_OK:
	case 1:								// 1 is OK too
		switch (SigStatus(stderrout, text)) {
		case NONE:
			ret = -1;
			break;
		case GOOD:
			ret = 1;
			break;
		case BAD:
		case NOKEY:
			ret = 0;
			break;
		}
		break;
	default:
		StderrDialog("Verify Signature failed!", stderrout);
		ret = 0;									// signature verify failed
		break;
	}
    
#ifdef _DEBUG
	MyDebugOutput("VERIFY_SIGNATURE returns %d\n", ret);
#endif // _DEBUG

	return ret;
}

///////////////////////////////////////////////////////////////////////////////
// KEY_MANAGEMENT

static const char* DialogFileKey;		// file with public key

BOOL
CALLBACK _export
DialogProcKey(
HWND hwndDlg,	/* handle of dialog box	*/
UINT msg,		/* message	*/
WPARAM wParam,	/* first message parameter	*/
LPARAM lParam	/* second message parameter	*/
)
{
	static BOOL AddKey;
	static BOOL ChkKey;
	int ret;							// return code
	
	switch (msg) {
	case WM_INITDIALOG:
		// center dialog box
		CenterDialog(hwndDlg);
		// init flags
		EnableWindow(GetDlgItem(hwndDlg, IDC_ADDKEY), AddKey = FALSE);
		EnableWindow(GetDlgItem(hwndDlg, IDC_CHKKEY), ChkKey = FALSE);
    	// load from file if present
    	do {
			long pos1, pos2;
			MyFile2 fil(DialogFileKey ? DialogFileKey : "", "rb", 8 * BUFSIZ);
			    		                                                   
			if (fil.Error())
				break;
			// find start of public key block
			ScanFile(fil, PGPBeginKey, FALSE);
			pos1 = fil.Tell();
			// find end of public key block
			ScanFile(fil, PGPEndKey, TRUE);
			pos2 = fil.Tell();
			// if public key found
			if (pos2 > pos1) {
				int len = (int) (pos2 - pos1);	// length of file segment
				MyBuffer buf((size_t) (len + 1));	// alloc buffer for entire seg
								    		
				if ((char*) buf) {	// if buffer allocated
					fil.Seek(pos1, SEEK_SET);	// position file
					len = fil.Read(buf, sizeof(char), (unsigned int) len);	// read entire seg
					buf[len] = '\0';			// turn seg into string
					RemoveCtrl(buf);			// remove control characters
					// setup stderr output in dialog edit control
					SetDlgItemText(hwndDlg, IDC_KEYMGMT, buf);	// load text
					PostMessage(GetDlgItem(hwndDlg, IDC_KEYMGMT),	// scroll to start of text
						EM_SETSEL, 0, MAKELONG(0, 0));
					EnableWindow(GetDlgItem(hwndDlg, IDC_ADDKEY), AddKey = TRUE);
				}
			}
    	} while (0);
		break;
		
	case WM_COMMAND :
		switch (wParam) {
		
		case IDC_KEYMGMT:
			switch (HIWORD(lParam)) {
			case EN_CHANGE:
				{
					int len = (int) SendMessage((HWND) LOWORD(lParam), WM_GETTEXTLENGTH, 0, 0L);
					BOOL chkok = (len != 0);
									
					if (AddKey != chkok)
						EnableWindow(GetDlgItem(hwndDlg, IDC_ADDKEY), AddKey = chkok);
					if (len > MAXNAME
					|| SendDlgItemMessage(hwndDlg, IDC_KEYMGMT, EM_GETLINECOUNT, 0, 0) > 1)
						chkok = FALSE;
					if (ChkKey != chkok) {
						EnableWindow(GetDlgItem(hwndDlg, IDC_CHKKEY), ChkKey = chkok);
					}
				}
				return TRUE;
            }
			break;

		case IDC_ADDKEY:
			{
				// make temp file for std output
				MyTempFile stderrout;
				// make temp file for key source
				MyTempFile sourcefile;
								
				// load sourcefile
				do {
					// allocate i/o buffer
					int len = GetWindowTextLength(GetDlgItem(hwndDlg, IDC_KEYMGMT)) + 1;
					MyBuffer buf(len);	// auto delete when goes out of scope
					if (! (char*) buf)
						break;
					// get dialog data into buffer
					GetWindowText(GetDlgItem(hwndDlg, IDC_KEYMGMT), buf, len);
									
					// open output file
					MyFile1 fil(sourcefile, WRITE);
					// write data to output file if it got opened
					if (fil != HFILE_ERROR)
						_lwrite(fil, buf, len);
				} while (0);	// never loop
								
				// run PGP
				ret = RunPGP(PGPwindow, "+bat -ka", "nul", stderrout, sourcefile, "", "nul");
							
				// std output capture file
				StderrDialog("Add Public Key to Key Ring", stderrout, hwndDlg);
					
				// restore focus
				SetFocus(GetDlgItem(hwndDlg, IDC_KEYMGMT));
					
				return TRUE;
			}

		case IDC_CHKKEY:
			{
				// allocate recipient buffer
				int len = GetWindowTextLength(GetDlgItem(hwndDlg, IDC_KEYMGMT)) + 1;
				MyBuffer buf(len);	// deleted auto when it goes out of scope
					
				// get dialog data into buffer
				if ((char*) buf)
				    GetWindowText(GetDlgItem(hwndDlg, IDC_KEYMGMT), buf, len);
									
				// make temp file for std output
				MyTempFile stderrout;
									
				// run PGP
				ret = RunPGP(PGPwindow, "+bat -kc", "nul", stderrout, FRAMEIT(buf), "", "nul");
						
				// std output capture file
				StderrDialog("Check Key Ring for Public Key", stderrout, hwndDlg);
						
				// restore focus
				SetFocus(GetDlgItem(hwndDlg, IDC_KEYMGMT));
						
				return TRUE;
			}
			
		case IDOK:
		case IDCANCEL:
			EndDialog(hwndDlg, 0);
			return TRUE;
		}
		break;
	}
	
	return FALSE;						// did not process
}

// return value is ignored
extern "C"
int
FAR PASCAL _export
KEY_MANAGEMENT(char *sourcefile)
{
#ifdef _DEBUG
	MyDebugOutput("KEY_MANAGEMENT called (f='%s')\n", sourcefile);
#endif // _DEBUG

	// save data for dialog box
	DialogFileKey = sourcefile;

	// modal dialog box
	DialogBox(hLibInstance,
		MAKEINTRESOURCE(IDD_KEYMGMTDLG),
		hwndParent,
		(DLGPROC) DialogProcKey);

	return 0;
}

///////////////////////////////////////////////////////////////////////////////

//	PGPJNENC.CPP
