/********************************************************************** 
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/
#include <stdio.h>
#include <stdlib.h>

#ifndef FREECIV_WIN32
#include <sys/signal.h>
#endif /* FREECIV_WIN32 */

#include <sys/types.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#ifndef FREECIV_WIN32
#include <pwd.h>
#endif /* FREECIV_WIN32 */
#include <string.h>

#if defined(AIX) || defined(__EMX__)
#include <sys/select.h>
#endif

#include <signal.h>

#ifndef FREECIV_WIN32
#include <sys/socket.h>
#include <netdb.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif /* FREECIV_WIN32 */
#ifdef FREECIV_WIN32
#define Win32_Winsock
#include <windows.h>
#endif /* FREECIV_WIN32 */
#include <log.h>
#include <sernet.h>
#include <shared.h>
#include <civserver.h>
#include <stdinhand.h>
#include <packets.h>

#define MY_FD_ZERO(p) memset((void *)(p), 0, sizeof(*(p)))

struct connection connections[MAX_CONNECTIONS];
int sock;
extern int errno;
extern int port;
extern int force_end_of_sniff;

extern enum server_states server_state;

#ifdef FREECIV_WIN32
extern HANDLE hStdIn;
extern HANDLE hStdOut;
#endif /* FREECIV_WIN32 */


/*****************************************************************************/
void close_connection(struct connection *pconn)
{
#ifndef FREECIV_WIN32
  close(pconn->sock);
#endif /* FREECIV_WIN32 */
#ifdef FREECIV_WIN32
  closesocket(pconn->sock);
#endif /* FREECIV_WIN32 */
  pconn->used=0;
}

/*****************************************************************************/
void close_connections_and_socket(void)
{
  int i;
  
  for(i=0; i<MAX_CONNECTIONS; i++) {
    if(connections[i].used) {
      close_connection(&connections[i]);
    }
  }
#ifndef FREECIV_WIN32
  close(sock);
#endif /* FREECIV_WIN32 */
#ifdef FREECIV_WIN32
  closesocket(sock);
  WSACleanup();
#endif /* FREECIV_WIN32 */
}

/*****************************************************************************
Get and handle:
- new connections,
- input from connections,
- input from server operator in stdin
Returns:
  0 if went past end-of-turn timeout
  2 if force_end_of_sniff found to be set
  1 otherwise (got and processed something?)
*****************************************************************************/
#ifndef FREECIV_WIN32
int sniff_packets(void)
{
  int i;
  int max_desc;
  fd_set readfs;
  struct timeval tv;
  static time_t time_at_turn_end;
  static int year;
  
  if(year!=game.year) {
    time_at_turn_end = time(NULL) + game.timeout;
    if (server_state == RUN_GAME_STATE) year=game.year;
  }
  
  while(1) {
    if(force_end_of_sniff) {
      force_end_of_sniff=0;
      return 2;
    }
    
    tv.tv_sec=1; tv.tv_usec=0;
    
    MY_FD_ZERO(&readfs);
    FD_SET(0, &readfs);	
    FD_SET(sock, &readfs);
    max_desc=sock;
    
    for(i=0; i<MAX_CONNECTIONS; i++) {
      if(connections[i].used) {
	FD_SET(connections[i].sock, &readfs);
      }
      max_desc=MAX(connections[i].sock, max_desc);
    }
    
    if(select(max_desc+1, &readfs, NULL, NULL, &tv)==0) { /* timeout */
      send_server_info_to_metaserver(0);
      if((game.timeout) 
	&& (time(NULL)>time_at_turn_end)
	&& (server_state == RUN_GAME_STATE)){
	return 0;
      }
      continue;
    }
  
    if(FD_ISSET(sock, &readfs)) {	     /* new players connects */
      freelog(LOG_DEBUG, "got new connection");
      if(server_accept_connection(sock)==-1)
	freelog(LOG_NORMAL, "failed accepting connection");
    }
    else if(FD_ISSET(0, &readfs)) {    /* input from server operator */
      int didget;
      char buf[BUF_SIZE+1];
      
      if((didget=read(0, buf, BUF_SIZE))==-1) {
	freelog(LOG_FATAL, "read from stdin failed");
	exit(1);
      }
      *(buf+didget)='\0';
      handle_stdin_input(buf);

      show_prompt();
    }
    else {                             /* input from a player */
      for(i=0; i<MAX_CONNECTIONS; i++)
	if(connections[i].used && FD_ISSET(connections[i].sock, &readfs)) {
	  if(read_socket_data(connections[i].sock, 
			      &connections[i].buffer)>0) {
	    char *packet;
	    int type;
	    while((packet=get_packet_from_connection(&connections[i], &type)))
	      handle_packet_input(&connections[i], packet, type);
	  }
	  else {
	    lost_connection_to_player(&connections[i]);
	    close_connection(&connections[i]);
	  }
	}
    }
    break;
  }
  if((game.timeout) 
    && (time(NULL)>time_at_turn_end)
    && (game.timeout)) return 0;
  return 1;
}
#endif /* FREECIV_WIN32 */

#ifdef FREECIV_WIN32
/* Now we provide our own version of sniff_packets because */
/* The win32 version does some pretty ugly lookin' stuff that */
/* I didn't want to inflict upon the normal sniff_packets() */
/*****************************************************************************
Get and handle:
- new connections,
- input from connections,
- input from server operator in stdin
Returns:
  0 if went past end-of-turn timeout
  2 if force_end_of_sniff found to be set
  1 otherwise (got and processed something?)
*****************************************************************************/
int sniff_packets(void)
{
  int i;
  int max_desc;
  fd_set readfs;
  struct timeval tv;
  static time_t time_at_turn_end;
  static int year;
  
  int didget = 0; 
  char buf[BUF_SIZE+1] = "";
  DWORD numEventsRead, dwCharsPrinted;
  INPUT_RECORD ir[32];
  int j;
  
  if(year!=game.year) {
    time_at_turn_end = time(NULL) + game.timeout;
    if (server_state == RUN_GAME_STATE) year=game.year;
  }
  
  while(1) {
    if(force_end_of_sniff) {
      force_end_of_sniff=0;
      return 2;
    }
    
    tv.tv_sec=0; tv.tv_usec=500;
    
    /*  MY_FD_ZERO(&readfs); */
    FD_ZERO(&readfs); /* For some reason, Winsock prefers this over MY_FD_ZERO() */
    /*  FD_SET(0, &readfs) */; /* Add stdin to the select()-able fd_set */
    FD_SET(sock, &readfs);
    /*      max_desc=sock; */ /* max_desc is irrelevant under Winsock */
    
      for(i=0; i<MAX_CONNECTIONS; i++) {
        if(connections[i].used) {
	FD_SET(connections[i].sock, &readfs);
      }
      max_desc=MAX(connections[i].sock, max_desc);
    }
    
    if(select(max_desc+1, &readfs, NULL, NULL, &tv)==0) { /* timeout */
      send_server_info_to_metaserver(0);
      
      switch(WaitForSingleObject(hStdIn, 1))
	{
	case WAIT_OBJECT_0:
	  ReadConsoleInput(hStdIn, ir, 32, &numEventsRead);
	  for(j=0; j<numEventsRead; j++)
	    {
	      if(ir[j].EventType == KEY_EVENT && /* We want the key-up event */
		 ir[j].Event.KeyEvent.bKeyDown == TRUE)
		{
		  /* Ignore special keys */
		  if(ir[j].Event.KeyEvent.dwControlKeyState == LEFT_ALT_PRESSED ||
		     ir[j].Event.KeyEvent.dwControlKeyState == LEFT_CTRL_PRESSED ||
		     ir[j].Event.KeyEvent.dwControlKeyState == RIGHT_ALT_PRESSED ||
		     ir[j].Event.KeyEvent.dwControlKeyState == RIGHT_CTRL_PRESSED)
		    {
		      continue;
		    }
		  if((ir[j].Event.KeyEvent.dwControlKeyState == SHIFT_PRESSED) &&
		     (ir[j].Event.KeyEvent.uChar.AsciiChar == 0))
		    {
		      continue; /* If the shift key is pressed and there is no
				   character to be read. Ignore it */
		    }
		  if(ir[j].Event.KeyEvent.uChar.AsciiChar == '\r' || 
		     ir[j].Event.KeyEvent.uChar.AsciiChar == '\n')
		    {
		      buf[didget++] = ir[j].Event.KeyEvent.uChar.AsciiChar;
		      buf[didget] = '\0';
		      WriteConsole(hStdOut, (buf + didget - 1), 1, 
				    &dwCharsPrinted, NULL);
		      handle_stdin_input(buf);
		      show_prompt();
		      buf[0] = '\0';
		      didget = 0;
		    }
		  else
		    {
		      buf[didget++] =  (((ir[j].Event).KeyEvent).uChar.AsciiChar);
		      buf[didget] = '\0';
		      WriteConsole(hStdOut, (buf + didget - 1),
				   1, &dwCharsPrinted, NULL);
		    }
		}
	    }
		  
	  fflush(stdin);
	  FlushConsoleInputBuffer(hStdIn);
	  break;
	case WAIT_TIMEOUT:
	  /* Do nothing */
	  break;
	default:
	  freelog(LOG_DEBUG, "Unhandled return value of WaitForSingleObject.");
	  break;
	}
      if((game.timeout) 
	 && (time(NULL)>time_at_turn_end)
	 && (server_state == RUN_GAME_STATE)){
	return 0;
      }
      continue;
    }
    
    if(FD_ISSET(sock, &readfs)) {	     /* new players connects */
      freelog(LOG_DEBUG, "got new connection");
      if(server_accept_connection(sock)==-1)
	freelog(LOG_NORMAL, "failed accepting connection");
    }
    else {                             /* input from a player */
      for(i=0; i<MAX_CONNECTIONS; i++)
	if(connections[i].used && FD_ISSET(connections[i].sock, &readfs)) {
	  if(read_socket_data(connections[i].sock, 
			      &connections[i].buffer)>0) {
	    char *packet;
	    int type;
	    while((packet=get_packet_from_connection(&connections[i], &type)))
	      handle_packet_input(&connections[i], packet, type);
	  }
	  else {
	    lost_connection_to_player(&connections[i]);
	    close_connection(&connections[i]);
	  }
	}
    }
      break;
    }
  if((game.timeout) 
    && (time(NULL)>time_at_turn_end)
    && (game.timeout)) return 0;
  return 1;
}
#endif /* FREECIV_WIN32 */  
/********************************************************************
 server accepts connections from client
********************************************************************/
int server_accept_connection(int sockfd)
{
  int fromlen;
  int new_sock;
  struct sockaddr_in fromend;
  struct hostent *from;

  fromlen = sizeof fromend;

  new_sock = accept(sockfd, (struct sockaddr *) &fromend, &fromlen);

  from=gethostbyaddr((char*)&fromend.sin_addr, sizeof(sizeof(fromend.sin_addr)), 
		     AF_INET);

  if(new_sock!=-1) {
    int i;
    for(i=0; i<MAX_CONNECTIONS; i++) {
      if(!connections[i].used) {
	connections[i].used=1;
	connections[i].sock=new_sock;
	connections[i].player=NULL;
	connections[i].buffer.ndata=0;
	connections[i].first_packet=1;
	connections[i].byte_swap=0;

	if(from) {
	  strncpy(connections[i].addr, from->h_name, ADDR_LENGTH);
	  connections[i].addr[ADDR_LENGTH-1]='\0';
	}
	else {
	   strcpy(connections[i].addr, "unknown");
	}
	freelog(LOG_DEBUG, "connection from %s", connections[i].addr);
	return 0;
      }
    }
    freelog(LOG_FATAL, "maximum number of connections reached");
    return -1;
  }

  return -1;
}



/********************************************************************
 open server socket to be used to accept client connections
********************************************************************/
int server_open_socket(void)
{
  /* setup socket address */
  struct sockaddr_in src;
  int opt;

  src.sin_addr.s_addr = INADDR_ANY;
  src.sin_family = AF_INET;
  src.sin_port = htons(port);


  /* broken pipes are ignored. */
#ifndef FREECIV_WIN32
  signal (SIGPIPE, SIG_IGN);
#endif /* FREECIV_WIN32 */
  
  if((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    freelog(LOG_FATAL, "socket failed: %s", mystrerror(errno));
    exit(1);
  }

  opt=1; 
  if(setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
		(char*)&opt, sizeof(opt))) {
/*		(const char*)&opt, sizeof(opt))) {      gave me warnings -- Syela */
    freelog(LOG_FATAL, "setsockopt failed: %s", mystrerror(errno));
  }

  if(bind(sock, (struct sockaddr *) &src, sizeof (src)) < 0) {
    freelog(LOG_FATAL, "bind failed: %s", mystrerror(errno));
    exit(1);
  }

  if(listen(sock, MAX_CONNECTIONS) < 0) {
    freelog(LOG_FATAL, "listen failed: %s", mystrerror(errno));
    exit(1);
  }

  return 0;
}


/********************************************************************
...
********************************************************************/
void init_connections(void)
{
  int i;
  
#ifdef FREECIV_WIN32
 /* Code to initialize Windows sockets */
  WSADATA wsa;
  if(WSAStartup(MAKEWORD(1,1), &wsa))
    {
      freelog(LOG_FATAL, "Unable to initialize Winsock 1.1");
      exit(1);
    }
#endif
   for(i=0; i<MAX_CONNECTIONS; i++) { 
    connections[i].used=0;
    connections[i].buffer.ndata=0;
  }
}
