/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.tsfile.compatibility;

import org.apache.tsfile.file.metadata.ChunkGroupMetadata;
import org.apache.tsfile.file.metadata.IDeviceID.Factory;
import org.apache.tsfile.read.TsFileReader;
import org.apache.tsfile.read.TsFileSequenceReader;
import org.apache.tsfile.read.common.Path;
import org.apache.tsfile.read.common.RowRecord;
import org.apache.tsfile.read.expression.QueryExpression;
import org.apache.tsfile.read.query.dataset.QueryDataSet;
import org.apache.tsfile.write.schema.Schema;

import org.junit.Test;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.tsfile.read.TsFileCheckStatus.COMPLETE_FILE;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

public class CompatibilityTest {

  private String fileName =
      "target" + File.separator + "test-classes" + File.separator + "v3TsFile";

  /** The file is generated by the TsFileWriterTest version 3. */
  @Test
  public void testReadV3() {
    readOneRow();
  }

  @Test
  public void testV3SelfCheck() throws IOException {
    Schema schema = new Schema();
    List<ChunkGroupMetadata> chunkGroupMetadataList = new ArrayList<>();
    try (TsFileSequenceReader sequenceReader = new TsFileSequenceReader(fileName)) {
      assertEquals(COMPLETE_FILE, sequenceReader.selfCheck(schema, chunkGroupMetadataList, false));
      assertTrue(sequenceReader.isComplete());
    }
    assertTrue(schema.containsDevice(Factory.DEFAULT_FACTORY.create("d1")));
    assertTrue(schema.containsDevice(Factory.DEFAULT_FACTORY.create("d2")));
    assertEquals(2, chunkGroupMetadataList.size());
  }

  private void readOneRow() {
    readOneRow(5);
  }

  private void readOneRow(int s2Value) {
    try {
      TsFileReader tsFileReader = new TsFileReader(new TsFileSequenceReader(fileName));
      QueryDataSet dataSet =
          tsFileReader.query(
              QueryExpression.create()
                  .addSelectedPath(new Path("d1", "s1", true))
                  .addSelectedPath(new Path("d1", "s2", true))
                  .addSelectedPath(new Path("d1", "s3", true)));
      while (dataSet.hasNext()) {
        RowRecord result = dataSet.next();
        assertEquals(2, result.getFields().size());
        assertEquals(10000, result.getTimestamp());
        assertEquals(5.0f, result.getFields().get(0).getFloatV(), 0.00001);
        assertEquals(s2Value, result.getFields().get(1).getIntV());
      }
      tsFileReader.close();
    } catch (IOException e) {
      e.printStackTrace();
      fail(e.getMessage());
    }
  }
}
