/*****************************************************************************/
/* File name: ActionExtract.java                                             */
/* Purpose: to keep an option and a corresponding flag for Jtar              */
/* Last modified: 23.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*                                                                           */
/* This program is distributed in the hope that it will be useful, but       */
/* WITHOUT ANY WARRANTY; without even the implied warranty of                */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General */
/* Public License for more details.                                          */
/*                                                                           */
/* This software is distributed under the terms                              */
/* of the GNU General Public License as published                            */
/* by the Free Software Foundation; either version 2, or (at your option)    */
/* any later version.                                                        */
/* See file COPYING for your rights (GNU GPL)                                */
/*****************************************************************************/

package jtar;

import java.awt.*;
import java.io.*;
import java.util.*;
import general.Constants;

class

ActionExtract

                   extends ActionRead

     implements jtar.Constants, general.Constants

{

  private final static String RCSID = "$Header: /d/JTAR12/jtar/RCS/ActionExtract.java,v 1.6 1999/05/23 22:35:01 vklimov Exp $";

  private final static String HEADER_DIALOG_TITLE   = "Extract";
  private final static int    HEADER_DIALOG_ROWS    = 23;
  private final static int    HEADER_DIALOG_COLUMNS = 77;

  private static final String OUTPUT_RANDOM_ACCESS_FILE_MODE = "rw";

  private static boolean warnedOnceStrippingLeadingSlash = false;
  private static boolean conttypeDiagnosed = false;
  private final static String EXTRACT_COMPLETED_MSG = "\nExtract completed";


  ActionExtract( char    option,
                 boolean flag   )
/*****************************************************************************/
/* Function name: ActionExtract                                              */
/* File name: ActionExtract.java                                             */
/* Purpose: constructor for the ActionExtract object                         */
/* Parameters: option - 't', 'x', 'c' etc                                    */
/*             flag - corresponding flag                                     */
/* Returns:                                                                  */
/* Last modified: 11.12.98                                                   */
/* Author: Victor Klimov, 1998                                               */
/*****************************************************************************/
    {
                                                    /* ActionExtract */
/*---------------------------------------------------------------------------*/

      super( option, flag );

    }
/*end of ActionExtract*/



  int
  extractSparseFile( RandomAccessFile outputRandomAccessFile,
                     int totalSize, String name,
                     BufferRead readBuffer,
                     Vector sparseVector                       )
/*****************************************************************************/
/* Function name: extractSparseFile                                          */
/* File name: ActionExtract.java                                             */
/* Purpose: to extract a sparse file                                         */
/* Parameters: outputRandomAccessFile - file to write to                     */
/*             totalSize - the total number of bytes to write                */
/*             name - name of the file to write to                           */
/*             readBuffer - an object to read in tar blocks                  */
/*             sparseVector - vector, containing info on a sparse file       */
/* Returns: sizeLeft                                                         */
/* Last modified: 03.05.99                                                   */
/* Author of the Java translation: Victor Klimov, 1999                       */
/*****************************************************************************/
    {
      Block dataBlock = null;
      SpArray previousSpArrayElement;
      SpArray currentSpArrayElement;

      int sizeLeft;
      int sparseInd = 0;
      int written = 0;
                                                  /* extractSparseFile */
/*---------------------------------------------------------------------------*/

      sizeLeft = totalSize;             /* sizeLeft is initially totalSize */

      while( sizeLeft > 0 )
        {
          dataBlock = readBuffer.
            getCurrentDataBlock();

          if (dataBlock == null)
            {
              util.Error.
                reportAndSetExitCode(
                  "ActionExtract.extractSparseFile: " +
                  "unexpected EOF on archive file" );
              return sizeLeft;                              /* return ! */
            }
          else
            ;
        /*end of if on null*/

          try
            {
              previousSpArrayElement = (SpArray)
                sparseVector.elementAt( sparseInd );
              outputRandomAccessFile.
                seek( previousSpArrayElement.offset );

              sparseInd++;
              currentSpArrayElement = (SpArray)
                sparseVector.elementAt( sparseInd );

              written = currentSpArrayElement.numbytes;

              while( written > BLOCKSIZE )
                {
                  outputRandomAccessFile.
                    write( dataBlock.buffer, 0,
                           BLOCKSIZE            );
                  written -= BLOCKSIZE;
                  sizeLeft -= BLOCKSIZE;
                  readBuffer.
                    permitToFillUpNewCurrentBlock();
                  dataBlock = readBuffer.
                    getCurrentDataBlock();

                  if (dataBlock == null)
                    {
                      util.Error.
                        reportAndSetExitCode(
                          "ActionExtract.extractSparseFile: " +
                          " while-loop on written,\n" +
                          "unexpected EOF on archive file" );
                      return sizeLeft;                          /* return ! */
                    }
                  else
                    ;
                /*end of if on null*/
                }
            /*end of while-loop on written*/

              outputRandomAccessFile.
                write( dataBlock.buffer, 0,
                       written              );
            }
          catch( ArrayIndexOutOfBoundsException e )
            {
              util.Error.warn(
                "\nShould not happen:\n" +
                e.getMessage()             );
            }
          catch( IOException e )
            {
              util.Error.
                reportAndSetExitCode( name +
                  ": Could not write to file" +
                  e.getMessage()             );
            }
        /*end of try-catch on writing*/

          sizeLeft -= written;
          readBuffer.
            permitToFillUpNewCurrentBlock();

        }
    /*end of while-loop on sizeLeft*/

      readBuffer.
        permitToFillUpNewCurrentBlock();

      return sizeLeft;

    }
/*end of extractSparseFile*/


  int
  extractRegularFile( RandomAccessFile outputRandomAccessFile,
                      StringBuffer currentFileNameStringBuffer,
                      BufferRead readBuffer                   )
/*****************************************************************************/
/* Function name: extractRegularFile                                         */
/* File name: ActionExtract.java                                             */
/* Purpose: to extract a regular file                                        */
/* Parameters: outputRandomAccessFile - file to write to                     */
/*             currentFileNameStringBuffer                                   */
/*             readBuffer - an object to read in tar blocks                  */
/* Returns: totalWritten                                                     */
/* Last modified: 12.12.98                                                   */
/* Author of the Java translation: Victor Klimov, 1998                       */
/*****************************************************************************/
    {
      Block dataBlock = null;
      int size;
      int written = 0;
      int totalWritten = 0;
                                                  /* extractRegularFile */
/*---------------------------------------------------------------------------*/

      for (size = AllActions.currentStat.size;
           size > 0;
           size -= written)
        {
          if (AllActions.multiVolume)
            {
              readBuffer.saveName = new
                String( currentFileNameStringBuffer );
              readBuffer.saveTotsize =
                AllActions.currentStat.size;
            }
          else
            ;
        /*end of if on multiVolume*/


          written = readBuffer.
            availableSpaceAfter();

          dataBlock = readBuffer.
            getCurrentDataBlock();

          if (dataBlock == null)
            {
              util.Error.
                reportAndSetExitCode(
                  "ActionExtract.extractRegularFile: " +
                  "unexpected EOF on archive file" );
              break;                                  /* break ! */
            }
          else
            ;
        /*end of if on null*/

          if (written == 0)
            written =
              dataBlock.buffer.length;
          else
            ;
        /*end of if on written*/

          if (written > size)
            written = size;
          else
            ;
        /*end of if on written*/

          if (written > dataBlock.buffer.length)
            written =
              dataBlock.buffer.length;
          else
            ;
        /*end of if on written*/

          try
            {
              outputRandomAccessFile.
                write(
                  dataBlock.buffer, 0,
                  written              );
              totalWritten += written;
            }
          catch( IOException e )
            {
              util.Error.
                reportAndSetExitCode(
                    currentFileNameStringBuffer.
                     toString()                  +
                  ": Error writing to file"   );

              readBuffer.
                skipFile( (size - written) );
              break;                                   /* break ! */
            }
        /*end of try-catch*/

          readBuffer.
            permitToFillUpNewCurrentBlock();
        }
    /*end of for-loop on size*/

      return totalWritten;

    }
/*end of extractRegularFile*/


  void
  extractFile( StringBuffer currentFileNameStringBuffer,
               RandomAccessFile outputRandomAccessFile,
               Header tarHeader,
               BufferRead readBuffer,
               Vector sparseVector                       )
/*****************************************************************************/
/* Function name: extractFile                                                */
/* File name: ActionExtract.java                                             */
/* Purpose: to extract a file                                                */
/* Parameters: currentFileNameStringBuffer                                   */
/*             outputRandomAccessFile - file to write to                     */
/*             tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/*             sparseVector - vector, containing info on a sparse file       */
/* Returns:                                                                  */
/* Last modified: 12.12.98                                                   */
/* Author: Victor Klimov, 1998                                               */
/*****************************************************************************/
    {
      String name;
      int size;
                                                   /* extractFile */
/*---------------------------------------------------------------------------*/

      if (tarHeader.currentHeader.typeflag == GNUTYPE_SPARSE)
        {
          name = new String( currentFileNameStringBuffer );
          size = extractSparseFile(
                   outputRandomAccessFile, 
                   AllActions.currentStat.size,
                   name,
                   readBuffer,
                   sparseVector                       );
        }
      else /* something other, Not GNUTYPE_SPARSE */
        {
          size = 
            extractRegularFile(
              outputRandomAccessFile,
              currentFileNameStringBuffer,
              readBuffer                   );
        }
    /*end of if on GNUTYPE_SPARSE*/

      if (AllActions.multiVolume)
        readBuffer.saveName = null;
      else
        ;
    /*end of if on multiVolume*/

    }
/*end of extractFile*/


  void
  processOutputFileCreationError(
    Exception e,
    StringBuffer currentFileNameStringBuffer,
    Header tarHeader,
    BufferRead readBuffer,
    Vector sparseVector,
    io.TextDisplayingDialog headerListDialog   )
/*****************************************************************************/
/* Function name: processOut                                                 */
/* File name: ActionExtract.java                                             */
/* Purpose: to process an error in output file creation                      */
/* Parameters: e - the occured exception                                     */
/*             currentFileNameStringBuffer                                   */
/*             tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/*             sparseVector - vector, containing info on a sparse file       */
/*             headerListDialog                                              */
/* Returns:                                                                  */
/* Last modified: 03.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      boolean resultAgainFile;
      boolean neededCreateIntermediateDirs;
                                       /* processOutputFileCreationError */
/*---------------------------------------------------------------------------*/

                                           /* maybe there aren't */
                                           /* intermediate dirs  */

      neededCreateIntermediateDirs =
        MakeDirectories.
          makeDirectories(
            currentFileNameStringBuffer.toString(),
            tarHeader,
            readBuffer,
            headerListDialog                       );
      if (neededCreateIntermediateDirs)
        {
          resultAgainFile =
            againFile(
              currentFileNameStringBuffer,
              tarHeader,
              readBuffer,
              sparseVector,
              headerListDialog             );
          if (resultAgainFile)
            return;                                    /* return ! */
          else
            ;
        /*end of if on resultAgainFile*/
        }
      else
        {
           util.Error.
             reportAndSetExitCode(
                  currentFileNameStringBuffer.
                    toString()                 +
               ": Could not create file\n" +
               "ActionExtract." +
               "processOutputFileCreationError:\n" +
               e.getMessage()                        );
        }
    /*end of if on neededCreateIntermediateDirs*/

      if (tarHeader.currentHeader.isextended != 0)
        readBuffer.skipExtendedHeaders();
      else
        ;
    /*end of if on isextended*/

      readBuffer.skipFile(
        AllActions.currentStat.size );

      if (AllActions.backup)
        io.Backup.undoLastBackup();
      else
        ;
    /*end of if on backup*/

    }
/*end of processOutputFileCreationError*/


  boolean
  againFile( StringBuffer currentFileNameStringBuffer,
             Header tarHeader,
             BufferRead readBuffer,
             Vector sparseVector,
             io.TextDisplayingDialog headerListDialog   )
/*****************************************************************************/
/* Function name: againFile                                                  */
/* File name: ActionExtract.java                                             */
/* Purpose: to process additionaly a directory                               */
/* Parameters: currentFileNameStringBuffer                                   */
/*             tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/*             sparseVector - vector, containing info on a sparse file       */
/*             headerListDialog                                              */
/* Returns: true - if everything is all right, false - otherwise             */
/* Last modified: 03.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      File outputFile;
      RandomAccessFile outputRandomAccessFile;
                                                   /* againFile */
/*---------------------------------------------------------------------------*/

      if (tarHeader.currentHeader.typeflag == CONTTYPE)
        {

          if (!conttypeDiagnosed)
            {
              conttypeDiagnosed = true;
              util.Error.
                warn( "Extracting contiguous " +
                      "files as regular files" );
            }
          else
            ;
        /*end of if on conttypeDiagnosed*/
        }
      else /* Not CONTTYPE */
        ;
    /*end of if on CONTTYPE*/

      try
        {
          outputFile = new
            File( currentFileNameStringBuffer.toString() );
          outputRandomAccessFile = new
            RandomAccessFile( outputFile,
                              OUTPUT_RANDOM_ACCESS_FILE_MODE );
        }
      catch( NullPointerException e)
        {
          return false;                                      /* return ! */
        }
      catch( IllegalArgumentException e)
        {
          return false;                                      /* return ! */
        }
      catch( SecurityException e)
        {
          processOutputFileCreationError(
            e,
            currentFileNameStringBuffer,
            tarHeader,
            readBuffer,
            sparseVector,
            headerListDialog    );

          return false;                                      /* return ! */
        }
      catch( IOException e)
        {
          processOutputFileCreationError(
            e,
            currentFileNameStringBuffer,
            tarHeader,
            readBuffer,
            sparseVector,
            headerListDialog                 );

          return false;                                      /* return ! */
        }
    /*end of try-catch*/

      extractFile( currentFileNameStringBuffer,
                   outputRandomAccessFile,
                   tarHeader,  
                   readBuffer,
                   sparseVector                 );

      try
        {
          outputRandomAccessFile.close();
        }
      catch( IOException e)
        {
          util.Error.
            reportAndSetExitCode(
                currentFileNameStringBuffer.
                  toString()                 +
              ": Error while closing"       );
          if (AllActions.backup)
            io.Backup.undoLastBackup();
          else
            ;
        /*end of if on backup*/
          return false;                                      /* return ! */
        }
    /*end of try-catch*/

      return true;

    }
/*end of againFile*/


  boolean
  againDir( StringBuffer currentFileNameStringBuffer,
            BufferRead readBuffer                     )
/*****************************************************************************/
/* Function name: againDir                                                   */
/* File name: ActionExtract.java                                             */
/* Purpose: to process additionaly a directory                               */
/* Parameters: currentFileNameStringBuffer                                   */
/*             readBuffer - an object to read in tar blocks                  */
/* Returns: true - if everything is all right, false - otherwise             */
/* Last modified: 03.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      boolean removedOK;
      boolean resultAgainDir;

      int nameLength;
      int pathSeparatorIndexLast;

      boolean created;

      File currentFile = null;
                                                   /* againDir */
/*---------------------------------------------------------------------------*/

      try
        {
          currentFile = new
            File( currentFileNameStringBuffer.toString() );

          created = currentFile.mkdirs();
          if (!created)
            {
              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/

              return false;                                /* return ! */

            }
          else
            ;
        /*end of if on created*/
        }
      catch( NullPointerException e )
        {
          readBuffer.close();
          util.Error.warn(
            "can\'t create directory with no name" );
          return false;                                /* return ! */
        }
      catch( SecurityException e )
        {
          if ( (currentFile != null ) &&
              (currentFile.exists())
             )
            {
              if ( ! AllActions.
                    keepOldFiles )
                {
                  removedOK =
                    io.Remove.
                      removeAnyFile(
                        currentFileNameStringBuffer.toString(),
                        false /* not recursive */               );
                  if (removedOK)
                    resultAgainDir =
                      againDir(
                        currentFileNameStringBuffer,
                        readBuffer                    );
                  else
                    return false;                       /* return ! */
                /*end of if on removedOK*/
                }
              else
                ;
            /*end of if on keepOldFiles*/
            }
          else
            ;
        /*end of if on exists*/

          if (AllActions.backup)
            io.Backup.undoLastBackup();
          else
            ;
        /*end of if on backup*/

          util.Error.
            reportAndSetExitCode(
                currentFileNameStringBuffer.
                  toString()                  +
              ": Could not create directory" );

          return false;                             /* return ! */

        }
    /*end of try-catch on mkdir*/

      return true;

    }
/*end of againDir*/


  void
  extractDirectory( StringBuffer currentFileNameStringBuffer,
                    Header tarHeader,
                    BufferRead readBuffer                    )
/*****************************************************************************/
/* Function name: extractDirectory                                           */
/* File name: ActionExtract.java                                             */
/* Purpose: to extract a directory                                           */
/* Parameters: currentFileNameStringBuffer                                   */
/*             tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/* Returns:                                                                  */
/* Last modified: 12.12.98                                                   */
/* Author: Victor Klimov, 1998                                               */
/*****************************************************************************/
    {
      boolean resultAgainDir;
                                                   /* extractDirectory */
/*---------------------------------------------------------------------------*/

                                                    /* Check for trailing /, */
                                                    /* and zap as many       */
                                                    /* as we find            */
      currentFileNameStringBuffer =
        new StringBuffer(
          util.String.removeTrailingPathSeparator(
            currentFileNameStringBuffer.toString() ) );

                                      /* Read the entry and delete files   */
                                      /* that aren't listed in the archive */
      if (AllActions.incremental.flag)
        {
          /* gnuRestore(skipcrud); */

            readBuffer.
              skipFile(
                AllActions.currentStat.size);
        }
      else /* regular, not incremental */
        {
          if (tarHeader.currentHeader.typeflag ==
              GNUTYPE_DUMPDIR)
            {
            readBuffer.
              skipFile(
                AllActions.currentStat.size);
            }
          else
            ;
        /*end of if on GNUTYPE_DUMPDIR*/
        }
    /*end of if on incremental*/

      resultAgainDir =
        againDir( currentFileNameStringBuffer,
                  readBuffer                   );

    }
/*end of extractDirectory*/


  Vector
  doFirstThingsForSparse( Header tarHeader,
                          BufferRead readBuffer )
/*****************************************************************************/
/* Function name: doFirstThingsForSparse                                     */
/* File name: ActionExtract.java                                             */
/* Purpose: to do some preliminary things for the sparse case                */
/*                                                                           */
/*          JK - What we want to do if the file is sparse is loop through    */
/*          the array of sparse structures in the header and read in and     */
/*          translate the character strings representing 1) the offset at    */
/*          which to write and 2) how many bytes to write into numbers,      */
/*          which we store into the scratch array, "sparseVector".  This     */
/*          array makes our life easier the same way it did in creating      */
/*          the tar file that had to deal with a sparse file.                */
/*                                                                           */
/*          After we read in the first five (at most) sparse structures,     */
/*          we check to see if the file has an extended header, i.e.,        */
/*          if more sparse structures are needed to describe the contents    */
/*          of the new file.  If so, we read in the extended headers and     */
/*          continue to store their contents into the sparseVector.          */
/*                                                                           */
/* Parameters: tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/* Returns: sparseVector - if everything is all right, null - otherwise      */
/* Last modified: 12.12.98                                                   */
/* Author of the Java translation: Victor Klimov, 1998                       */
/*****************************************************************************/
    {
      int ind;
      int counter;
      int spArraySize;
      int numbytes;
      Vector sparseVector;
      SpArray tempSpArray;
      Block exhdr = null;
                                               /* doFirstThingsForSparse */
/*---------------------------------------------------------------------------*/

      spArraySize = SP_VECTOR_INIT_SIZE;
      sparseVector = new Vector( spArraySize );

      for (counter = 0;
           counter < SPARSES_IN_OLDGNU_HEADER;
           counter++
          )
        {
          tempSpArray = new SpArray();
          tempSpArray.offset = Octal.fromOct(
              SPARSE_DESCRIPTOR_FIELD_LENGTH_OFFSET,
              tarHeader.currentHeader.sp[counter].offset);
          tempSpArray.numbytes = Octal.fromOct(
              SPARSE_DESCRIPTOR_FIELD_LENGTH_NUMBYTES,
              tarHeader.currentHeader.sp[counter].numbytes);

          sparseVector.addElement(
            (Object) tempSpArray   );

          if (tempSpArray.numbytes == 0 )
            break;                             /* break ! */
          else
            ;
        /*end of if on numbytes*/
        }
    /*end of for-loop on sparses*/

      if (tarHeader.currentHeader.isextended != 0)
        {
                                             /* read in the list     */
                                             /* of extended headers  */
                                             /* and translate them   */
                                             /* into the sparseVector*/
                                             /* as before            */

          ind = SPARSES_IN_OLDGNU_HEADER;

          while( true )
            {
              exhdr = readBuffer.
                getCurrentDataBlock();

              if (exhdr == null) /* end of file */
                {
                  readBuffer.close();
                  return null;                          /* return ! */
                }
              else
                ;
            /*end of if on null*/

              for (counter = 0;
                   counter < SPARSES_IN_SPARSE_HEADER;
                   counter++
                  )
                {
                  numbytes = Octal.fromOct(
                    SPARSE_DESCRIPTOR_FIELD_LENGTH_NUMBYTES,
                    exhdr.sp[counter].numbytes               );
                  if (numbytes == 0)
                    break;                             /* break ! */
                  else
                    ;
                /*end of if on 0*/

                  tempSpArray = new SpArray();
                  tempSpArray.offset = Octal.fromOct(
                    SPARSE_DESCRIPTOR_FIELD_LENGTH_OFFSET,
                    exhdr.sp[counter].offset                 );
                  tempSpArray.numbytes = Octal.fromOct(
                    SPARSE_DESCRIPTOR_FIELD_LENGTH_NUMBYTES,
                    exhdr.sp[counter].numbytes               );

                  sparseVector.addElement(
                    (Object) tempSpArray   );
                }
            /*end of for-loop on counter*/

              if ( exhdr.isextended != 0 )
                break; /* regular header */            /* break ! */
              else /* extended header */
                {
                  ind += SPARSES_IN_SPARSE_HEADER;
                  readBuffer.
                    permitToFillUpNewCurrentBlock();
                }
            /*end of if on isextended*/
            }
        /*end of while-loop*/

          readBuffer.
            permitToFillUpNewCurrentBlock();

        }
      else /* no extended headers */
        ;
    /*end of isextended*/

      return sparseVector;

    }
/*end of doFirstThingsForSparse*/


  boolean
  scanCommandLineOptionsVector(
    Vector commandLineOptionsVector,
    String currentFileNameString        )
/*****************************************************************************/
/* Function name: scanCommandLineOptionsVector                               */
/* File name: ActionExtract.java                                             */
/* Purpose: to scan the list of remaining options (that should be only       */
/*          file names) and check whether the current file name is in there  */
/* Parameters: commandLineOptionsVector - list of the file names             */
/*             currentFileNameString                                         */
/* Returns: true - current file name is contained in the options list        */
/*          false - not found                                                */
/* Last modified: 23.05.99                                                   */
/* Author of the Java translation: Victor Klimov, 1999                       */
/*****************************************************************************/
    {
      boolean continueToLoop = true;
      int i;
      int nFileNames;
      String aFileName;
                                       /* scanCommandLineOptionsVector */
/*---------------------------------------------------------------------------*/

      if (commandLineOptionsVector == null)
                                       /* no file names - extract everything */
        return true;                                            /* return ! */
      else
        {
          nFileNames =
            commandLineOptionsVector.size();
          if (nFileNames == 0) /* no file names - extract everything */
            return true;                                        /* return ! */
          else
            ;
        /*end of if on nFileNames*/
        }
    /*end of if on null*/

      currentFileNameString =
        util.String.
          removeTrailingPathSeparator(
            currentFileNameString      );
      
      for ( i = 0;
            continueToLoop &&
            (i < nFileNames);   i++ )
        {
          aFileName = (String)
            commandLineOptionsVector.elementAt( i );
          aFileName =
            util.String.
              removeTrailingPathSeparator(
                aFileName                  );
          if ( ( aFileName.equals(
                 currentFileNameString ) )        ||
               currentFileNameString.startsWith(
                 aFileName + File.separator    )
             )
            continueToLoop = false;
          else
            ;
        /*end of if*/
        }
    /*end of for-loop*/

      return !continueToLoop;

    }
/*end of scanCommandLineOptionsVector*/

  boolean
  prepareToExtractArchiveEntry(
    Header tarHeader,
    BufferRead readBuffer,
    StringBuffer currentFileNameStringBuffer,
    Vector commandLineOptionsVector,
    io.TextDisplayingDialog headerListDialog )
/*****************************************************************************/
/* Function name: prepareToExtractArchiveEntry                               */
/* File name: ActionExtract.java                                             */
/* Purpose: to prepare to extract an element from a tar archive              */
/* Parameters: tarHeader - an object to handle tar headers                   */
/*             readBuffer - an object to read in tar blocks                  */
/*             currentFileNameStringBuffer                                   */
/*             commandLineOptionsVector                                      */
/*             headerListDialog                                              */
/* Returns: true - the current entry must be skipped                         */
/*          false - go on and extract the current entry                      */
/* Last modified: 19.05.99                                                   */
/* Author of the Java translation: Victor Klimov, 1999                       */
/*****************************************************************************/
    {
      boolean backupMadeOrUnneeded = false;
      boolean needToExtractThisOne = false;
      int pathSeparatorLength = File.separator.length();
      String currentFileNameStringNew;

                                          /* prepareToExtractArchiveEntry */
/*---------------------------------------------------------------------------*/

      readBuffer.
        permitToFillUpNewCurrentBlock();
      tarHeader.decodeHeader( false ); /* no user group yet */

      needToExtractThisOne =
        scanCommandLineOptionsVector(
          commandLineOptionsVector,
          currentFileNameStringBuffer.toString() );

      if ( !needToExtractThisOne ||
           ( AllActions.interactive &&
             ! io.Interaction.confirm(
                 MESSAGE_ACTION_STRING_EXTRACT,
                 currentFileNameStringBuffer.toString() )
           )
         )
        {
          if ( tarHeader.currentHeader.isextended != 0 )
            readBuffer.skipExtendedHeaders();
          else
            ;
        /*end of if*/

          readBuffer.skipFile(
            AllActions.
              currentStat.size );

          return true;                                 /* return ! */

        }
      else
        ;
    /*end of if on interactive*/
                                                 /* print the block     */
                                                 /* from currentHeader */
                                                 /* and currentStat    */
      tarHeader.printHeader( (Buffer) readBuffer,
                             headerListDialog,
                             false, null          );

                                 /* Check for fully specified file names */
                                 /* and other atrocities                 */

      currentFileNameStringNew =
        util.String.
          removeLeadingPathSeparator(
            currentFileNameStringBuffer.
              toString()                  );

      if ( ! currentFileNameStringNew.equals(
               currentFileNameStringBuffer.
                 toString()                 )  &&
           ! warnedOnceStrippingLeadingSlash )
        {
          warnedOnceStrippingLeadingSlash = true;
          util.Error.warn( "Removing leading " +
                           File.separator +
                           "from absolute path names " +
                           "in the archive"                 );
        }
      else
        ;
    /*end of if on warnedOnceStrippingLeadingSlash*/

      currentFileNameStringBuffer =
        new StringBuffer(
          currentFileNameStringNew );

                                        /* Take a safety backup          */
                                        /* of a previously existing file */
      if ( AllActions.backup &&
           ! AllActions.toStdout )
        {
          backupMadeOrUnneeded =
            io.Backup.checkPathMakeBackup(
              currentFileNameStringBuffer.toString(),
              false ); /* not an archive file */

          if ( ! backupMadeOrUnneeded )
            {
              util.Error.reportAndSetExitCode(
                  currentFileNameStringBuffer.
                    toString()                  +
                ": Was unable to backup this file" );

              if (tarHeader.currentHeader.isextended != 0 )
                readBuffer.skipExtendedHeaders();
              else
                ;
            /*end of if on isextended*/

              readBuffer.skipFile(
                AllActions.currentStat.size);

              return true;                                 /* return ! */

            }
          else /* backup made OK */
            ;
        /*end of if on backupMadeOrUnneeded*/

        }
      else /* no need to backup */
        ;
    /*end of if*/

      return false;

    }
/*end of prepareToExtractArchiveEntry*/


  boolean
  action( Vector commandLineOptionsVector,
          Header dummyTarHeader,
          BufferRead dummyReadBuffer,
          Frame parentWindow                )
/*****************************************************************************/
/* Function name: action to extract archive members                          */
/* File name: ActionExtract.java                                             */
/* Purpose: to extract elements from a tar archive                           */
/* Parameters: commandLineOptionsVector - vector, containig strings          */
/*                                 of command line arguments to Jtar         */
/*             dummyTarHeader                                                */
/*             dummyReadBuffer                                               */
/*             parentWindow                                                  */
/* Returns: true - if everything is all right, false - otherwise             */
/* Last modified: 03.05.99                                                   */
/* Author of the Java translation: Victor Klimov, 1999                       */
/*****************************************************************************/
    {
      boolean continueToLoop = true;
      boolean skipCurrentEntry;

      int written;
      int nameLength;
      int pathSeparatorLength = File.separator.length();
      int pathSeparatorIndexLast;

      boolean readOK = true;

      Vector sparseVector = null;

      BufferRead readBuffer = new BufferRead();
      Header tarHeader = new Header();
      StringBuffer currentFileNameStringBuffer;

      boolean resultSuperAction = true;
                                                          /* action */
/*---------------------------------------------------------------------------*/

      readOK =
        readBuffer.open( parentWindow );

      if (!readOK)
        return false;                                  /* return ! */
      else
        ;
    /*end of if on readOK*/

      io.TextDisplayingDialog headerListDialog = new
        io.TextDisplayingDialog( parentWindow,
                              HEADER_DIALOG_TITLE,
                              "",          /* originally nothing to display */
                              HEADER_DIALOG_ROWS,
                              HEADER_DIALOG_COLUMNS,
                              false /* not modal, don't wait for anything */ );
      headerListDialog.show();

      while( continueToLoop )
        {
          resultSuperAction =
            super.action( commandLineOptionsVector,
                          tarHeader,
                          readBuffer,
                          parentWindow               );

          if ( (tarHeader.status == HEADER_ZERO_BLOCK) ||
               (tarHeader.status == HEADER_END_OF_FILE) ||
               (tarHeader.status == HEADER_FAILURE)
             )
            {

            break;                                          /* break ! */

            }
          else
            ;
        /*end of if on status*/

          currentFileNameStringBuffer = new
            StringBuffer(
              c.String.truncateNull(
                AllActions.currentFileName ) );

          skipCurrentEntry =
            prepareToExtractArchiveEntry(
              tarHeader,
              readBuffer,
              currentFileNameStringBuffer,
              commandLineOptionsVector,
              headerListDialog             );

          if (skipCurrentEntry)
            {
              continue;                                  /* continue ! */
            }
          else
            ;
        /*end of if on skipCurrentEntry*/
                                               /* extract the archive entry */
                                               /* according to its type     */

          switch( tarHeader.currentHeader.typeflag)
            {

            case GNUTYPE_SPARSE: /* ------------ case ------------- */

              sparseVector =
                doFirstThingsForSparse(
                  tarHeader,
                  readBuffer            );

              if ( sparseVector == null )
                {
                  continueToLoop = false;
                  break;                                 /* break ! */
                }
              else
                ;
            /*end of if on resultDoFirstThingsForSparse*/

            /* Fall through.  */

            case AREGTYPE: /* ------------ case ------------- */
            case REGTYPE:
            case CONTTYPE:
                                                   /* Appears to be a file */
                                                   /* But BSD tar uses     */
                                                   /* the convention that  */
                                                   /* a slash suffix means */
                                                   /* a directory          */

              nameLength = currentFileNameStringBuffer.length();

              pathSeparatorIndexLast =
                currentFileNameStringBuffer.toString().
                  lastIndexOf( File.separator,
                               nameLength -
                                 pathSeparatorLength );

              if ( ( pathSeparatorIndexLast +
                     pathSeparatorLength      ==
                     nameLength               )   &&
                   ( pathSeparatorIndexLast !=
                     JAVA_ERROR_CODE           )
                 )
                extractDirectory(
                  currentFileNameStringBuffer,
                  tarHeader,
                  readBuffer                   );
              else
                skipCurrentEntry = !
                  againFile( currentFileNameStringBuffer,
                             tarHeader,
                             readBuffer,
                             sparseVector,
                             headerListDialog                 );
            /*end of if on trailing slash*/

              break;

            case SYMTYPE: /* ------------ case ------------- */

              util.Error.
                reportAndSetExitCode(
                    currentFileNameStringBuffer.
                     toString()                  +
                  ": the current version does " +
                  "not know how to create symlinks" );

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/

              break;

            case LNKTYPE: /* ------------ case ------------- */

              util.Error.
                reportAndSetExitCode(
                    currentFileNameStringBuffer.
                     toString()                  +
                  ": the current version does " +
                  "not know how to create links"  );

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/

              break;

            case CHRTYPE: /* ------------ case ------------- */
            case BLKTYPE: /* ------------ case ------------- */

              util.Error.
                reportAndSetExitCode(
                    currentFileNameStringBuffer.
                     toString()                  +
                  ": the current version does " +
                  "not know how to make os nodes"       );

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
             /*end of if on backup*/

              break;

            case FIFOTYPE: /* ------------ case ------------- */

              util.Error.
                reportAndSetExitCode(
                    currentFileNameStringBuffer.
                     toString()                  +
                  ": the current version does " +
                  "not know how to make fifo pipe files" );

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/

              break;

            case DIRTYPE: /* ------------ case ------------- */
            case GNUTYPE_DUMPDIR:

              nameLength =
                currentFileNameStringBuffer.length();

              extractDirectory(
                  currentFileNameStringBuffer,
                  tarHeader,
                  readBuffer                   );

              break;

            case GNUTYPE_VOLHDR: /* ------------ case ------------- */

              if (AllActions.verbose.flag)
                headerListDialog.append( 
                  "Reading " +
                  currentFileNameStringBuffer );
              else
                ;
            /*end of if on verbose*/

              break;

            case GNUTYPE_NAMES: /* ------------ case ------------- */
              /* extractMangle(); */
              util.Error.
                reportAndSetExitCode(
                  "The current version can " +
                  "not handle GNUTYPE_NAMES records: " +
                    currentFileNameStringBuffer.
                     toString()                            );

              readBuffer.skipFile(
                AllActions.currentStat.size);

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/
              break;

            case GNUTYPE_MULTIVOL: /* ------------ case ------------- */

              util.Error.
                reportAndSetExitCode(
                  "The current version can " +
                  "not extract " +
                    currentFileNameStringBuffer.
                     toString()                  +
                  " -- file is continued from another volume" );

              readBuffer.skipFile(
                AllActions.currentStat.size);

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/
              break;

            case GNUTYPE_LONGNAME: /* ------------ case ------------- */
            case GNUTYPE_LONGLINK:

              util.Error.
                reportAndSetExitCode(
                  "Visible long name error" );

              readBuffer.skipFile(
                AllActions.currentStat.size);

              if (AllActions.backup)
                io.Backup.undoLastBackup();
              else
                ;
            /*end of if on backup*/
              break;

            default:

              util.Error.warn(
                "Unknown file type " +
                tarHeader.currentHeader.typeflag +
                " for " + currentFileNameStringBuffer +
                " extracted as normal file"            );

              againFile( currentFileNameStringBuffer,
                         tarHeader,
                         readBuffer,
                         sparseVector,
                         headerListDialog                  );
            }
        /*end of switch on typeflag*/

        }
    /*end of while-loop*/

      headerListDialog.append( EXTRACT_COMPLETED_MSG );

      readBuffer.close();
      Names.namesNotFound();            /* print names not found */

      return resultSuperAction;

    }
/*end of action*/


} /*end of class ActionExtract */

/*****************************************************************************/
/* End of file: ActionExtract.java                                           */
/*****************************************************************************/
