/*****************************************************************************/
/* File name: RamFile.java                                                   */
/* Purpose: to present a filesystem-like directory and file tree             */
/*                                                                           */
/* Last modified: 20.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/* For the list of authors and contributors of the original C version        */
/* of the GNU tar see files AUTHORS and THANKS                               */
/*                                                                           */
/* This program is distributed in the hope that it will be useful, but       */
/* WITHOUT ANY WARRANTY; without even the implied warranty of                */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General */
/* Public License for more details.                                          */
/*                                                                           */
/* This software is distributed under the terms                              */
/* of the GNU General Public License as published                            */
/* by the Free Software Foundation; either version 2, or (at your option)    */
/* any later version.                                                        */
/* See file COPYING for your rights (GNU GPL)                                */
/*****************************************************************************/

package io;

import java.io.*;
import java.util.*;
import util.*;


public
class

RamFile
                           implements Filable

{

  private final static java.lang.String RCSID = "$Header: /d/JTAR12/io/RCS/RamFile.java,v 1.4 1999/05/21 02:02:45 vklimov Exp $";

  protected java.lang.String path = null;
  protected java.lang.String name = null;
  protected RamFile parentDirectory = null;
  protected RamFileVector elements = null;
  protected int type = THIS_IS_A_DIRECTORY;

  private final static java.lang.String TO_STRING_INDENT_STRING = "  ";

  private final static int MAX_DEPTH = 10;
  private static int depth = 0;


  public
  RamFile()
/*****************************************************************************/
/* Function name: RamFile                                                    */
/* File name: RamFile.java                                                   */
/* Purpose: constructor for the root dir of the RamFile class                */
/* Parameters:                                                               */
/* Returns:                                                                  */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                           /* RamFile */
/*---------------------------------------------------------------------------*/

      this.name = File.separator;
      this.path = File.separator;

    }
/*end of RamFile*/


  private
  RamFile( java.lang.String name,
           java.lang.String path,
           int type,
           RamFile parentDirectory )
/*****************************************************************************/
/* Function name: RamFile                                                    */
/* File name: RamFile.java                                                   */
/* Purpose: constructor for the RamFile class                                */
/* Parameters: name                                                          */
/*             path                                                          */
/*             type                                                          */
/*             parentDirectory                                               */
/* Returns:                                                                  */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                           /* RamFile */
/*---------------------------------------------------------------------------*/

      this.name = name;
      this.path = path;
      this.type = type;
      this.parentDirectory = parentDirectory;

    }
/*end of RamFile*/


  public
  boolean

  addElement( java.lang.String path,
              int type               )
/*****************************************************************************/
/* Function name: addElement                                                 */
/* File name: RamFile.java                                                   */
/* Purpose: to insert an element into the tree                               */
/*          This method scans the tree from the root and determines          */
/*          the place where to put the element according to the rules        */
/*          of the local OS                                                  */
/*    Note: leading and trailing path separators are removed                 */
/* Parameters: path                                                          */
/*             type - should be io.Filable.THIS_IS_A_DIRECTORY or            */
/*                              io.Filable.THIS_IS_A_REGULAR_FILE            */
/* Returns: true - if added OK, false - otherwise                            */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      boolean result = false;
      java.lang.String originalPath = null;
      java.lang.String firstDirName = null;
      RamFile existingDir = null;
      int separatorPlaceInThePath =
            general.Constants.JAVA_ERROR_CODE;
      int pathPlaceInElementsVector =
            general.Constants.JAVA_ERROR_CODE;
                                                         /* addElement */
/*---------------------------------------------------------------------------*/

      if (elements == (RamFileVector) null)
        elements = new RamFileVector();
      else
        ;
    /*end of if on null*/

      path = path.trim();

      if ( path.startsWith( File.separator ) )
        path =
          util.String.
            removeLeadingPathSeparator( path );
      else
        ;
    /*end of if*/

      if ( path.endsWith( File.separator ) )
        path =
          util.String.
            removeTrailingPathSeparator( path );
      else
        ;
    /*end of if*/

      originalPath = path;

      separatorPlaceInThePath =
        path.indexOf( File.separator );

      if (separatorPlaceInThePath ==                   /* single file name */
          general.Constants.JAVA_ERROR_CODE)
        {
          pathPlaceInElementsVector =
            elements.indexOfAName( path );

          if (pathPlaceInElementsVector !=            /* file already exists */
              general.Constants.JAVA_ERROR_CODE)
            {
              return false;                                  /* return ! */
            }
          else                       /* not found, may add the element here */
            {                               /* first order element */
               elements.
                  addElement( new RamFile( path, path, type, this ) );

               return true;                                   /* return ! */

            }
        /*end of if on pathPlaceInElementsVector*/

        }
      else /* this is a path, containing at least one separator */
        {
          firstDirName =
            path.substring( 0, separatorPlaceInThePath );
          path =
            path.substring( separatorPlaceInThePath );

          pathPlaceInElementsVector =
            elements.indexOfAName( firstDirName );

          if (pathPlaceInElementsVector !=           /* something does exist */
              general.Constants.JAVA_ERROR_CODE)
            {
              try
                {
                  existingDir = (RamFile)
                    elements.elementAt(
                      pathPlaceInElementsVector );
                  if (existingDir.type !=             /* this is NOT a dir ! */
                      io.Filable.THIS_IS_A_DIRECTORY)
                    {
                      return false;                              /* return ! */
                    }
                  else
                    {
                      result =
                        existingDir.addElement(
                          originalPath, path, type );
                      return result;                           /* return ! */
                    }
                /*end of if on type*/
                }
              catch( ArrayIndexOutOfBoundsException e ) /* no such element*/
                {
                  elements.
                    addElement( new
                      RamFile( firstDirName, firstDirName,
                               io.Filable.THIS_IS_A_DIRECTORY,
                               this                              ) );
                }
            /*end of try-catch*/
            }
          else /* add a new directory */
            {
              elements.
                addElement( new
                  RamFile( firstDirName, firstDirName,
                           io.Filable.THIS_IS_A_DIRECTORY,
                           this                              ) );

            }
        /*end of if on pathPlaceInElementsVector*/

          result =
            addElement( originalPath, originalPath, type );
        }
    /*end of if on separatorPlaceInThePath*/

      return result;

     }
/*end of addElement*/


  private
  boolean

  addElement( java.lang.String originalPath,
              java.lang.String path,
              int type                        )
/*****************************************************************************/
/* Function name: addElement                                                 */
/* File name: RamFile.java                                                   */
/* Purpose: to insert an element into the tree - this method determines      */
/*          the place where to put the element according to the rules        */
/*          of the local OS                                                  */
/* Parameters: originalPath - this is the path with which the public method  */
/*                            was called, only leading and trailing path     */
/*                            separator were removed if there were           */
/*             path - one or more leading directory names already removed    */
/*                    from this string                                       */
/*             type - should be io.Filable.THIS_IS_A_DIRECTORY or            */
/*                              io.Filable.THIS_IS_A_REGULAR_FILE            */
/* Returns: true - if added OK, false - otherwise                            */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      boolean result = false;
      java.lang.String firstDirName = null;
      java.lang.String firstDirPath = null;
      java.lang.String pathWithFirstDir = null;
      int pathPlaceInFirstDirPath;
      RamFile existingDir = null;
      int separatorPlaceInThePath =
            general.Constants.JAVA_ERROR_CODE;
      int pathPlaceInElementsVector =
            general.Constants.JAVA_ERROR_CODE;
                                                         /* addElement */
/*---------------------------------------------------------------------------*/

      if (elements == (RamFileVector) null)
        elements = new RamFileVector();
      else
        ;
    /*end of if on null*/

      path = path.trim();

      if ( path.startsWith( File.separator ) )
        path =
          util.String.
            removeLeadingPathSeparator( path );
      else
        ;
    /*end of if*/

      if ( path.endsWith( File.separator ) )
        path =
          util.String.
            removeTrailingPathSeparator( path );
      else
        ;
    /*end of if*/

      separatorPlaceInThePath =
        path.indexOf( File.separator );

      if (separatorPlaceInThePath ==                   /* single file name */
          general.Constants.JAVA_ERROR_CODE)
        {
          pathPlaceInElementsVector =
            elements.indexOfAName( path );

          if (pathPlaceInElementsVector !=            /* file already exists */
              general.Constants.JAVA_ERROR_CODE)
            {
              return false;                                  /* return ! */
            }
          else                       /* not found, may add the element here */
            {                               /* first order element */
              elements.
                addElement( new
                  RamFile( path, originalPath, type, this ) );

              return true;                                   /* return ! */
            }
        /*end of if on pathPlaceInElementsVector*/

        }
      else /* this is a path, containing at least one separator */
        {
          pathWithFirstDir = path;
          firstDirName =
            path.substring( 0, separatorPlaceInThePath );
          path =
            path.substring( separatorPlaceInThePath );

          firstDirPath = originalPath;
          pathPlaceInFirstDirPath =
            firstDirPath.lastIndexOf( path );
          firstDirPath =
            firstDirPath.substring( 0,
                                    pathPlaceInFirstDirPath );

          pathPlaceInElementsVector =
            elements.indexOfAName( firstDirName );

          if (pathPlaceInElementsVector !=           /* something does exist */
              general.Constants.JAVA_ERROR_CODE)
            {
              try
                {
                  existingDir = (RamFile)
                    elements.elementAt(
                      pathPlaceInElementsVector );
                  if (existingDir.type !=             /* this is NOT a dir ! */
                      io.Filable.THIS_IS_A_DIRECTORY)
                    {
                      return false;                              /* return ! */
                    }
                  else
                    {
                      result =
                        existingDir.addElement(
                          originalPath, path, type );
                      return result;                           /* return ! */
                    }
                /*end of if on type*/
                }
              catch( ArrayIndexOutOfBoundsException e ) /* no such element*/
                {
                  elements.
                    addElement( new
                      RamFile( firstDirName, firstDirPath,
                               io.Filable.THIS_IS_A_DIRECTORY,
                               this                            ) );
                }
            /*end of try-catch*/
            }
          else /* add a new directory */
            {
                elements.
                  addElement( new
                    RamFile( firstDirName, firstDirPath,
                             io.Filable.THIS_IS_A_DIRECTORY,
                               this                           ) );

            }
        /*end of if on pathPlaceInElementsVector*/

          result =
            addElement( originalPath, pathWithFirstDir, type );

        }
    /*end of if on separatorPlaceInThePath*/

      return result;

     }
/*end of addElement*/


  public
  RamFile

  findElement( java.lang.String path )
/*****************************************************************************/
/* Function name: findElement                                                */
/* File name: RamFile.java                                                   */
/* Purpose: to get an element from the root object of the tree               */
/*          This method scans the tree from the root and returns             */
/*          the RamFile object with the given path                           */
/*    Note: leading and trailing path separators do not count                */
/* Parameters: path                                                          */
/* Returns: the RamFile object or                                            */
/*          null if not found                                                */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      RamFile result = null;
      java.lang.String originalPath = null;
      java.lang.String firstDirName = null;
      RamFile existingDir = null;
      int separatorPlaceInThePath =
            general.Constants.JAVA_ERROR_CODE;
      int pathPlaceInElementsVector =
            general.Constants.JAVA_ERROR_CODE;
                                                         /* findElement */
/*---------------------------------------------------------------------------*/

      if (elements == (RamFileVector) null)
        return null;                                   /* return ! */
      else
        ;
    /*end of if on null*/

      path = path.trim();

      if (path.equals( File.separator ) )
        return this;                                   /* return ! */
      else
        ;
    /*end of if on separator*/

      if ( path.startsWith( File.separator ) )
        path =
          util.String.
            removeLeadingPathSeparator( path );
      else
        ;
    /*end of if*/

      if ( path.endsWith( File.separator ) )
        path =
          util.String.
            removeTrailingPathSeparator( path );
      else
        ;
    /*end of if*/

      originalPath = path;

      separatorPlaceInThePath =
        path.indexOf( File.separator );

      if (separatorPlaceInThePath ==                   /* single file name  */
          general.Constants.JAVA_ERROR_CODE)           /* should be here or */
        {                                              /* there is no such  */
                                                       /* thing at all      */
          pathPlaceInElementsVector =
            elements.indexOfAName( path );

          if (pathPlaceInElementsVector !=             /* file does exist */
              general.Constants.JAVA_ERROR_CODE)
            {
              try
                {
                  return
                    getElement( pathPlaceInElementsVector );    /* return ! */
                }
              catch( ArrayIndexOutOfBoundsException e )
                {
                  return null;                                  /* return ! */
                }
            /*end of try-catch*/
            }
          else                                                 /* not found */
            {
              return null;                                     /* return ! */
            }
        /*end of if on pathPlaceInElementsVector*/

        }
      else /* this is a path, containing at least one separator */
        {
          firstDirName =
            path.substring( 0, separatorPlaceInThePath );
          path =
            path.substring( separatorPlaceInThePath );

          pathPlaceInElementsVector =
            elements.indexOfAName( firstDirName );

          if (pathPlaceInElementsVector !=           /* something does exist */
              general.Constants.JAVA_ERROR_CODE)
            {
              try
                {
                  existingDir = (RamFile)
                    elements.elementAt(
                      pathPlaceInElementsVector );
                  if (existingDir.type !=             /* this is NOT a dir ! */
                      io.Filable.THIS_IS_A_DIRECTORY)
                    {
                      return null;                              /* return ! */
                    }
                  else
                    {
                      result =
                        existingDir.findElement( path );

                      return result;                           /* return ! */
                    }
                /*end of if on type*/
                }
              catch( ArrayIndexOutOfBoundsException e ) /* no such element*/
                {
                  return null;                                 /* return ! */
                }
            /*end of try-catch*/
            }
          else /* not found */
            {
              return null;                                     /* return ! */
            }
        /*end of if on pathPlaceInElementsVector*/

        }
    /*end of if on separatorPlaceInThePath*/

     }
/*end of findElement*/


  public
  RamFile

  getElement( int numberOfElement ) throws ArrayIndexOutOfBoundsException
/*****************************************************************************/
/* Function name: getElement                                                 */
/* File name: RamFile.java                                                   */
/* Purpose: to get an element from the current RamFile object                */
/* Parameters: numberOfElement                                               */
/* Returns: the RamFile object representing the element                      */
/* Throws:  ArrayIndexOutOfBoundsException if numberOfElement is not valid   */
/* Last modified: 16.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                         /* getElement */
/*---------------------------------------------------------------------------*/

      if ( (elements == (RamFileVector) null)        ||
           (type != io.Filable.THIS_IS_A_DIRECTORY )
         )
        throw new ArrayIndexOutOfBoundsException();         /* throw ! */
      else
        ;
    /*end of if on null*/

      return (RamFile) elements.elementAt( numberOfElement );

     }
/*end of getElement*/


  public
  int

  elementCount()
/*****************************************************************************/
/* Function name: elementCount                                               */
/* File name: RamFile.java                                                   */
/* Purpose: to get the number of elements in this RamFile object             */
/*          representing a directory                                         */
/* Parameters:                                                               */
/* Returns: number of elements                                               */
/* Last modified: 16.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                         /* elementCount */
/*---------------------------------------------------------------------------*/

      if ( (elements == (RamFileVector) null)        ||
           (type != io.Filable.THIS_IS_A_DIRECTORY )
         )
        return 0;                                            /* return ! */
      else
        ;
    /*end of if on null*/

      return elements.size();

     }
/*end of elementCount*/


  private
  java.lang.String

  toString0( java.lang.String indentPrefix )
/*****************************************************************************/
/* Function name: toString0                                                  */
/* File name: RamFile.java                                                   */
/* Purpose: to represent a file tree under a dir element as a string         */
/* Parameters: indentPrefix                                                  */
/* Returns: the string representation of the file tree                       */
/* Last modified: 13.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      int i;
      RamFile currentElement;
      int nElements;
      java.lang.String result = new java.lang.String();
                                                         /* toString0 */
/*---------------------------------------------------------------------------*/

      indentPrefix =
        indentPrefix + TO_STRING_INDENT_STRING;

      if (elements == (RamFileVector) null)
        {
          result = result + indentPrefix + "nothing\n";
          return result;                                    /* return ! */
        }
      else
        ;
    /*end of if on null*/

      nElements = elements.size();
      i=0;
      while( i < nElements )
        {
          result = result + indentPrefix +
            ( (RamFile) elements.elementAt(i)).name +
            "\n";
          i++;
        }
    /*end of while-loop*/
      result = result + "\n";
                                                       /* list subdirs */
      i=0;
      while( i < nElements )
        {
          currentElement = (RamFile) elements.elementAt(i);
          if (currentElement.type ==
              io.Filable.THIS_IS_A_DIRECTORY)
            {
              result = result + "\n" + indentPrefix +
                "Directory " +
                currentElement.name + " contains:\n";
              result = result +
                currentElement.toString0( indentPrefix );
            }
          else
            ;
        /*end of if on THIS_IS_A_DIRECTORY*/
          i++;
        }
    /*end of while-loop*/
      if (nElements == 0)
        result = result + indentPrefix + "nothing\n";
      else
        ;
    /*end of if on 0*/

      return result;

     }
/*end of toString0*/


  public
  java.lang.String

  toString()
/*****************************************************************************/
/* Function name: toString                                                   */
/* File name: RamFile.java                                                   */
/* Purpose: to represent the file tree as a string                           */
/* Parameters:                                                               */
/* Returns: the string representation of the file tree                       */
/* Last modified: 13.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      int i;
      RamFile currentElement;
      int nElements;
      java.lang.String result = new java.lang.String();
                                                         /* toString */
/*---------------------------------------------------------------------------*/

      if (elements == (RamFileVector) null)
        return result;                                    /* return ! */
      else
        ;
    /*end of if on null*/

      nElements = elements.size();
      i=0;
      while( i < nElements )
        {
          result = result +
            ( (RamFile) elements.elementAt(i)).name +
            "\n";
          i++;
        }
    /*end of while-loop*/
      result = result + "\n";
                                                       /* list subdirs */
      i=0;
      while( i < nElements )
        {
          currentElement = (RamFile) elements.elementAt(i);
          if (currentElement.type ==
              io.Filable.THIS_IS_A_DIRECTORY)
            {
              result = result + "\nDirectory " +
                currentElement.name + " contains:\n";
              result = result +
                currentElement.toString0( "" );
            }
          else
            ;
        /*end of if on THIS_IS_A_DIRECTORY*/
          i++;
        }
    /*end of while-loop*/

      return result;

     }
/*end of toString*/


  public
  java.lang.String[]

  list()
/*****************************************************************************/
/* Function name: list                                                       */
/* File name: RamFile.java                                                   */
/* Purpose: to return an array of file names contained in this RamFile       */
/*          object representing a directory                                  */
/* Parameters:                                                               */
/* Returns: array of file names or                                           */
/*          null if this is not a directory or there is no files in it       */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      int i;
      RamFile currentElement;
      int nElements;
      java.lang.String[] resultArray = null;
                                                           /* list */
/*---------------------------------------------------------------------------*/

      if (elements == (RamFileVector) null)
        return null;                                       /* return ! */
      else
        ;
    /*end of if on null*/

      nElements = elements.size();
      resultArray = new java.lang.String[ nElements ];
      i=0;
      while( i < nElements )
        {
          resultArray[i] =
            ( (RamFile) elements.elementAt(i)).name;
          i++;
        }
    /*end of while-loop*/

      return resultArray;

    }
/*end of list*/


  public
  java.lang.String

  getParent()
/*****************************************************************************/
/* Function name: getParent                                                  */
/* File name: RamFile.java                                                   */
/* Purpose: to return path of the parent directory of this RamFile object    */
/* Parameters:                                                               */
/* Returns: the path string or                                               */
/*          null if there is no parent directory                             */
/* Last modified: 18.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                          /* getParent */
/*---------------------------------------------------------------------------*/

      if (parentDirectory == (RamFile) null)
        {
          return null;                                     /* return ! */
        }
      else
        ;
    /*end of if on null*/

      return parentDirectory.path;
             

    }
/*end of getParent*/


  public
  boolean

  exists()
/*****************************************************************************/
/* Function name: exists                                                     */
/* File name: RamFile.java                                                   */
/* Purpose: always returns true for this file system                         */
/* Parameters:                                                               */
/* Returns: true                                                             */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                          /* exists */
/*---------------------------------------------------------------------------*/

      return true;

    }
/*end of exists*/


  public
  boolean

  isDirectory()
/*****************************************************************************/
/* Function name: isDirectory                                                */
/* File name: RamFile.java                                                   */
/* Purpose: to return true if this RamFile object represents a directory     */
/* Parameters:                                                               */
/* Returns: true if this is a directory                                      */
/*          false - if this not a directory                                  */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                       /* isDirectory */
/*---------------------------------------------------------------------------*/

      if (type == THIS_IS_A_DIRECTORY)
        return true;
      else
        return false;
    /*end of if*/

    }
/*end of isDirectory*/


  public
  java.lang.String

  getCanonicalPath()
/*****************************************************************************/
/* Function name: getCanonicalPath                                           */
/* File name: RamFile.java                                                   */
/* Purpose: this method returns just the path string that we have            */
/*          for this RamFile object for this file system                     */
/* Parameters:                                                               */
/* Returns: the path string or                                               */
/*          null if there is no path                                         */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                  /* getCanonicalPath */
/*---------------------------------------------------------------------------*/

      return path;

    }
/*end of getCanonicalPath*/


  public
  boolean

  isFile()
/*****************************************************************************/
/* Function name: isFile                                                     */
/* File name: RamFile.java                                                   */
/* Purpose: to true if this RamFile object represents a regular file         */
/* Parameters:                                                               */
/* Returns: true - if the is a regular file                                  */
/*          false - otherwise                                                */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                         /* isFile */
/*---------------------------------------------------------------------------*/

      if (type == THIS_IS_A_REGULAR_FILE)
        return true;
      else
        return false;
    /*end of if*/

    }
/*end of isFile*/


  public
  java.lang.String

  getName()
/*****************************************************************************/
/* Function name: getName                                                    */
/* File name: RamFile.java                                                   */
/* Purpose: to return the name string of this RamFile object                 */
/* Parameters:                                                               */
/* Returns: the name string                                                  */
/* Last modified: 14.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
                                                       /* getName */
/*---------------------------------------------------------------------------*/

      return name;

    }
/*end of getName*/


  public
  java.lang.String
  getPath()
    {

      return "";

    }




  public
  long
  lastModified()
    {

      return 0;

    }


  public
  long
  length()
    {
      return 0;
    }



  public
  boolean
  mkdir( java.lang.String dirName )
    {
      return true;
    }


  public
  boolean
  mkdirs( java.lang.String dirName )
    {
      return true;
    }



  public
  static
  void

  main( java.lang.String[] args )
/*****************************************************************************/
/* Function name: main                                                       */
/* File name: RamFile.java                                                   */
/* Purpose: main to test the RamFile class                                   */
/* Parameters: args                                                          */
/* Returns:                                                                  */
/* Last modified: 16.05.99                                                   */
/* Author: Victor Klimov, 1999                                               */
/*****************************************************************************/
    {
      int i;
      boolean result;
      boolean continueToLoop = true;
      RamFile file;
      int type = io.Filable.THIS_IS_UNKNOWN_TYPE;
                                                            /* main */
/*---------------------------------------------------------------------------*/

      if (args.length<1)
        {
          System.out.println(
            "\nUsage: java RamFile <path1> <type1> " +
            "[ [ path2 type2 ] ...]" );
          System.out.println( "type: f - for regular file, " +
                              "d - for directories"            );
          System.exit( 1 );
        }
      else
        ;
    /*end of if*/

      file = new RamFile();

      i=0;
      while( (i<args.length-1) &&
             continueToLoop
           )

        {
          if (args[i+1].equals( "f" ) )
            type = io.Filable.THIS_IS_A_REGULAR_FILE;
          else
            if (args[i+1].equals( "d" ))
              type = io.Filable.THIS_IS_A_DIRECTORY;
            else
              continueToLoop = false;
          /*end of if on d*/
        /*end of if on f*/

          if (continueToLoop)
            {
              result =
                file.addElement( args[i], type );
              System.out.println( "\ni=" + i +
                                  " was adding " +
                                  args[i] +
                                  "\nresult=" + result );
              i+=2;
            }
          else
            ;
        /*end of if on continueToLoop*/
        }
    /*end of while-loop*/

      System.out.println( "\nfile=\n" +
                          file.toString() );

    }
/*end of main*/

} /*end of RamFile */

/*****************************************************************************/
/* End of file: RamFile.java                                                 */
/*****************************************************************************/
